<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class Pigee_Insurance_Checkout_Block {

    public static function init() {
        add_action( 'wp_enqueue_scripts', [ __CLASS__, 'enqueue_script' ] );
		add_action( 'wp_footer', [ __CLASS__, 'output_insurance_data' ], 5 ); // Output data later
    }

    public static function enqueue_script() {

        if ( ! function_exists( 'is_checkout' ) || ! is_checkout() ) {
            return;
        }

        $asset_path = __DIR__ . '/build/insurance-checkout.asset.php';
        if ( ! file_exists( $asset_path ) ) {
            return;
        }

        $asset = require $asset_path;

        $deps = array_unique(
            array_merge(
                $asset['dependencies'],
                [ 'wc-blocks-checkout' ]
            )
        );

        wp_enqueue_script(
            'pigee-insurance-checkout',
            plugins_url(
                'blocks/build/insurance-checkout.js',
                dirname( __FILE__ )
            ),
            $deps,
            $asset['version'],
            true
        );
    }

	 /**
     * Output insurance data in footer - SAFE because WooCommerce is fully loaded by now
     */
    public static function output_insurance_data() {
        if ( ! function_exists( 'is_checkout' ) || ! is_checkout() ) {
            return;
        }
		
		// Check if WooCommerce is ready
        if ( ! class_exists( 'WooCommerce' ) ) {
            error_log('❌ Pigee: WooCommerce not loaded yet');
            return;
        }

		// Check if Blocks checkout
		$is_blocks_checkout = has_block('woocommerce/checkout') || 
                         has_block('woocommerce/cart') ||
                         (function_exists('wc_blocks_container') && class_exists('Automattic\WooCommerce\Blocks\Package'));
						 
        // Get insurance options safely
        $insurance_options = self::get_insurance_options_safe();
        
        if ( empty( $insurance_options ) ) {
            error_log('⚠️ Pigee: No insurance options available');
            $insurance_options = [
                'no_cover' => [
                    'label' => 'No Insurance',
                    'price' => 0,
                    'symbol' => get_woocommerce_currency_symbol(),
                    'currency' => get_woocommerce_currency(),
                    'request_id' => '',
                    'description' => 'No insurance protection'
                ]
            ];
        }

        // Get selected insurance safely
        $selected = 'no_cover';
        if ( WC()->session && WC()->session->get( 'pigee_insurance' ) ) {
            $selected = WC()->session->get( 'pigee_insurance' );
        }

		// Prepare data
		$insurance_data = [
			'options' => $insurance_options,
			'currency' => get_woocommerce_currency_symbol(),
			'selected' => $selected,
			'root' => esc_url_raw(rest_url()),
			'nonce' => wp_create_nonce('wp_rest'),
			'is_blocks_checkout' => $is_blocks_checkout,
		];
	
        error_log('✅ Pigee: Outputting insurance data with ' . count($insurance_options) . ' options');

		// Output as inline script
		echo '<script>';
		echo 'window.PIGEE_INSURANCE_DATA = ' . wp_json_encode($insurance_data) . ';';
		echo 'console.log("PIGEE: Insurance data loaded", window.PIGEE_INSURANCE_DATA ? "✅" : "❌");';
		echo '</script>';
    }
	
	/**
     * Safe way to get insurance options
     */
    private static function get_insurance_options_safe() {
        // Check if WooCommerce cart is available
        if ( ! WC() || ! WC()->cart || WC()->cart->is_empty() ) {
            error_log('⚠️ Pigee: Cart not available for insurance calculation');
            return self::get_fallback_insurance_options();
        }

        try {
            $item_total = (float) WC()->cart->get_subtotal(); 
            $currency = get_woocommerce_currency();
            
            $api_key = get_api_key_from_shipping();
            $api_mode = get_api_mode_from_shipping();
                    
            $api = new PigeeShippingPayments_API($api_key, $api_mode);
                    
            $payload = [
                'order_from' => 'wp',
                'shop_url' => get_bloginfo('url'),
                'item_total' => $item_total,
                'currency'   => $currency,
            ];

            $response = $api->send_request('getinsurance', $payload);
            
            if ( ! empty( $response['status'] ) && ! empty( $response['insurances'] ) ) {
                $insurance = [];
                foreach( $response['insurances'] as $ins ) {
                    $k = $ins['key'];
                    $insurance[$k] = [
                        'request_id' => $ins['request_id'] ?? '',
                        'label' => $ins['label'] ?? 'Unknown',
                        'description' => $ins['description'] ?? '',
                        'price' => $ins['cost'] ?? 0,
                        'symbol' => $ins['symbol'] ?? '$',
                        'currency' => $ins['currency'] ?? 'USD',
                    ];
                }
                return $insurance;
            }
        } catch ( Exception $e ) {
            error_log('❌ Pigee: Error fetching insurance options: ' . $e->getMessage());
        }

        // Fallback if API fails
        return self::get_fallback_insurance_options();
    }

	 /**
     * Fallback insurance options
     */
    private static function get_fallback_insurance_options() {
        return [
            'no_cover' => [
                'label' => 'No Cover',
                'request_id' => '',
                'description' => 'No insurance protection. Shipment travels entirely at sender’s risk. Low-value, replaceable items',
                'price' => 0,
                'symbol' => get_woocommerce_currency_symbol(),
                'currency' => get_woocommerce_currency(),
            ]
        ];
    }
}

/**
 * Helper function for backward compatibility
 */
function pigee_get_insurance_options() {
    return Pigee_Insurance_Checkout_Block::get_insurance_options_safe();
}