<?php
if ( ! defined( 'ABSPATH' ) ) exit;

use Automattic\WooCommerce\StoreApi\Utilities\CartController;
use Automattic\WooCommerce\StoreApi\Utilities\OrderController;

if (!defined('ABSPATH')) {
    exit;
}

include_once plugin_dir_path(__FILE__) . 'wc-payzec-helper.php';

// Function to register REST API endpoints:
add_action('rest_api_init', function () {
    register_rest_route('payzec/v1', '/getCart/(?P<cartId>\d+)', array(
        'methods' => 'GET',
        'callback' => 'payzec_get_cart',
        'permission_callback' => 'payzec_validate_token'
    ));

    register_rest_route('payzec/v1', '/getOrder/(?P<orderId>\d+)', array(
        'methods' => 'GET',
        'callback' => 'payzec_get_order',
        'permission_callback' => 'payzec_validate_token'
    ));

    register_rest_route('payzec/v1', '/updateOrder', array(
        'methods' => 'POST',
        'callback' => 'payzec_order_update',
        'permission_callback' => 'payzec_validate_token'
    ));

    register_rest_route('payzec/v1', '/createOrder', array(
        'methods' => 'POST',
        'callback' => 'payzec_create_order',
        'permission_callback' => 'payzec_validate_token'
    ));

    register_rest_route('payzec/v1', '/verify', array(
        'methods' => 'GET',
        'callback' => 'payzec_verify_token',
        'permission_callback' => 'payzec_validate_token'
    ));

    register_rest_route('payzec/v1', '/autoLogin', array(
        'methods' => 'POST',
        'callback' => 'payzec_auto_login',
        'permission_callback' => 'payzec_validate_token'
    ));

    register_rest_route('payzec/v1', '/set-addresses', array(
        'methods' => 'POST',
        'callback' => 'payzec_set_addresses',
        'permission_callback' => 'payzec_validate_token'
    ));

    register_rest_route('payzec/v1', '/get-shipping-cost', array(
        'methods' => 'POST',
        'callback' => 'payzec_get_shipping_cost',
        'permission_callback' => 'payzec_validate_token'
    ));

    register_rest_route('payzec/v1', '/user-data', array(
        'methods' => 'GET',
        'callback' => 'payzec_get_payzec_user_data',
        'permission_callback' => '__return_true',
    ));
});

function payzec_get_payzec_user_data()
{
    if (!function_exists('wp_get_current_user')) {
        include(ABSPATH . "wp-includes/pluggable.php");
    }

    $is_logged_in = is_user_logged_in();
    $payzec_cart_id = payzec_maybe_load_cart()->get_cart_contents_count() > 0 ? payzec_get_cart_id() : false;
    $show_payzec_icon = isset(payzec_get_settings()['show_icon']) ? payzec_get_settings()['show_icon'] : "yes";

    return array(
        'logged_in' => $is_logged_in,
        'payzec_cart_id' => $payzec_cart_id,
        'show_payzec_icon' => $show_payzec_icon
    );
}

// Function to validate token:
function payzec_validate_token()
{
    $token = get_option('payzec_token');
    $request_token = sanitize_text_field($_SERVER['HTTP_X_PAYZEC_TOKEN'] ?? '');

    if ($token === $request_token) {
        return true;
    } else {
        return new WP_Error('rest_forbidden', esc_html__('Invalid token.', 'payzec-gateway'), array('status' => 401));
    }
}

function payzec_set_addresses($data)
{
    try {
        $customerData = $data['customer'];
        $cartId = $data['cartId'];

        global $wpdb;
        $table_name = $wpdb->prefix . 'woocommerce_sessions';
        $cart_data = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table_name WHERE session_id = %s", $cartId));
        if (!$cart_data) {
            return new WP_Error('rest_not_found', esc_html__('Cart not found.', 'payzec-gateway'), array('status' => 404));
        }

        $sessionData = (unserialize($cart_data->session_value));
        $sessionCustomer = unserialize($sessionData['customer']);

        wp_set_current_user($sessionCustomer['id']);
        payzec_set_woocommerce_session_by_id($cartId);
        WC()->customer->set_props($customerData);
        WC()->cart->set_cart_contents(unserialize($sessionData['cart']));

        $cartSession = (new WC_Cart_Session(WC()->cart));
        $cartSession->get_cart_from_session();

        payzec_calculate_shipping($customerData);
        WC()->cart->calculate_totals();
        WC()->cart->calculate_shipping();
        WC()->cart->calculate_fees();
        update_metadata('user', get_current_user_id(), 'billing_first_name', $customerData['first_name']);
        update_metadata('order', $cartId, '_billing_first_name', $customerData['first_name']);

        $wpdb->update($table_name, array('session_value' => maybe_serialize(WC()->session->get_session_data())), array('session_id' => $cartId));

    } catch (Exception $e) {
        return new WP_Error('rest_exception', esc_html__('An error occurred.'), array('status' => 500));
    }

    return ['status' => 'success'];
}

function payzec_set_woocommerce_session_by_id($session_id) {
    global $wpdb;

    // Table name for WooCommerce sessions
    $table_name = $wpdb->prefix . 'woocommerce_sessions';

    // Retrieve the session data from the database
    $session_data = $wpdb->get_var($wpdb->prepare("SELECT session_value FROM $table_name WHERE session_id = %s", $session_id));

    if (!$session_data) {
        return new WP_Error('session_not_found', __('Session not found.', 'payzec-gateway'));
    }

    // Unserialize the session data
    $session_data = maybe_unserialize($session_data);

    // Load WooCommerce session handler
    WC()->session->destroy_session();
    WC()->session->set_customer_session_cookie(true);

    // Ensure session handler is available
    if (!is_a(WC()->session, 'WC_Session_Handler')) {
        return new WP_Error('session_handler_unavailable', __('WooCommerce session handler is not available.', 'payzec-gateway'));
    }

    // Set session data
    foreach ($session_data as $key => $value) {
        WC()->session->set($key, unserialize($value));
    }

    // Set the session cookie to the new session ID
    WC()->session->set_customer_session_cookie(true);

    return true;
}

// Function to get cart details:
function payzec_get_cart($data)
{
    $cartId = $data['cartId'];

    global $wpdb;
    $table_name = $wpdb->prefix . 'woocommerce_sessions';
    $cart_data = $wpdb->get_row($wpdb->prepare("SELECT session_value FROM $table_name WHERE session_id = %s", $cartId));
    if (!$cart_data) {
        return new WP_Error('rest_not_found', esc_html__('Cart not found.', 'payzec-gateway'), array('status' => 404));
    }

    $sessionData = unserialize($cart_data->session_value);
    $cart = unserialize($sessionData['cart']);
    $items = [];
    foreach ($cart as $quoteItem) {
        /** @var WC_Product $product */
        $product = $quoteItem['variation_id'] ? new WC_Product_Variation($quoteItem['variation_id']) : new WC_Product($quoteItem['product_id']);
        if(!$product) {
            continue;
        }

        $item = [];
        $item['id'] = $product->get_id();
        $item['sku'] = strlen($product->get_sku()) > 2 ? $product->get_sku() : 'payzecId_' . $product->get_id();
        $item['name'] = $product->get_name();
        $item['price'] = (float)$product->get_price();
        $item['quantity'] = (float)$quoteItem['quantity'];
        $item['subtotal'] = (float)$quoteItem['line_total'];
        $item['img'] = wp_get_attachment_image_src(get_post_thumbnail_id($product->get_id()), 'single-post-thumbnail')[0] ?? '';
        $item['url'] = get_permalink($quoteItem['product_id']);
        $items[] = $item;
    }


    $cartTotals = $sessionData['cart_totals'] ? unserialize($sessionData['cart_totals']) : 0;

    if($cartTotals['shipping_total']) {
        $items[] = [
            'id' => 'shipping_item',
            'sku' => 'shipping_item',
            'name' => 'Shipping',
            'price' => (float)$cartTotals['shipping_total'],
            'quantity' => (float)1,
            'subtotal' => (float)$cartTotals['shipping_total'],
        ];
    }

    $cartInfo = [
        'currency' => get_woocommerce_currency(),
        'items' => $items,
        'total' => (float)$cartTotals['total'],
        'shipping_amount' => (float)$cartTotals['shipping_total'],
        'shipping_method' => payzec_shipping_methods(),
        'billing_address' => $cart['billing_address'] ?? '',
        'shipping_address' => $cart['shipping_address'] ?? '',
    ];

    return $cartInfo;
}

function payzec_get_shipping_cost($data)
{
    $cartId = $data['cartId'];
    $shippingAddress = $data['shipping_address'];

    global $wpdb;
    $table_name = $wpdb->prefix . 'woocommerce_sessions';
    $cart_data = $wpdb->get_row($wpdb->prepare("SELECT session_value FROM $table_name WHERE session_id = %s", $cartId));

    if (!$cart_data) {
        return new WP_Error('rest_not_found', esc_html__('Cart not found.', 'payzec-gateway'), array('status' => 404));
    }

    // Set the WooCommerce session by cart ID
    payzec_set_woocommerce_session_by_id($cartId);
    $cartSession = (new WC_Cart_Session(WC()->cart));
    $cartSession->get_cart_from_session();

    // Set shipping address for the cart
    WC()->customer->set_shipping_country($shippingAddress['country'] ?? '');
    WC()->customer->set_shipping_state($shippingAddress['state'] ?? '');
    WC()->customer->set_shipping_postcode($shippingAddress['postcode'] ?? '');
    WC()->customer->set_shipping_city($shippingAddress['city'] ?? '');
    WC()->customer->set_shipping_address($shippingAddress['address_1'] ?? '');
    WC()->customer->set_shipping_address_2($shippingAddress['address_2'] ?? '');
    WC()->customer->set_billing_country($shippingAddress['country'] ?? '');
    WC()->customer->set_billing_state($shippingAddress['state'] ?? '');
    WC()->customer->set_billing_postcode($shippingAddress['postcode'] ?? '');
    WC()->customer->set_billing_city($shippingAddress['city'] ?? '');
    WC()->customer->set_billing_address($shippingAddress['address_1'] ?? '');
    WC()->customer->set_billing_address_2($shippingAddress['address_2'] ?? '');

    WC()->customer->save_data();

    // Calculate shipping methods
    WC()->cart->calculate_shipping();

    $packages = WC()->shipping()->get_packages();

    $available_methods = [];

    foreach ($packages as $package_key => $package) {
        foreach ($package['rates'] as $rate) {
            $available_methods[] = [
                'id' => $rate->get_id(),
                'label' => $rate->get_label(),
                'cost' => (float)$rate->get_cost(),
                'taxes' => $rate->get_taxes(),
                'method_id' => $rate->get_method_id(),
                'instance_id' => $rate->get_instance_id(),
            ];
        }
    }

    return $available_methods;
}

function payzec_get_order($data)
{
    $orderId = $data['orderId'];
    $order = wc_get_order($orderId);
    if (!$order) {
        return new WP_Error('rest_not_found', esc_html__('Order not found.', 'payzec-gateway'), array('status' => 404));
    }

    $ship_to_different_address = get_post_meta($orderId, '_ship_to_different_address', true);
    $addresses_match = ($ship_to_different_address === '0');  // If 'ship_to_different_address' is '0', addresses match

    $orderData = [
        'id' => $order->get_id(),
        'status' => $order->get_status(),
        'total' => $order->get_total(),
        'currency' => $order->get_currency(),
        'items' => [],
        'addresses_match' => $addresses_match,
    ];
    foreach ($order->get_items() as $item_id => $item) {
        $product = $item->get_product();
        $orderData['items'][] = [
            'id' => $product->get_id(),
            'sku' => strlen($product->get_sku()) > 2 ? $product->get_sku() : 'payzecId_' . $product->get_id(),
            'name' => $product->get_name(),
            'price' => (float)$product->get_price(),
            'quantity' => (float)$item->get_quantity(),
            'subtotal' => (float)$item->get_total(),
            'img' => wp_get_attachment_image_src(get_post_thumbnail_id($product->get_id()), 'single-post-thumbnail')[0] ?? '',
            'url' => get_permalink($product->get_id()),
        ];
    }

    if($order->get_shipping_total()) {
        $orderData['items'][] = [
            'id' => 'shipping_item',
            'sku' => 'shipping_item',
            'name' => 'Shipping',
            'price' => (float)$order->get_shipping_total(),
            'quantity' => (float)1,
            'subtotal' => (float)$order->get_shipping_total(),
        ];
    }

    $orderData['billing_address'] = $order->get_address('billing');
    $orderData['shipping_address'] = $order->get_address('shipping');
    $orderData['shipping_amount'] = $order->get_shipping_total();
    $orderData['shipping_method'] = $order->get_shipping_method();
    $orderData['billing_address_same_as_shipping'] = $order->get_formatted_billing_address() === $order->get_formatted_shipping_address();

    return $orderData;
}

function payzec_order_update($data)
{
    $orderId = $data['orderId'];
    $order = wc_get_order($orderId);
    if (!$order) {
        return new WP_Error('rest_not_found', esc_html__('Order not found.', 'payzec-gateway'), array('status' => 404));
    }

    $status = $data['status'];
    WC()->cart->empty_cart();
    if ($status) {
        $order->payment_complete();
        $order->set_status($status);
        $order->save();
        //redirect to order success
        return ['status' => 'success', 'redirect' => $order->get_checkout_order_received_url()];
        exit;
    } else {
        //redirect to order failed
        return ['status' => 'failed', 'redirect' => $order->get_checkout_payment_url()];
        exit;
    }
}

function payzec_create_order($data)
{
    // Empty cart
    WC()->cart->empty_cart();

    $data = $data->get_json_params();

    if (empty($data)) {
        return new WP_Error('no_data', 'No data provided', array('status' => 400));
    }

    // Parse shipping and billing address
    $shipping_address = $data['shipping_address'];
    $billing_address = $data['billing_address'];

    // Create a new order
    $order = wc_create_order();

    // Set billing address
    $order->set_address(array(
        'first_name' => $billing_address['first_name'],
        'last_name' => $billing_address['last_name'],
        'company' => '',
        'email' => $billing_address['email'],
        'phone' => $billing_address['phone'],
        'address_1' => $billing_address['street'],
        'address_2' => '',
        'city' => $billing_address['city'],
        'state' => $billing_address['state'],
        'postcode' => $billing_address['zip'],
        'country' => $billing_address['country'],
    ), 'billing');

    // Set shipping address
    $order->set_address(array(
        'first_name' => $shipping_address['first_name'],
        'last_name' => $shipping_address['last_name'],
        'company' => '',
        'address_1' => $shipping_address['street'],
        'address_2' => '',
        'city' => $shipping_address['city'],
        'state' => $shipping_address['state'],
        'postcode' => $shipping_address['zip'],
        'country' => $shipping_address['country'],
    ), 'shipping');

    // Add items
    foreach ($data['items'] as $item) {
        if($item['id'] == 'shipping_item') {
            continue;
        }

        $product_id = $item['id'];
        $quantity = $item['quantity'];

        $order->add_product(wc_get_product($product_id), $quantity);
    }

    // Set order total
    $order->set_total($data['amount']);

    if($data['shippingCost']) {
        $shipping = new WC_Order_Item_Shipping();
        $shipping->set_method_title( 'Shipping' );
        $shipping->set_method_id( 'shipping' ); // set an existing Shipping method ID
        $shipping->set_total( $data['shippingCost']); // optional
        $order->add_item( $shipping );
    }

    // Set payment method
    $order->set_payment_method('payzec'); // Use your payment method slug here

    $order->set_created_via('payzec');

    // Mark order as paid if status is 'paid'
    if ($data['status'] === 'paid') {
        $order->payment_complete();
    }

    $order->set_status('processing');

    $order->calculate_totals();
    // Save the order
    $order->save();

    return rest_ensure_response(array(
        'order_id' => $order->get_id(),
        'status' => 'success',
    ));
}

function payzec_verify_token($data)
{
    return ['status' => 'success'];
}

function payzec_shipping_methods()
{
    $shipping_methods_data = array();

    // Get all shipping zones
    $zones = WC_Shipping_Zones::get_zones();
    $zones[] = array('id' => 0); // Add default zone

    foreach ($zones as $zone) {
        $shipping_zone = new WC_Shipping_Zone($zone['id']);
        $zone_methods = $shipping_zone->get_shipping_methods();
        $zone_name = $shipping_zone->get_zone_name();

        $methods = array();
        foreach ($zone_methods as $method) {
            // Get method cost
            $cost = isset($method->cost) ? $method->cost : '0.00';

            // Check for instance settings cost (for methods like flat rate)
            if ('flat_rate' === $method->id || 'local_pickup' === $method->id) {
                $instance_settings = $method->instance_settings;
                if (isset($instance_settings['cost']) && !empty($instance_settings['cost'])) {
                    $cost = $instance_settings['cost'];
                }
            }

            $methods[] = array(
                'title' => $method->title,
                'id' => $method->id,
                'cost' => $cost,
            );
        }

        $shipping_methods_data[] = array(
            'zone' => $zone_name,
            'methods' => $methods,
        );
    }

    return $shipping_methods_data;
}


