<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

/*
Plugin Name: Payzec Gateway
Plugin URI: https://payzec.com/
Description: A generic WooCommerce payment integrator by Payzec.
Version: 1.0
Author: M.Can Özdemir
Author URI: https://github.com/mucan54
*/

if (!in_array('woocommerce/woocommerce.php', apply_filters('active_plugins', get_option('active_plugins')))) return;

include_once plugin_dir_path(__FILE__) . 'wc-payzec-helper.php';

add_filter('woocommerce_payment_gateways', 'payzec_add_payzec_gateway');
function payzec_add_payzec_gateway($gateways) {
    $gateways[] = 'WC_Gateway_Payzec';
    return $gateways;
}

add_action( 'woocommerce_cart_item_set_quantity', function ()
{
    try{
        WC()->session->set('payzec_cart_id', payzec_get_cart_id());
    } catch (\Exception $exception) {
        error_log($exception->getMessage());
    }
});

add_action('plugins_loaded', 'payzec_init_payzec_gateway');

function payzec_init_payzec_gateway() {
    class WC_Gateway_Payzec extends WC_Payment_Gateway_CC {
        protected $token;
        public function __construct() {
            $this->id = 'payzec';
            $this->icon = '';
            $this->has_fields = true;
            $this->method_title = 'Payzec Gateway';
            $this->method_description = 'Payzec payment gateway for secure card transactions.';
            //$this->template_name      = 'credit-card.php';
            $this->init_form_fields();
            $this->init_settings();
            $this->supports = array(
                'products'
            );

            $this->title = $this->get_option('title');
            $this->description = $this->get_option('description');
            $this->enabled = $this->get_option('enabled');
            $this->token = $this->get_option('token');

            add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
            add_action('woocommerce_receipt_' . $this->id, array($this, 'receipt_page'));
            add_action('wp_enqueue_scripts','payzec_add_init_js');
            //add_action('wp_footer', 'add_html_to_footer');

            add_action('wp_ajax_payzec_regenerate_token', array($this, 'payzec_regenerate_token'));
            // Enqueue the script
            add_action('admin_enqueue_scripts', array($this, 'payzec_enqueue_admin_scripts'));
            add_filter('payzec_generate_regenerate_token_button_html', array($this, 'payzec_generate_regenerate_token_button_html'));
        }

        public function payzec_enqueue_admin_scripts() {
            wp_enqueue_script('payzec-admin', plugins_url('/assets/js/payzec-admin.js', __FILE__), array('jquery'), null, true);

            wp_localize_script('payzec-admin', 'payzec_admin', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce'    => wp_create_nonce('payzec_regenerate_token_nonce')
            ));
        }

        public function payzec_regenerate_token() {
            check_ajax_referer('payzec_regenerate_token_nonce', 'nonce');

            $new_token = bin2hex(openssl_random_pseudo_bytes(16));
            $settings = get_option('woocommerce_payzec_settings', array());
            $settings['token'] = $new_token;
            update_option('payzec_token', $new_token);
            update_option('woocommerce_payzec_settings', $settings);

            wp_send_json_success(array('token' => $new_token));
        }

        public function generate_regenerate_token_button_html($key, $data) {
            $field = $this->get_field_key($key);

            ob_start();
            ?>
            <tr valign="top">
                <th scope="row" class="titledesc">
                    <label for="<?php echo esc_attr($field); ?>"><?php echo wp_kses_post($data['title']); ?></label>
                </th>
                <td class="forminp">
                    <button type="button" id="payzec_regenerate_token_button" class="button-secondary">Regenerate Token</button>
                    <p class="description"><?php echo wp_kses_post($data['description']); ?></p>
                </td>
            </tr>
            <?php
            return ob_get_clean();
        }

        public function generate_token_field_html($key, $data) {
            $field = $this->get_field_key($key);
            $token = $this->get_option('token');

            ob_start();
            ?>
            <tr valign="top">
                <th scope="row" class="titledesc">
                    <label for="<?php echo esc_attr($field); ?>"><?php echo wp_kses_post($data['title']); ?></label>
                </th>
                <td class="forminp">
                    <input type="text" id="woocommerce_payzec_token" value="<?php echo esc_attr($token); ?>" disabled />
                    <button type="button" id="copy_payzec_token_button" class="button-secondary">Copy Token</button>
                    <p class="description"><?php echo wp_kses_post($data['description']); ?></p>
                </td>
            </tr>
            <?php
            return ob_get_clean();
        }

        public function payzec_get_payzec_token()
        {
            if (!get_option('payzec_token')) {
                $token = bin2hex(openssl_random_pseudo_bytes(16));
                update_option('payzec_token', $token);
            }

            return get_option('payzec_token');
        }

        public function init_form_fields() {
            $this->form_fields = array(
                'enabled' => array(
                    'title' => 'Enable/Disable',
                    'type' => 'checkbox',
                    'label' => 'Enable Payzec Gateway',
                    'default' => 'no'
                ),
                'title' => array(
                    'title' => 'Title',
                    'type' => 'text',
                    'description' => 'This controls the title which the user sees during checkout.',
                    'default' => 'Credit Card (Payzec)',
                    'desc_tip' => true,
                ),
                'description' => array(
                    'title' => 'Description',
                    'type' => 'textarea',
                    'description' => 'This controls the description which the user sees during checkout.',
                    'default' => 'Pay with your credit card via Payzec.',
                ),
                'token' => array(
                    'title' => 'Token',
                    'type' => 'token_field',
                    'disabled' => true,
                    'default' => $this->payzec_get_payzec_token(),
                    'value' => $this->payzec_get_payzec_token(),
                    'description' => 'Token generated for authenticating requests.',
                ),
                'regenerate_token' => array(
                    'title' => 'Regenerate Token',
                    'type' => 'regenerate_token_button',
                    'description' => 'Click to generate a new token.',
                ),
                'show_icon' => array(
                    'title' => 'Pay with Payzec',
                    'type' => 'checkbox',
                    'label' => 'Show payzec icon and enable one click checkout.',
                    'default' => 'no'
                ),
            );
        }

        public function process_payment( $order_id ) {
            WC()->session->set('payzec-payment-proccess-id', $order_id);
            return array(
                'result' => 'success',
                'redirect' => 'https://app.payzec.com/payOrder?orderId=' . payzec_encrypt($order_id) . '&site=' . payzec_getsiteurl()
            );
        }

        public function receipt_page($order) {
            echo '<p>Thank you for your order, please click the button below to pay with Payzec.</p>';
        }

        public function fields() {
            return '
        <div class="form-row form-row-wide"><label>Card Holder Name <span class="required">*</span></label>
		<input id="payzec_ccName" type="text" autocomplete="off">
		</div>
        <div class="form-row form-row-wide"><label>Card Number <span class="required">*</span></label>
		<input id="payzec_ccNo" type="text" autocomplete="off">
		</div>
		<div class="form-row form-row-first">
			<label>Expiry Date <span class="required">*</span></label>
			<input id="payzec_expdate" type="text" autocomplete="off" placeholder="MM / YY">
		</div>
		<div class="form-row form-row-last">
			<label>Card Code (CVC) <span class="required">*</span></label>
			<input id="payzec_cvv" type="password" autocomplete="off" placeholder="CVC">
		</div>
		<div class="clear"></div>';
        }

        public function validate_fields(){
            return true;
        }
    }
}

add_action( 'woocommerce_blocks_loaded', 'payzec_rudr_gateway_block_support' );
function payzec_rudr_gateway_block_support() {
    if( ! class_exists( 'Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType' ) ) {
    	return;
     }

    require_once __DIR__ . '/class-wc-payzec-gateway-blocks-support.php';

    add_action(
        'woocommerce_blocks_payment_method_type_registration',
        function( Automattic\WooCommerce\Blocks\Payments\PaymentMethodRegistry $payment_method_registry ) {
            $payment_method_registry->register( new WC_Payzec_Gateway_Blocks_Support );
        }
    );
}

add_action( 'before_woocommerce_init', 'payzec_rudr_cart_checkout_blocks_compatibility' );

function payzec_rudr_cart_checkout_blocks_compatibility() {

    if( class_exists( '\Automattic\WooCommerce\Utilities\FeaturesUtil' ) ) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility(
            'cart_checkout_blocks',
            __FILE__,
            true
        );
    }
}
