<?php
/**
 * Structured logging helper.
 *
 * @package Paypercut\Payments\Support
 */

declare(strict_types=1);

namespace Paypercut\Payments\Support;

use function function_exists;
use function get_option;
use function sprintf;
use function wc_get_logger;
use function wp_json_encode;

/**
 * Wrapper around WooCommerce's logger with a simple API.
 */
final class Logger {
	/**
	 * WooCommerce log source identifier.
	 *
	 * @var string
	 */
	private const SOURCE = 'paypercut-payments-for-woocommerce';

	/**
	 * Prevent instantiation.
	 */
	private function __construct() {}

	/**
	 * Log a debug level message.
	 *
	 * @param string               $message Message to log.
	 * @param array<string, mixed> $context Optional context.
	 */
	public static function debug( string $message, array $context = array() ): void {
		self::log( 'debug', $message, $context );
	}

	/**
	 * Log an info level message.
	 *
	 * @param string               $message Message to log.
	 * @param array<string, mixed> $context Optional context.
	 */
	public static function info( string $message, array $context = array() ): void {
		self::log( 'info', $message, $context );
	}

	/**
	 * Log an error level message.
	 *
	 * @param string               $message Message to log.
	 * @param array<string, mixed> $context Optional context.
	 */
	public static function error( string $message, array $context = array() ): void {
		self::log( 'error', $message, $context );
	}

	/**
	 * Write a log entry at the provided level.
	 *
	 * @param string               $level   Log severity.
	 * @param string               $message Message to log.
	 * @param array<string, mixed> $context Optional context values.
	 */
	public static function log( string $level, string $message, array $context = array() ): void {
		if ( ! self::should_log() || ! function_exists( 'wc_get_logger' ) ) {
			return;
		}

		wc_get_logger()->log(
			$level,
			self::format_message( $message, $context ),
			array( 'source' => self::SOURCE )
		);
	}

	/**
	 * Determine whether logging is enabled in gateway settings.
	 *
	 * @return bool
	 */
	private static function should_log(): bool {
		$gateway_option_keys = array(
			'woocommerce_paypercut_payments_settings',
			'woocommerce_paypercut_bnpl_settings',
		);

		foreach ( $gateway_option_keys as $option_key ) {
			$settings = get_option( $option_key, array() );

			if ( empty( $settings ) ) {
				continue;
			}

			if ( 'yes' === ( $settings['logging'] ?? 'no' ) ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Build the log message body.
	 *
	 * @param string               $message Message to log.
	 * @param array<string, mixed> $context Optional context values.
	 *
	 * @return string
	 */
	private static function format_message( string $message, array $context ): string {
		$version  = defined( 'PAYPERCUT_PAYMENTS_VERSION' ) ? PAYPERCUT_PAYMENTS_VERSION : 'dev';

		$lines = array(
			'',
			sprintf( '==== Paypercut Payments Version: %s ====', $version ),
			'',
			$message,
		);

		if ( ! empty( $context ) ) {
			$json = wp_json_encode( $context, JSON_PRETTY_PRINT );

			$lines[] = '';
			$lines[] = 'Context:';
			$lines[] = is_string( $json ) ? $json : 'n/a';
		}

		return implode( PHP_EOL, $lines ) . PHP_EOL;
	}
}
