<?php
/**
 * Verifies payment status when customer returns from hosted checkout.
 * This serves as a fallback if the webhook hasn't arrived yet.
 *
 * @package Paypercut\Payments\Services
 */

declare(strict_types=1);

namespace Paypercut\Payments\Services;

use Paypercut\Payments\Api\PaypercutApi;
use Paypercut\Payments\Gateway\PaypercutGateway;
use Paypercut\Payments\Support\Logger;
use Throwable;
use WC_Order;
use function wc_get_order;

final class ReturnPaymentVerifier {

	/**
	 * Verify payment status when customer lands on thank you page.
	 * This serves as a fallback if the webhook hasn't arrived yet.
	 *
	 * @param int $order_id Order ID.
	 *
	 * @return void
	 */
	public function verify_order_payment( int $order_id ): void {
		$order = wc_get_order( $order_id );

		if ( ! $order instanceof WC_Order ) {
			return;
		}

		if ( PaypercutGateway::ID !== $order->get_payment_method() ) {
			return;
		}

		$current_status = $order->get_status();
		Logger::info(
			'Payment status verification: Checking order',
			array(
				'order_id'       => $order_id,
				'current_status' => $current_status,
			)
		);

		$session_id = $order->get_meta( PaypercutGateway::META_SESSION_ID );

		if ( empty( $session_id ) ) {
			return;
		}

		$updater = new OrderStatusUpdater();
		if ( $updater->is_verification_attempted( $order ) ) {
			return;
		}

		$api_secret = $this->get_api_secret_from_gateway();

		if ( empty( $api_secret ) ) {
			Logger::error(
				'Payment status verification: API secret not found',
				array( 'order_id' => $order_id )
			);

			return;
		}

		$updater->mark_verification_attempted( $order );

		try {
			$api = new PaypercutApi( $api_secret );
			$session = $api->get_checkout_session( $session_id );

			$payment_status = (string) ( $session['payment_status'] ?? $session['status'] ?? '' );

			if ( '' === $payment_status ) {
				Logger::info(
					'Payment status verification: No payment status found',
					array(
						'order_id'   => $order_id,
						'session_id' => $session_id,
					)
				);
				return;
			}

			if ( 'paid' === $payment_status ) {
				$updater->mark_order_as_processing( $order, $session, 'Payment status verification' );
			}
		} catch ( Throwable $exception ) {
			Logger::error(
				'Payment status verification: Failed to verify payment',
				array(
					'order_id'  => $order_id,
					'session_id' => $session_id,
				)
			);
		}
	}

	/**
	 * Get API secret from gateway settings.
	 *
	 * @return string
	 */
	private function get_api_secret_from_gateway(): string {
		$gateway_id = 'woocommerce_' . PaypercutGateway::ID . '_settings';
		$settings   = get_option( $gateway_id, array() );

		return isset( $settings['api_client_secret'] ) ? (string) $settings['api_client_secret'] : '';
	}
}

