<?php
/**
 * Thin wrapper around PaypercutApi for Paypercut checkout sessions.
 *
 * Supports both the core Paypercut API and the BNPL checkout API via
 * a configurable base URI.
 *
 * @package Paypercut\Payments\Services
 */

declare(strict_types=1);

namespace Paypercut\Payments\Services;

use Paypercut\Payments\Api\PaypercutApi;
use Paypercut\Payments\Support\Logger;
use Throwable;

final class PaypercutCheckoutClient {
	private PaypercutApi $api;

	/**
	 * Private constructor. Use factory methods instead.
	 *
	 * @param string $api_secret API secret.
	 * @param string $base_uri    Base URI for the API.
	 */
	private function __construct( string $api_secret, string $base_uri ) {
		$this->api = new PaypercutApi( $api_secret, null, $base_uri );
	}

	/**
	 * Create a client for Paypercut payments API calls.
	 *
	 * @param string $api_secret API secret.
	 *
	 * @return self
	 */
	public static function forPayments( string $api_secret ): self {
		return new self( $api_secret, PaypercutApi::DEFAULT_BASE_URI );
	}

	/**
	 * Create a client for BNPL API calls.
	 *
	 * @param string $api_secret API secret.
	 * @param string $mode        Mode: 'test' or 'live'. Defaults to 'test'.
	 *
	 * @return self
	 */
	public static function forBnpl( string $api_secret, string $mode = 'test' ): self {
		$base_uri = 'live' === $mode
			? PaypercutApi::BNPL_LIVE_BASE_URI
			: PaypercutApi::BNPL_TEST_BASE_URI;

		return new self( $api_secret, $base_uri );
	}

	public function create( array $payload, array $options = array() ): array {
		try {
			$response = $this->api->create_checkout_session( $payload, $options );
			$session_id = $response['id'] ?? '';
			Logger::info(
				'Paypercut checkout session response received.',
				array(
					'session_id' => $session_id,
				)
			);

			return $response;
		} catch ( Throwable $exception ) {
			Logger::error( 'Failed creating Paypercut checkout session.' );
			throw $exception;
		}
	}

	/**
	 * Create a BNPL attempt via the BNPL API.
	 *
	 * @param array<string,mixed> $payload
	 * @param array<string,mixed> $options
	 *
	 * @return array<string,mixed>
	 */
	public function create_bnpl_attempt( array $payload, array $options = array() ): array {
		try {
			$response = $this->api->create_bnpl_attempt( $payload, $options );
			return $response;
		} catch ( Throwable $exception ) {
			Logger::error( 'Failed creating Paypercut BNPL attempt.' );
			throw $exception;
		}
	}
}
