<?php
/**
 * Handles order status updates based on Paypercut payment status.
 *
 * @package Paypercut\Payments\Services
 */

declare(strict_types=1);

namespace Paypercut\Payments\Services;

use Paypercut\Payments\Gateway\PaypercutGateway;
use Paypercut\Payments\Support\Logger;
use WC_Order;

final class OrderStatusUpdater {

	/**
	 * Update order to processing status when payment is confirmed as paid.
	 *
	 * @param WC_Order            $order          Order instance.
	 * @param array<string,mixed> $checkout_data  Checkout session data from Paypercut.
	 * @param string              $context        Context for logging (e.g., 'Embedded checkout', 'Webhook', 'Payment status verification').
	 *
	 * @return bool True if order was updated, false otherwise.
	 */
	public function mark_order_as_processing( WC_Order $order, array $checkout_data, string $context ): bool {
		$order_id       = $order->get_id();
		$current_status = $order->get_status();

		if ( ! $order->has_status( array( 'pending', 'on-hold', 'failed' ) ) ) {
			Logger::info(
				sprintf( '%s: Payment confirmed, but order already processed', $context ),
				array(
					'order_id'       => $order_id,
					'current_status' => $current_status,
				)
			);
			return false;
		}

		$payment_id     = isset( $checkout_data['payment_id'] ) ? (string) $checkout_data['payment_id'] : '';
		$session_id     = $checkout_data['id'] ?? '';
		$transaction_id = $payment_id ?: (string) $session_id;

		$order->set_transaction_id( $transaction_id );
		$order->update_status( 'processing', $this->get_order_note( $context ) );
		$order->save();

		Logger::info(
			sprintf( '%s: Order status updated to processing', $context ),
			array(
				'order_id'       => $order_id,
				'old_status'     => $current_status,
				'new_status'     => 'processing',
				'transaction_id' => $transaction_id,
				'session_id'     => $session_id,
			)
		);

		return true;
	}

	/**
	 * Mark order as failed or cancelled based on payment status.
	 *
	 * @param WC_Order $order          Order instance.
	 * @param string   $payment_status Payment status from Paypercut.
	 * @param string   $context        Context for logging.
	 *
	 * @return bool True if order was updated, false otherwise.
	 */
	public function mark_order_as_failed( WC_Order $order, string $payment_status, string $context ): bool {
		$order_id       = $order->get_id();
		$current_status = $order->get_status();

		if ( $order->has_status( array( 'cancelled', 'refunded', 'failed' ) ) ) {
			return false;
		}

		$failed_statuses = array( 'failed', 'cancelled', 'expired', 'unpaid' );
		if ( ! in_array( $payment_status, $failed_statuses, true ) ) {
			return false;
		}

		$new_status = ( 'failed' === $payment_status ) ? 'failed' : 'cancelled';
		$order->update_status( $new_status, sprintf( 'Paypercut payment status: %s.', $payment_status ) );

		Logger::info(
			sprintf( '%s: Order status updated', $context ),
			array(
				'order_id'       => $order_id,
				'old_status'     => $current_status,
				'new_status'     => $new_status,
				'payment_status' => $payment_status,
			)
		);

		return true;
	}

	/**
	 * Mark verification as attempted to prevent duplicate verification.
	 *
	 * @param WC_Order $order Order instance.
	 *
	 * @return void
	 */
	public function mark_verification_attempted( WC_Order $order ): void {
		$order->update_meta_data( PaypercutGateway::META_VERIFICATION_ATTEMPTED, time() );
		$order->save_meta_data();
	}

	/**
	 * Check if verification has already been attempted.
	 *
	 * @param WC_Order $order Order instance.
	 *
	 * @return bool
	 */
	public function is_verification_attempted( WC_Order $order ): bool {
		return (bool) $order->get_meta( PaypercutGateway::META_VERIFICATION_ATTEMPTED );
	}

	/**
	 * Get order note message based on context.
	 *
	 * @param string $context Context of the update.
	 *
	 * @return string
	 */
	private function get_order_note( string $context ): string {
		$notes = array(
			'Embedded checkout'               => 'Payment verified and completed via Paypercut embedded checkout.',
			'Webhook'                         => 'Payment completed via Paypercut webhook.',
			'Payment status verification'     => 'Payment verified (webhook may arrive later).',
		);

		return $notes[ $context ] ?? 'Payment verified via Paypercut.';
	}
}

