<?php
/**
 * Produces checkout payloads for Paypercut sessions.
 *
 * @package Paypercut\Payments\Services
 */

declare(strict_types=1);

namespace Paypercut\Payments\Services;

use Paypercut\Payments\Gateway\PaypercutGateway;
use Paypercut\Payments\Gateway\PaypercutBnplGateway;
use WC_Cart;
use WC_Order;
use WC_Order_Item_Product;
use function __;
use function array_merge;
use function get_current_user_id;
use function get_woocommerce_currency;
use function sanitize_email;
use function sanitize_text_field;
use function wp_get_current_user;
use function wc_get_price_decimals;
use function wc_get_checkout_url;
use function wc_get_endpoint_url;

final class CheckoutPayloadFactory {
	public function build( WC_Cart $cart, PaypercutGateway $gateway ): array {
		$currency = get_woocommerce_currency();
		$decimals = wc_get_price_decimals();

		$total = (float) $cart->total;

		$amount = $this->to_minor_units( $total, $decimals );

		$line_items = array();

		foreach ( $cart->get_cart() as $cart_item ) {
			$product_name = isset( $cart_item['data'] ) && $cart_item['data'] ? $cart_item['data']->get_name() : __( 'Item', 'paypercut-payments-for-woocommerce' );
			$quantity     = (int) ( $cart_item['quantity'] ?? 1 );
			$line_subtotal = (float) ( $cart_item['line_subtotal'] ?? 0 );
			$unit_price   = $this->to_minor_units( $quantity > 0 ? ( $line_subtotal / $quantity ) : 0, $decimals );

			$line_items[] = array_filter(
				array(
					'name'        => $product_name,
					'description' => isset( $cart_item['product_id'] ) ? (string) $cart_item['product_id'] : '',
					'quantity'    => $quantity,
					'unit_price'  => $unit_price,
				)
			);
		}

		return $this->assemble_payload(
			$gateway,
			'embedded',
			$line_items,
			$amount,
			$currency,
			array(
				'locale'            => $this->resolve_locale(),
				'customer_id'       => $this->resolve_customer_id(),
				'customer_email'    => $this->resolve_customer_email( $cart ),
				'metadata'          => array(),
			)
		);
	}

	public function build_from_order( WC_Order $order, PaypercutGateway $gateway ): array {
		$decimals = wc_get_price_decimals();
		$currency = $order->get_currency() ?: get_woocommerce_currency();
		$amount   = $this->to_minor_units( (float) $order->get_total(), $decimals );

		$line_items = array();

		foreach ( $order->get_items() as $item ) {
			if ( ! $item instanceof WC_Order_Item_Product ) {
				continue;
			}

			$quantity   = max( 1, (int) $item->get_quantity() );
			$line_total = (float) $item->get_total();
			$unit_price = $this->to_minor_units( $quantity > 0 ? ( $line_total / $quantity ) : 0, $decimals );

			$line_items[] = array_filter(
				array(
					'name'        => $item->get_name() ?: __( 'Item', 'paypercut-payments-for-woocommerce' ),
					'description' => $item->get_product_id() ? (string) $item->get_product_id() : '',
					'quantity'    => $quantity,
					'unit_price'  => $unit_price,
				)
			);
		}

		$payload = $this->assemble_payload(
			$gateway,
			'hosted',
			$line_items,
			$amount,
			$currency,
			array(
				'locale'            => $this->resolve_locale(),
				'customer_id'       => $this->resolve_order_customer_id( $order ),
				'customer_email'    => $this->resolve_order_email( $order ),
				'client_reference_id' => (string) $order->get_id(),
				'metadata'          => array(
					'orderId' => (string) $order->get_id(),
				),
			)
		);

		$order_success_url = $order->get_checkout_order_received_url();
		$checkout_url      = wc_get_checkout_url();

		$payload['success_url'] = $order_success_url;
		$payload['return_url']  = $order_success_url;
		$payload['cancel_url']  = $checkout_url;

		return $payload;
	}

	public function build_setup( PaypercutGateway $gateway ): array {
		$currency = get_woocommerce_currency();

		return $this->assemble_payload(
			$gateway,
			'embedded',
			array(),
			0,
			$currency,
			array(
				'locale'             => $this->resolve_locale(),
				'customer_id'        => $this->resolve_customer_id(),
				'customer_email'     => $this->resolve_current_user_email(),
				'metadata'           => array(),
			),
			'setup',
			'auto'
		);
	}

	private function resolve_locale(): string {
		$locale = str_replace( '_', '-', get_locale() );
		
		$supported_locales = array(
			'bg-BG', 'cs-CZ', 'da-DK', 'de-DE', 'el-GR', 'en-GB',
			'es-ES', 'et-EE', 'fi-FI', 'fr-FR', 'hr-HR', 'hu-HU',
			'it-IT', 'lt-LT', 'lv-LV', 'mt-MT', 'nl-NL', 'pl-PL',
			'pt-PT', 'ro-RO', 'sk-SK', 'sl-SI', 'sv-SE',
		);

		return in_array( $locale, $supported_locales, true ) ? $locale : 'auto';
	}

	private function to_minor_units( float $value, int $decimals ): int {
		$multiplier = pow( 10, max( $decimals, 0 ) );

		return (int) round( $value * $multiplier );
	}

	private function resolve_customer_email( WC_Cart $cart ): string {
		$account_email = $this->resolve_current_user_email();

		if ( '' !== $account_email ) {
			return $account_email;
		}

		if ( method_exists( $cart, 'get_customer' ) && $cart->get_customer() && $cart->get_customer()->get_billing_email() ) {
			return sanitize_email( (string) $cart->get_customer()->get_billing_email() );
		}

		return '';
	}

	private function resolve_customer_id(): string {
		if ( get_current_user_id() ) {
			return (string) get_current_user_id();
		}

		return '';
	}

	private function resolve_current_user_email(): string {
		$user = wp_get_current_user();

		if ( $user && $user->user_email ) {
			return sanitize_email( (string) $user->user_email );
		}

		return '';
	}

	private function resolve_order_email( WC_Order $order ): string {
		return sanitize_email( (string) $order->get_billing_email() );
	}

	private function resolve_order_customer_id( WC_Order $order ): string {
		$customer_id = (int) $order->get_customer_id();

		if ( $customer_id > 0 ) {
			return (string) $customer_id;
		}

		return $this->resolve_customer_id();
	}

	private function assemble_payload(
		$gateway,
		string $ui_mode,
		array $line_items,
		int $amount,
		string $currency,
		array $context,
		string $mode = 'payment',
		string $submit_type = 'auto',
		array $payment_method_types = array( 'card' )
	): array {
		$metadata_defaults = array(
			'platform'         => 'woocommerce',
			'platform_version' => defined( 'WC_VERSION' ) ? (string) constant( 'WC_VERSION' ) : '',
			'plugin_version'   => defined( 'PAYPERCUT_PAYMENTS_VERSION' ) ? (string) constant( 'PAYPERCUT_PAYMENTS_VERSION' ) : '',
			'wordpress_version' => get_bloginfo( 'version' ),
			'php_version'       => PHP_VERSION,
			'site_url'          => home_url(),
		);

		$metadata = array_merge(
			$metadata_defaults,
			$context['metadata'] ?? array()
		);

		$checkout_url = wc_get_checkout_url();
		$success_url  = wc_get_endpoint_url( 'order-received', '', $checkout_url );

		$payload = array(
			'mode'                 => $mode,
			'ui_mode'              => $ui_mode,
			'payment_method_types' => $payment_method_types,
			'submit_type'          => $submit_type,
			'success_url'          => $success_url,
			'cancel_url'           => $checkout_url,
			'return_url'           => $success_url,
			'metadata'             => $metadata,
			'wallet_options'       => array(
				'apple_pay'  => array( 'display' => 'auto' ),
				'google_pay' => array( 'display' => 'auto' ),
			),
			'saved_payment_method_options' => method_exists( $gateway, 'allows_saved_payment_methods' )
				? array(
					'payment_method_save' => $gateway->allows_saved_payment_methods() ? 'enabled' : 'disabled',
				)
				: array(),
			'amount_total'         => $amount,
			'amount'               => $amount,
			'currency'             => $currency,
			'line_items'           => $line_items,
			'locale'               => $context['locale'] ?? 'auto',
			'client_customer_id'   => $context['customer_id'] ?? '',
			'customer_email'       => $context['customer_email'] ?? '',
		);

		if ( 'setup' === $mode ) {
			$payload['setup_intent_data'] = array( 'usage' => 'off_session' );
		} elseif ( 'payment' === $mode && method_exists( $gateway, 'allows_saved_payment_methods' ) && $gateway->allows_saved_payment_methods() ) {
			$payload['payment_intent_data'] = array( 'setup_future_usage' => 'on_session' );
		}

		return $payload;
	}

	/**
	 * Build BNPL attempt payload from an order.
	 *
	 * @param WC_Order           $order   Order instance.
	 * @param PaypercutBnplGateway $gateway BNPL gateway.
	 *
	 * @return array<string,mixed>
	 */
	public function build_bnpl_from_order( WC_Order $order, PaypercutBnplGateway $gateway ): array {
		$currency = $order->get_currency() ?: get_woocommerce_currency();
		$decimals = wc_get_price_decimals();

		$order_success_url = $order->get_checkout_order_received_url();
		$checkout_url      = wc_get_checkout_url();

		$billing_address = array(
			'person'  => array(
				'first_name'   => (string) $order->get_billing_first_name(),
				'last_name'    => (string) $order->get_billing_last_name(),
				'other_names'  => '',
				'phone'        => (string) $order->get_billing_phone(),
				'email'        => $this->resolve_order_email( $order ),
				'national_id'  => '',
			),
			'address' => array(
				'country_code'   => (string) $order->get_billing_country(),
				'city'           => (string) $order->get_billing_city(),
				'postal_code'    => (string) $order->get_billing_postcode(),
				'street1'        => (string) $order->get_billing_address_1(),
				'street2'        => (string) $order->get_billing_address_2(),
				'country_region' => '',
				'district'       => '',
				'block'          => '',
				'entrance'       => '',
				'floor'          => '',
				'apartment'      => '',
			),
		);

		$shipping_address = array(
			'person'  => array(
				'first_name'   => (string) ( $order->get_shipping_first_name() ?: $order->get_billing_first_name() ),
				'last_name'    => (string) ( $order->get_shipping_last_name() ?: $order->get_billing_last_name() ),
				'other_names'  => '',
				'phone'        => (string) $order->get_billing_phone(),
				'email'        => $this->resolve_order_email( $order ),
				'national_id'  => '',
			),
			'address' => array(
				'country_code'   => (string) ( $order->get_shipping_country() ?: $order->get_billing_country() ),
				'city'           => (string) ( $order->get_shipping_city() ?: $order->get_billing_city() ),
				'postal_code'    => (string) ( $order->get_shipping_postcode() ?: $order->get_billing_postcode() ),
				'street1'        => (string) ( $order->get_shipping_address_1() ?: $order->get_billing_address_1() ),
				'street2'        => (string) ( $order->get_shipping_address_2() ?: $order->get_billing_address_2() ),
				'country_region' => '',
				'district'       => '',
				'block'          => '',
				'entrance'       => '',
				'floor'          => '',
				'apartment'      => '',
			),
		);

		$items = array();

		foreach ( $order->get_items() as $item ) {
			if ( ! $item instanceof WC_Order_Item_Product ) {
				continue;
			}

			$product      = $item->get_product();
			$quantity     = max( 1, (int) $item->get_quantity() );
			$unit_price   = (float) $item->get_total() / $quantity;
			$unit_price_minor = $this->to_minor_units( $unit_price, $decimals );
			$image_url    = $product && $product->get_image_id() ? wp_get_attachment_url( $product->get_image_id() ) : '';
			$item_url     = $product ? get_permalink( $product->get_id() ) : '';

			$items[] = array(
				'title'        => $item->get_name() ?: __( 'Item', 'paypercut-payments-for-woocommerce' ),
				'image_url'    => $image_url ?: '',
				'item_url'     => $item_url ?: '',
				'sku'          => $product && $product->get_sku() ? (string) $product->get_sku() : '',
				'item_version' => '',
				'qty'          => (string) $quantity,
				'unit_price'   => (string) $unit_price_minor,
			);
		}

		$shipping_amount = (float) $order->get_shipping_total();
		$tax_amount      = (float) $order->get_total_tax();
		$discount_amount = (float) $order->get_discount_total();
		$total_amount    = (float) $order->get_total();

		$shipping_amount_minor = $this->to_minor_units( $shipping_amount, $decimals );
		$tax_amount_minor      = $this->to_minor_units( $tax_amount, $decimals );
		$discount_amount_minor = $this->to_minor_units( $discount_amount, $decimals );
		$total_amount_minor    = $this->to_minor_units( $total_amount, $decimals );

		$payload = array(
			'application_id'        => '',
			'merchant_purchase_ref' => (string) $order->get_id(),
			'customer_redirect'     => array(
				'approved_url' => $order_success_url,
				'canceled_url' => $checkout_url,
			),
			'purchase_details'      => array(
				'billing_address'  => $billing_address,
				'shipping_address' => $shipping_address,
				'customer'         => array(
					'customer_ref' => $this->resolve_order_customer_id( $order ),
				),
				'shopping_card'    => array(
					'currency_code'  => $currency,
					'shipping_amount'=> (string) $shipping_amount_minor,
					'tax_amount'     => (string) $tax_amount_minor,
					'total_amount'   => (string) $total_amount_minor,
					'discount_amount'=> (string) $discount_amount_minor,
					'items'          => $items,
				),
			),
			'transaction_flow_config' => array(
				'transaction_capture' => 'TRANSACTION_CAPTURE_UNSPECIFIED',
			),
		);

		return $payload;
	}
}
