<?php
/**
 * Verifies BNPL attempt status when customer returns from BNPL hosted checkout.
 * This serves as a fallback if the BNPL webhook hasn't arrived yet.
 *
 * @package Paypercut\Payments\Services
 */

declare(strict_types=1);

namespace Paypercut\Payments\Services;

use Paypercut\Payments\Api\PaypercutApi;
use Paypercut\Payments\Gateway\PaypercutBnplGateway;
use Paypercut\Payments\Support\Logger;
use Throwable;
use WC_Order;
use function get_option;
use function wc_get_order;

final class BnplReturnVerifier {

	/**
	 * Verify BNPL attempt status when customer lands on thank you page.
	 *
	 * @param int $order_id Order ID.
	 *
	 * @return void
	 */
	public function verify_order_payment( int $order_id ): void {
		$order = wc_get_order( $order_id );

		if ( ! $order instanceof WC_Order ) {
			return;
		}

		if ( PaypercutBnplGateway::ID !== $order->get_payment_method() ) {
			return;
		}

		$current_status = $order->get_status();
		Logger::info(
			'BNPL attempt verification: Checking order',
			array(
				'order_id'       => $order_id,
				'current_status' => $current_status,
			)
		);

		$attempt_id = (string) $order->get_meta( PaypercutBnplGateway::META_BNPL_ATTEMPT_ID, true );

		if ( '' === $attempt_id ) {
			Logger::info(
				'BNPL attempt verification: No attempt ID on order, skipping',
				array( 'order_id' => $order_id )
			);
			return;
		}

		$credentials = $this->get_bnpl_credentials();
		$api_secret  = $credentials['api_secret'];
		$base_uri    = $credentials['base_uri'];

		if ( '' === $api_secret || '' === $base_uri ) {
			Logger::error(
				'BNPL attempt verification: API credentials not configured',
				array( 'order_id' => $order_id )
			);
			return;
		}

		$updater = new OrderStatusUpdater();

		try {
			$api     = new PaypercutApi( $api_secret, null, $base_uri );
			$payload = $api->get_bnpl_attempt_status( $attempt_id );

			$attempt = isset( $payload['attempt'] ) && is_array( $payload['attempt'] )
				? $payload['attempt']
				: array();

			// Validate attempt_id matches the stored one
			$response_attempt_id = $attempt['attempt_id'] ?? '';
			if ( empty( $response_attempt_id ) || $response_attempt_id !== $attempt_id ) {
				Logger::error(
					'BNPL attempt verification: attempt_id mismatch - response attempt_id does not match order stored attempt_id',
					array(
						'order_id'            => $order_id,
						'response_attempt_id' => $response_attempt_id,
						'stored_attempt_id'   => $attempt_id,
					)
				);
				return;
			}

			// Validate merchant_purchase_ref matches the order ID
			$merchant_purchase_ref = $attempt['merchant_purchase_ref'] ?? '';
			$order_id_string = (string) $order_id;
			if ( empty( $merchant_purchase_ref ) || $merchant_purchase_ref !== $order_id_string ) {
				Logger::error(
					'BNPL attempt verification: merchant_purchase_ref mismatch - response merchant_purchase_ref does not match order ID',
					array(
						'order_id'                => $order_id,
						'merchant_purchase_ref'   => $merchant_purchase_ref,
					)
				);
				return;
			}

			$status = (string) ( $attempt['status'] ?? '' );

			if ( '' === $status ) {
				Logger::info(
					'BNPL attempt verification: No attempt status found',
					array(
						'order_id'   => $order_id,
						'attempt_id' => $attempt_id,
					)
				);
				return;
			}

			if ( 'ATTEMPT_STATUS_CAPTURED' === $status ) {
				$checkout_data = $attempt;
				if ( ! empty( $attempt['attempt_id'] ) ) {
					$checkout_data['id'] = $attempt['attempt_id'];
				}

				$updater->mark_order_as_processing( $order, $checkout_data, 'BNPL Attempt verification' );
				$order->add_order_note(
					sprintf(
						/* translators: %s: BNPL attempt ID */
						__( 'Paypercut BNPL attempt %s captured.', 'paypercut-payments-for-woocommerce' ),
						esc_html( $attempt['attempt_id'] ?? $attempt_id )
					)
				);
				$order->save();
				Logger::info(
					'BNPL Attempt verification: Order marked as processing',
					array(
						'order_id'      => $order->get_id(),
						'attempt_id'    => $attempt['attempt_id'] ?? $attempt_id,
						'attempt_status' => $status,
					)
				);
				return;
			}

			if ( 'ATTEMPT_STATUS_ERRORED' === $status ) {
				$order->update_status( 'failed', __( 'Paypercut BNPL attempt errored.', 'paypercut-payments-for-woocommerce' ) );
				$order->add_order_note(
					sprintf(
						__( 'Paypercut BNPL attempt %1$s errored. Reason: %2$s', 'paypercut-payments-for-woocommerce' ),
						esc_html( $attempt['attempt_id'] ?? $attempt_id ),
						esc_html( $attempt['status_reason'] ?? '' )
					)
				);
				$order->save();
				Logger::info(
					'BNPL Attempt verification: Order marked as failed',
					array(
						'order_id'      => $order->get_id(),
						'attempt_id'    => $attempt['attempt_id'] ?? $attempt_id,
						'attempt_status' => $status,
					)
				);
				return;
			}

			if ( 'ATTEMPT_STATUS_DECLINED' === $status ) {
				$order->update_status( 'cancelled', __( 'Paypercut BNPL attempt declined.', 'paypercut-payments-for-woocommerce' ) );
				$order->add_order_note(
					sprintf(
						__( 'Paypercut BNPL attempt %1$s declined. Reason: %2$s', 'paypercut-payments-for-woocommerce' ),
						esc_html( $attempt['attempt_id'] ?? $attempt_id ),
						esc_html( $attempt['status_reason'] ?? '' )
					)
				);
				$order->save();
				Logger::info(
					'BNPL Attempt verification: Order marked as cancelled',
					array(
						'order_id'      => $order->get_id(),
						'attempt_id'    => $attempt['attempt_id'] ?? $attempt_id,
						'attempt_status' => $status,
					)
				);
				return;
			}

			Logger::info(
				'BNPL Attempt verification: unhandled attempt status',
				array(
					'order_id'       => $order->get_id(),
					'attempt_status' => $status,
					'attempt_id'     => $attempt['attempt_id'] ?? $attempt_id,
				)
			);
		} catch ( Throwable $exception ) {
			Logger::error(
				'BNPL attempt verification: Failed to verify BNPL attempt',
				array(
					'order_id'   => $order_id,
					'attempt_id' => $attempt_id,
					'error'      => $exception->getMessage(),
				)
			);
		}
	}

	/**
	 * Get BNPL API credentials and base URI from gateway settings.
	 *
	 * @return array{api_secret:string,base_uri:string}
	 */
	private function get_bnpl_credentials(): array {
		$gateway_id = 'woocommerce_' . PaypercutBnplGateway::ID . '_settings';
		$settings   = get_option( $gateway_id, array() );

		$api_secret = isset( $settings['api_client_secret'] ) ? (string) $settings['api_client_secret'] : '';
		$mode       = isset( $settings['mode'] ) ? (string) $settings['mode'] : 'test';

		$base_uri = 'live' === $mode
			? PaypercutApi::BNPL_LIVE_BASE_URI
			: PaypercutApi::BNPL_TEST_BASE_URI;

		return array(
			'api_secret' => $api_secret,
			'base_uri'   => $base_uri,
		);
	}
}

