<?php
/**
 * Plugin bootstrap.
 *
 * @package Paypercut\Payments
 */

declare(strict_types=1);

namespace Paypercut\Payments;

use Paypercut\Payments\Assets\AdminAssets;
use Paypercut\Payments\Assets\FrontendAssets;
use Paypercut\Payments\Gateway\PaypercutBnplGateway;
use Paypercut\Payments\Gateway\PaypercutGateway;
use Paypercut\Payments\Gateway\PaypercutPaymentToken;
use Paypercut\Payments\Http\Ajax\CreateCheckoutSession as CreateCheckoutSessionAjax;
use Paypercut\Payments\Http\Ajax\DeleteWebhook as DeleteWebhookAjax;
use Paypercut\Payments\Http\Ajax\SetupWizard as SetupWizardAjax;
use Paypercut\Payments\Http\Ajax\TestConnection as TestConnectionAjax;
use Paypercut\Payments\Http\Rest\CheckoutSessionWebhook;
use Paypercut\Payments\Http\Rest\BnplWebhook;
use Paypercut\Payments\Services\CheckoutPayloadFactory;
use Paypercut\Payments\Services\BnplReturnVerifier;
use Paypercut\Payments\Services\ReturnPaymentVerifier;
use WC_Order;

final class Plugin {

	private AdminAssets $admin_assets;

	private FrontendAssets $frontend_assets;

	public function __construct() {
		$this->admin_assets    = new AdminAssets();
		$this->frontend_assets = new FrontendAssets();
	}

	/**
	 * Boot the plugin.
	 */
	public function boot(): void {
		$this->load_textdomain();

		add_filter( 'woocommerce_payment_token_class', array( $this, 'register_payment_token_class' ), 10, 2 );
		add_filter( 'woocommerce_payment_gateways', array( $this, 'register_gateway' ) );
		add_action( 'admin_enqueue_scripts', array( $this->admin_assets, 'enqueue' ) );
		add_action( 'wp_enqueue_scripts', array( $this->frontend_assets, 'enqueue' ) );

		// Register AJAX handlers
		$checkout_ajax = new CreateCheckoutSessionAjax( new CheckoutPayloadFactory() );
		add_action( 'wp_ajax_paypercut_create_checkout_session', array( $checkout_ajax, 'handle' ) );
		add_action( 'wp_ajax_nopriv_paypercut_create_checkout_session', array( $checkout_ajax, 'handle' ) );

		$setup_wizard_ajax = new SetupWizardAjax();
		add_action( 'wp_ajax_paypercut_setup_wizard', array( $setup_wizard_ajax, 'handle' ) );

		$test_connection_ajax = new TestConnectionAjax();
		add_action( 'wp_ajax_paypercut_test_connection', array( $test_connection_ajax, 'handle' ) );

		$delete_webhook_ajax = new DeleteWebhookAjax();
		add_action( 'wp_ajax_paypercut_delete_webhook', array( $delete_webhook_ajax, 'handle' ) );

		add_action(
			'rest_api_init',
			static function(): void {
				( new CheckoutSessionWebhook() )->register_routes();
				( new BnplWebhook() )->register_routes();
			}
		);

		$return_verifier = new ReturnPaymentVerifier();
		add_action( 'woocommerce_before_thankyou', array( $return_verifier, 'verify_order_payment' ), 10, 1 );

		$bnpl_return_verifier = new BnplReturnVerifier();
		add_action( 'woocommerce_before_thankyou', array( $bnpl_return_verifier, 'verify_order_payment' ), 10, 1 );
		
		add_action( 'woocommerce_rest_checkout_process_payment_with_context', array( $this, 'process_payment_for_blocks' ), 10, 2 );

		add_filter( 'woocommerce_my_account_my_orders_actions', array( $this, 'remove_pay_cancel_actions' ), 10, 2 );
	}

	/**
	 * Load plugin textdomain for translations.
	 *
	 * Note: For WordPress.org plugins, translations are auto-loaded by WordPress.
	 * This method is intentionally empty for WordPress.org compatibility.
	 *
	 * @return void
	 */
	private function load_textdomain(): void {
	}

	/**
	 * Register payment token class for Paypercut tokens.
	 *
	 * @param string $class Default token class name.
	 * @param string $type  Token type.
	 * @return string Token class name.
	 */
	public function register_payment_token_class( string $class, string $type ): string {
		if ( 'Paypercut_CC' === $type ) {
			if ( class_exists( PaypercutPaymentToken::class ) ) {
				return PaypercutPaymentToken::class;
			}
		}

		return $class;
	}

	/**
	 * @param array<int, string> $gateways
	 */
	public function register_gateway( array $gateways ): array {
		$gateways[] = PaypercutGateway::class;
		$gateways[] = PaypercutBnplGateway::class;

		return $gateways;
	}
	
	/**
	 * Process payment for Blocks checkout.
	 *
	 * @param object $context PaymentContext object containing order and payment data
	 * @param object $result PaymentResult object to set the payment result
	 */
	public function process_payment_for_blocks( $context, &$result ): void {
		$payment_method_id = $context->payment_method;
		
		if ( PaypercutGateway::ID !== $payment_method_id ) {
			return;
		}
		
		$order = $context->order;
		$payment_data = $context->payment_data;
		
		foreach ( $payment_data as $key => $value ) {
			$_POST[ $key ] = $value;
		}
		
		$gateways = WC()->payment_gateways()->get_available_payment_gateways();
		
		if ( empty( $gateways[ PaypercutGateway::ID ] ) ) {
			throw new \Exception( esc_html( __( 'Payment gateway not available.', 'paypercut-payments-for-woocommerce' ) ) );
		}
		
		$gateway = $gateways[ PaypercutGateway::ID ];
		$payment_result = $gateway->process_payment( $order->get_id() );
		
		if ( isset( $payment_result['result'] ) && 'success' === $payment_result['result'] ) {
			$result->set_status( 'success' );
			$result->set_redirect_url( $payment_result['redirect'] );
		} else {
			$result->set_status( 'failure' );
		}
	}

	/**
	 * Remove pay and cancel actions from My Account orders list for Paypercut orders.
	 *
	 * @param array    $actions Order actions.
	 * @param WC_Order $order   Order object.
	 * @return array Filtered order actions.
	 */
	public function remove_pay_cancel_actions( array $actions, $order ): array {
		if ( ! $order instanceof WC_Order ) {
			return $actions;
		}

		$payment_method = $order->get_payment_method();

		if ( PaypercutGateway::ID === $payment_method || PaypercutBnplGateway::ID === $payment_method ) {
			unset( $actions['pay'], $actions['cancel'] );
		}

		return $actions;
	}
}
