<?php
/**
 * REST endpoint for handling Paypercut checkout session webhooks.
 *
 * @package Paypercut\Payments\Http\Rest
 */

declare(strict_types=1);

namespace Paypercut\Payments\Http\Rest;

use Paypercut\Payments\Api\PaypercutApi;
use Paypercut\Payments\Gateway\PaypercutGateway;
use Paypercut\Payments\Services\OrderStatusUpdater;
use Paypercut\Payments\Support\Logger;
use Throwable;
use WC_Order;
use WP_REST_Request;
use WP_REST_Response;
use WP_Error;
use function wc_get_order;

/**
 * Handles checkout_session.completed webhook events.
 *
 * Updates WooCommerce order status based on checkout session completion.
 */
final class CheckoutSessionWebhook extends AbstractWebhook {

	/**
	 * Get the webhook event type this handler processes.
	 *
	 * @return string
	 */
	protected function get_webhook_event_type(): string {
		return 'checkout_session.completed';
	}

	/**
	 * Get the REST API route path for this webhook.
	 *
	 * @return string
	 */
	protected function get_route_path(): string {
		return '/webhook/checkout-session';
	}

	/**
	 * Register REST API routes for this webhook.
	 *
	 * Note: This endpoint is publicly accessible (no WordPress user authentication required)
	 * because it must be callable by Paypercut's servers. Security is ensured through:
	 * 1. Signature verification via verify_webhook_signature() (validates request signature)
	 * 2. API verification in the handle() method (verifies session with Paypercut's API before processing)
	 */
	public function register_routes(): void {
		register_rest_route(
			'paypercut/v1',
			$this->get_route_path(),
			array(
				'methods'             => \WP_REST_Server::CREATABLE,
				'callback'            => array( $this, 'handle' ),
				'permission_callback' => array( $this, 'verify_webhook_signature' ),
			)
		);
	}

	/**
	 * Verify webhook signature using Paypercut's signature algorithm.
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return bool|WP_Error True if signature is valid, WP_Error otherwise.
	 */
	public function verify_webhook_signature( WP_REST_Request $request ) {
		$gateway = $this->get_gateway_instance();
		if ( ! $gateway ) {
			Logger::error( 'Webhook: gateway instance not available' );
			return new \WP_Error( 'gateway_unavailable', __( 'Gateway not available.', 'paypercut-payments-for-woocommerce' ), array( 'status' => 500 ) );
		}

		$webhook_secret = $gateway->get_webhook_secret();
		if ( empty( $webhook_secret ) ) {
			Logger::error( 'Webhook: webhook secret not configured, signature verification required' );
			return new \WP_Error( 'webhook_secret_not_configured', __( 'Webhook secret not configured. Please set up webhooks in gateway settings.', 'paypercut-payments-for-woocommerce' ), array( 'status' => 500 ) );
		}

		$raw_body = $request->get_body();
		if ( empty( $raw_body ) ) {
			Logger::error( 'Webhook: empty request body' );
			return new \WP_Error( 'empty_body', __( 'Empty request body.', 'paypercut-payments-for-woocommerce' ), array( 'status' => 400 ) );
		}

		$signature_header = $request->get_header( 'Paypercut-Signature' );
		if ( empty( $signature_header ) ) {
			Logger::error( 'Webhook: missing Paypercut-Signature header' );
			return new \WP_Error( 'missing_signature', __( 'Missing signature header.', 'paypercut-payments-for-woocommerce' ), array( 'status' => 401 ) );
		}

		$parts = explode( ',', $signature_header );
		if ( count( $parts ) !== 2 ) {
			Logger::error( 'Webhook: invalid signature header format' );
			return new \WP_Error( 'invalid_signature_format', __( 'Invalid signature format.', 'paypercut-payments-for-woocommerce' ), array( 'status' => 401 ) );
		}

		$timestamp_part = explode( '=', $parts[0] );
		$signature_part = explode( '=', $parts[1] );

		if ( count( $timestamp_part ) !== 2 || count( $signature_part ) !== 2 || 't' !== $timestamp_part[0] || 'v1' !== $signature_part[0] ) {
			Logger::error( 'Webhook: invalid signature header structure' );
			return new \WP_Error( 'invalid_signature_structure', __( 'Invalid signature structure.', 'paypercut-payments-for-woocommerce' ), array( 'status' => 401 ) );
		}

		$timestamp         = (int) $timestamp_part[1];
		$received_signature = $signature_part[1];

		$current_time    = time();
		$time_difference = abs( $current_time - $timestamp );
		$timestamp_tolerance = 300;

		if ( $time_difference > $timestamp_tolerance ) {
			Logger::error(
				'Webhook: timestamp out of tolerance',
				array(
					'timestamp'       => $timestamp,
					'current_time'    => $current_time,
					'time_difference' => $time_difference,
				)
			);
			return new \WP_Error( 'timestamp_out_of_tolerance', __( 'Webhook timestamp is too old or too far in the future.', 'paypercut-payments-for-woocommerce' ), array( 'status' => 401 ) );
		}

		$signed_payload     = $timestamp . '.' . $raw_body;
		$expected_signature = hash_hmac( 'sha256', $signed_payload, $webhook_secret );

		if ( ! hash_equals( $expected_signature, $received_signature ) ) {
			Logger::error( 'Webhook: signature verification failed' );
			return new \WP_Error( 'invalid_signature', __( 'Invalid webhook signature.', 'paypercut-payments-for-woocommerce' ), array( 'status' => 401 ) );
		}

		Logger::info( 'Webhook: signature verified successfully' );
		return true;
	}

	/**
	 * Handle incoming webhook request.
	 *
	 * @param WP_REST_Request $request Incoming REST request.
	 *
	 * @return WP_REST_Response|WP_Error
	 */
	public function handle( WP_REST_Request $request ) {
		$delivery_id = $request->get_header( 'Paypercut-Delivery-Id' );
		if ( ! empty( $delivery_id ) ) {
			$processed = $this->is_delivery_processed( $delivery_id );
			if ( $processed ) {
				Logger::info(
					'Webhook: duplicate delivery detected',
					array( 'delivery_id' => $delivery_id )
				);
				return rest_ensure_response(
					new \WP_REST_Response(
						array(
							'received'  => true,
							'duplicate' => true,
						),
						200
					)
				);
			}
		}

		$payload = $request->get_json_params();

		Logger::info(
			'Checkout session webhook received payload',
			array(
				'payload' => $payload,
				'path'    => $request->get_route(),
			)
		);

		if ( ! $this->validate_payload( $payload ) ) {
			$webhook_type = $payload['type'] ?? '';
			if ( $webhook_type !== $this->get_webhook_event_type() ) {
				return $this->skip_webhook_response( 'unsupported_webhook_type', array( 'type' => $webhook_type ) );
			}
			return $this->create_error_response( 'Invalid webhook payload.', 400 );
		}

		$checkout = $this->extract_data_from_payload( $payload );
		if ( empty( $checkout ) ) {
			Logger::error( 'Webhook: unable to locate checkout data in payload' );
			return $this->skip_webhook_response( 'no_checkout_object' );
		}

		$session_id = (string) ( $checkout['id'] ?? '' );
		if ( '' === $session_id ) {
			Logger::error( 'Webhook: missing checkout session ID in payload' );
			return $this->skip_webhook_response( 'no_session_id' );
		}

		$verified_checkout = $this->get_verified_checkout_session( $session_id );
		if ( null === $verified_checkout ) {
			Logger::error(
				'Webhook: failed to verify checkout session with Paypercut API',
				array( 'session_id' => $session_id )
			);
			return $this->create_error_response( 'Unable to verify checkout session.', 401 );
		}

		$order_id = (int) ( $verified_checkout['metadata']['orderId'] ?? 0 );
		$order    = null;

		if ( $order_id > 0 ) {
			$order = wc_get_order( $order_id );
		}

		if ( ! $order instanceof WC_Order && ! empty( $session_id ) ) {
			$orders = wc_get_orders(
				array(
					'limit'      => 1,
					'meta_key'   => PaypercutGateway::META_SESSION_ID,
					'meta_value' => $session_id,
					'return'     => 'ids',
				)
			);

			if ( ! empty( $orders ) && is_array( $orders ) ) {
				$order_id = (int) $orders[0];
				$order    = wc_get_order( $order_id );
			}
		}

		if ( ! $order instanceof WC_Order ) {
			Logger::error(
				'Webhook: order not found',
				array(
					'order_id'   => $order_id,
					'session_id' => $session_id,
					'metadata'   => $verified_checkout['metadata'] ?? array(),
				)
			);
			return $this->skip_webhook_response( 'order_not_found' );
		}

		$payment_status = (string) ( $verified_checkout['payment_status'] ?? $verified_checkout['status'] ?? '' );
		if ( '' === $payment_status ) {
			Logger::error(
				'Webhook: no payment status found in verified checkout',
				array( 'order_id' => $order_id )
			);
			return $this->skip_webhook_response( 'no_payment_status' );
		}

		$this->update_order_from_checkout( $order, $verified_checkout, $payment_status );

		if ( ! empty( $delivery_id ) ) {
			$this->mark_delivery_as_processed( $delivery_id );
		}

		return rest_ensure_response(
			new WP_REST_Response(
				array(
					'received'       => true,
					'order_id'       => $order->get_id(),
					'payment_status' => $payment_status,
				),
				200
			)
		);
	}

	/**
	 * Check if a webhook delivery has already been processed.
	 *
	 * @param string $delivery_id Delivery ID from Paypercut-Delivery-Id header.
	 * @return bool True if already processed, false otherwise.
	 */
	private function is_delivery_processed( string $delivery_id ): bool {
		$transient_key = 'paypercut_webhook_delivery_' . md5( $delivery_id );
		$processed     = get_transient( $transient_key );
		return false !== $processed;
	}

	/**
	 * Mark a webhook delivery as processed.
	 *
	 * @param string $delivery_id Delivery ID from Paypercut-Delivery-Id header.
	 * @return void
	 */
	private function mark_delivery_as_processed( string $delivery_id ): void {
		$transient_key = 'paypercut_webhook_delivery_' . md5( $delivery_id );
		set_transient( $transient_key, true, 7 * DAY_IN_SECONDS );
	}

	/**
	 * Update WooCommerce order based on verified checkout status from Paypercut API.
	 *
	 * @param WC_Order              $order          Order instance.
	 * @param array<string,mixed>   $checkout       Verified checkout data from Paypercut API.
	 * @param string                $payment_status Verified payment status from Paypercut API.
	 *
	 * @return void
	 */
	private function update_order_from_checkout( WC_Order $order, array $checkout, string $payment_status ): void {
		$updater = new OrderStatusUpdater();

		if ( 'paid' === $payment_status ) {
			$updater->mark_order_as_processing( $order, $checkout, 'Webhook' );
			return;
		}

		if ( 'unpaid' === $payment_status ) {
			$updater->mark_order_as_failed( $order, $payment_status, 'Webhook' );
			return;
		}

		$updater->mark_order_as_failed( $order, $payment_status, 'Webhook' );
	}

	/**
	 * Get verified checkout session from Paypercut API.
	 * Always verifies the session status with Paypercut to ensure accuracy.
	 *
	 * @param string $session_id Checkout session ID from webhook payload.
	 *
	 * @return array<string,mixed>|null Verified checkout session data, or null if verification fails.
	 */
	private function get_verified_checkout_session( string $session_id ): ?array {
		$gateway = $this->get_gateway_instance();
		if ( null === $gateway ) {
			Logger::error(
				'Webhook: gateway instance not available',
				array( 'session_id' => $session_id )
			);
			return null;
		}

		$api_secret = $gateway->get_api_client_secret();
		if ( '' === $api_secret ) {
			Logger::error(
				'Webhook: API secret not configured',
				array( 'session_id' => $session_id )
			);
			return null;
		}

		try {
			$api = new PaypercutApi( $api_secret );
			$verified_checkout = $api->get_checkout_session( $session_id );

			Logger::info(
				'Webhook: checkout session verified with Paypercut API',
				array( 'session_id' => $session_id )
			);

			return $verified_checkout;
		} catch ( Throwable $exception ) {
			Logger::error(
				'Webhook: failed to verify checkout session',
				array(
					'session_id' => $session_id,
					'exception'  => get_class( $exception ),
				)
			);

			return null;
		}
	}
}
