<?php
/**
 * REST endpoint for handling Paypercut BNPL webhooks.
 *
 * @package Paypercut\Payments\Http\Rest
 */

declare(strict_types=1);

namespace Paypercut\Payments\Http\Rest;

use Paypercut\Payments\Api\PaypercutApi;
use Paypercut\Payments\Gateway\PaypercutBnplGateway;
use Paypercut\Payments\Services\OrderStatusUpdater;
use Paypercut\Payments\Support\Logger;
use Throwable;
use WC_Order;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;
use WP_Error;
use function wc_get_order;

/**
 * Handles BNPL webhook events.
 *
 * Updates WooCommerce order status based on BNPL order updates.
 */
final class BnplWebhook extends AbstractWebhook {

	/**
	 * Get the webhook event type this handler processes.
	 *
	 * @return string
	 */
	protected function get_webhook_event_type(): string {
		return 'bnpl_order_update';
	}

	/**
	 * Get the REST API route path for this webhook.
	 *
	 * @return string
	 */
	protected function get_route_path(): string {
		return '/webhook/bnpl';
	}

	/**
	 * Register REST API routes for this webhook.
	 *
	 * Overrides parent to use a more permissive permission callback.
	 * Webhooks from Paypercut servers don't require WordPress authentication.
	 * Validation is performed in the handle() method.
	 *
	 * @return void
	 */
	public function register_routes(): void {
		register_rest_route(
			'paypercut/v1',
			$this->get_route_path(),
			array(
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => array( $this, 'handle' ),
				'permission_callback' => array( $this, 'permission_check' ),
			)
		);
	}

	/**
	 * Permission check for webhook endpoint.
	 *
	 * This is intentionally permissive - webhooks come from external servers
	 * and don't require WordPress authentication. Actual validation happens
	 * in the handle() method.
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return bool Always returns true to allow webhook requests through.
	 */
	public function permission_check( WP_REST_Request $request ): bool {
		$body = $request->get_body();
		return ! empty( $body );
	}

	/**
	 * Handle incoming webhook request.
	 *
	 * @param WP_REST_Request $request Incoming REST request.
	 *
	 * @return WP_REST_Response|WP_Error
	 */
	public function handle( WP_REST_Request $request ) {
		$raw_body = $request->get_body();
		Logger::info(
			'BNPL Webhook: Received webhook request',
			array(
				'raw_body' => $raw_body,
			)
		);

		$payload = $request->get_json_params();
		
		// If get_json_params() returns null or empty (e.g., Content-Type header not set), try parsing manually
		if ( ( null === $payload || empty( $payload ) ) && ! empty( $raw_body ) ) {
			$decoded = json_decode( $raw_body, true );
			if ( json_last_error() === JSON_ERROR_NONE && is_array( $decoded ) ) {
				$payload = $decoded;
			} else {
				Logger::error(
					'BNPL Webhook: JSON decode error',
					array(
						'error' => json_last_error_msg(),
						'json_error_code' => json_last_error(),
						'raw_body' => $raw_body,
					)
				);
			}
		}

		Logger::info(
			'BNPL Webhook: Parsed payload',
			array(
				'payload' => $payload,
			)
		);

		if ( empty( $payload ) || ! is_array( $payload ) ) {
			Logger::error( 'BNPL Webhook: invalid JSON payload' );
			return $this->create_error_response( 'Invalid webhook payload.', 400 );
		}

		// Extract attempt data from payload
		$attempt = $payload['attempt'] ?? array();
		if ( empty( $attempt ) || ! is_array( $attempt ) ) {
			Logger::error( 'BNPL Webhook: missing attempt data in payload' );
			return $this->create_error_response( 'Missing attempt data in payload.', 400 );
		}

		// Extract order ID from attempt (use merchant_purchase_ref as it contains the order ID)
		$merchant_purchase_ref = $attempt['merchant_purchase_ref'] ?? '';
		if ( empty( $merchant_purchase_ref ) ) {
			Logger::error( 'BNPL Webhook: missing merchant_purchase_ref in attempt data' );
			return $this->skip_webhook_response( 'no_merchant_purchase_ref' );
		}
		
		$order_id = (int) $merchant_purchase_ref;
		if ( $order_id <= 0 ) {
			Logger::error(
				'BNPL Webhook: invalid order ID from merchant_purchase_ref',
				array( 'merchant_purchase_ref' => $merchant_purchase_ref )
			);
			return $this->skip_webhook_response( 'invalid_order_id' );
		}
		
		$order = wc_get_order( $order_id );
		if ( ! $order instanceof WC_Order ) {
			Logger::error(
				'BNPL Webhook: order not found',
				array( 'order_id' => $order_id, 'merchant_purchase_ref' => $merchant_purchase_ref )
			);
			return $this->skip_webhook_response( 'order_not_found' );
		}

		$attempt_status = $attempt['status'] ?? '';
		if ( empty( $attempt_status ) ) {
			Logger::error(
				'BNPL Webhook: no status found in attempt data',
				array( 'order_id' => $order->get_id() )
			);
			return $this->skip_webhook_response( 'no_attempt_status' );
		}

		$this->update_order_from_payload( $order, $attempt, $attempt_status );

		return rest_ensure_response(
			new WP_REST_Response(
				array(
					'received'       => true,
					'order_id'       => $order->get_id(),
					'attempt_status' => $attempt_status,
				),
				200
			)
		);
	}

	/**
	 * Update WooCommerce order based on BNPL webhook payload.
	 *
	 * @param WC_Order              $order          Order instance.
	 * @param array<string,mixed>   $attempt        Attempt data from webhook payload.
	 * @param string                $attempt_status Attempt status from payload.
	 *
	 * @return void
	 */
	private function update_order_from_payload( WC_Order $order, array $attempt, string $attempt_status ): void {
		$webhook_attempt_id = $attempt['attempt_id'] ?? '';
		$stored_attempt_id = $order->get_meta( PaypercutBnplGateway::META_BNPL_ATTEMPT_ID, true );
		
		if ( empty( $webhook_attempt_id ) || $webhook_attempt_id !== $stored_attempt_id ) {
			Logger::error(
				'BNPL Webhook: attempt_id mismatch - webhook attempt_id does not match order stored attempt_id',
				array(
					'order_id' => $order->get_id(),
					'webhook_attempt_id' => $webhook_attempt_id,
					'stored_attempt_id' => $stored_attempt_id,
				)
			);
			return;
		}
		
		$webhook_merchant_purchase_ref = $attempt['merchant_purchase_ref'] ?? '';
		$order_id = (string) $order->get_id();
		
		if ( empty( $webhook_merchant_purchase_ref ) || $webhook_merchant_purchase_ref !== $order_id ) {
			Logger::error(
				'BNPL Webhook: merchant_purchase_ref mismatch - webhook merchant_purchase_ref does not match order ID',
				array(
					'order_id' => $order->get_id(),
					'webhook_merchant_purchase_ref' => $webhook_merchant_purchase_ref,
				)
			);
			return;
		}
		
		$updater = new OrderStatusUpdater();

		if ( 'ATTEMPT_STATUS_CAPTURED' === $attempt_status ) {
			$checkout_data = $attempt;
			if ( ! empty( $attempt['attempt_id'] ) ) {
				$checkout_data['id'] = $attempt['attempt_id'];
			}

			$updater->mark_order_as_processing( $order, $checkout_data, 'BNPL Webhook' );
			$order->add_order_note(
				sprintf(
					/* translators: %1$s: BNPL attempt ID */
					__( 'Paypercut BNPL attempt %1$s captured.', 'paypercut-payments-for-woocommerce' ),
					esc_html( $attempt['attempt_id'] ?? '' )
				)
			);
			$order->save();
			return;
		}

		if ( 'ATTEMPT_STATUS_ERRORED' === $attempt_status ) {
			$order->update_status( 'failed', __( 'Paypercut BNPL attempt errored.', 'paypercut-payments-for-woocommerce' ) );
			$order->add_order_note(
				sprintf(
					/* translators: %1$s: BNPL attempt ID, %2$s: Error reason */
					__( 'Paypercut BNPL attempt %1$s errored. Reason: %2$s', 'paypercut-payments-for-woocommerce' ),
					esc_html( $attempt['attempt_id'] ?? '' ),
					esc_html( $attempt['status_reason'] ?? '' )
				)
			);
			$order->save();
			Logger::info(
				'BNPL Webhook: Order marked as failed',
				array(
					'order_id'      => $order->get_id(),
					'attempt_id'    => $attempt['attempt_id'] ?? '',
					'attempt_status' => $attempt_status,
				)
			);
			return;
		}

		if ( 'ATTEMPT_STATUS_DECLINED' === $attempt_status ) {
			$order->update_status( 'cancelled', __( 'Paypercut BNPL attempt declined.', 'paypercut-payments-for-woocommerce' ) );
			$order->add_order_note(
				sprintf(
					/* translators: %1$s: BNPL attempt ID, %2$s: Decline reason */
					__( 'Paypercut BNPL attempt %1$s declined. Reason: %2$s', 'paypercut-payments-for-woocommerce' ),
					esc_html( $attempt['attempt_id'] ?? '' ),
					esc_html( $attempt['status_reason'] ?? '' )
				)
			);
			$order->save();
			Logger::info(
				'BNPL Webhook: Order marked as cancelled',
				array(
					'order_id'      => $order->get_id(),
					'attempt_id'    => $attempt['attempt_id'] ?? '',
					'attempt_status' => $attempt_status,
				)
			);
			return;
		}

		Logger::info(
			'BNPL Webhook: unhandled attempt status',
			array(
				'order_id'       => $order->get_id(),
				'attempt_status' => $attempt_status,
				'attempt_id'     => $attempt['attempt_id'] ?? '',
			)
		);
	}


	/**
	 * Get an instance of the BNPL gateway to access API credentials.
	 *
	 * @return PaypercutBnplGateway|null Gateway instance or null if not available.
	 */
	private function get_bnpl_gateway_instance(): ?PaypercutBnplGateway {
		$gateways = WC()->payment_gateways()->get_available_payment_gateways();
		$gateway  = $gateways[ PaypercutBnplGateway::ID ] ?? null;

		return $gateway instanceof PaypercutBnplGateway ? $gateway : null;
	}

	/**
	 * Verify webhook request from Paypercut BNPL.
	 *
	 * Overrides parent method to use BNPL gateway instead of main gateway.
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return bool True if request structure is valid, false otherwise.
	 */
	public function verify_webhook_signature( WP_REST_Request $request ): bool {
		$gateway = $this->get_bnpl_gateway_instance();
		if ( ! $gateway ) {
			Logger::error( 'BNPL Webhook: gateway instance not available' );
			return false;
		}

		$body = $request->get_body();
		if ( empty( $body ) ) {
			Logger::error( 'BNPL Webhook: empty request body' );
			return false;
		}

		$payload = $request->get_json_params();
		if ( empty( $payload ) || ! is_array( $payload ) ) {
			Logger::error( 'BNPL Webhook: invalid JSON payload' );
			return false;
		}

		return true;
	}
}

