<?php
/**
 * Base class for Paypercut webhook handlers.
 *
 * @package Paypercut\Payments\Http\Rest
 */

declare(strict_types=1);

namespace Paypercut\Payments\Http\Rest;

use Paypercut\Payments\Gateway\PaypercutGateway;
use Paypercut\Payments\Support\Logger;
use WP_Error;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;
use function rest_ensure_response;

/**
 * Abstract base class for webhook handlers.
 *
 * Provides common functionality for verifying webhooks, accessing gateway,
 * and handling responses.
 */
abstract class AbstractWebhook {

	/**
	 * Get the webhook event type this handler processes.
	 *
	 * @return string
	 */
	abstract protected function get_webhook_event_type(): string;

	/**
	 * Get the REST API route path for this webhook.
	 *
	 * @return string
	 */
	abstract protected function get_route_path(): string;

	/**
	 * Handle the incoming webhook request.
	 *
	 * @param WP_REST_Request $request Incoming REST request.
	 *
	 * @return WP_REST_Response|WP_Error
	 */
	abstract public function handle( WP_REST_Request $request );

	/**
	 * Verify webhook request from Paypercut.
	 *
	 * This method performs signature verification of webhook requests. Security is ensured through:
	 *
	 * 1. Signature verification (this method) - verifies request signature using HMAC
	 * 2. API verification (in handle() method) - verifies the session with Paypercut's API
	 *    before processing any order updates
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return bool|WP_Error True if signature is valid, WP_Error otherwise.
	 */
	abstract public function verify_webhook_signature( WP_REST_Request $request );

	/**
	 * Validate webhook payload structure.
	 *
	 * @param array<string,mixed> $payload Webhook payload.
	 *
	 * @return bool
	 */
	protected function validate_payload( array $payload ): bool {
		if ( empty( $payload ) || ! is_array( $payload ) ) {
			Logger::error( 'Webhook: empty or invalid payload received' );
			return false;
		}

		$webhook_type = $payload['type'] ?? '';
		if ( $webhook_type !== $this->get_webhook_event_type() ) {
			return false;
		}

		return true;
	}

	/**
	 * Create a response for a webhook that will be skipped (acknowledged but not processed).
	 *
	 * @param string $reason Reason for skipping the webhook.
	 * @param array  $extra  Additional data to include in the response.
	 *
	 * @return WP_REST_Response
	 */
	protected function skip_webhook_response( string $reason, array $extra = array() ): WP_REST_Response {
		return rest_ensure_response(
			new WP_REST_Response(
				array_merge(
					array(
						'received' => true,
						'ignored'  => true,
						'reason'   => $reason,
					),
					$extra
				),
				200
			)
		);
	}

	/**
	 * Create an error response for the webhook.
	 *
	 * @param string $message Error message.
	 * @param int    $status  HTTP status code.
	 *
	 * @return WP_Error
	 */
	protected function create_error_response( string $message, int $status = 400 ): WP_Error {
		return rest_ensure_response( new WP_Error( 'paypercut_webhook_error', $message, array( 'status' => $status ) ) );
	}

	/**
	 * Get an instance of the Paypercut gateway to access API credentials.
	 *
	 * @return PaypercutGateway|null Gateway instance or null if not available.
	 */
	protected function get_gateway_instance(): ?PaypercutGateway {
		$gateways = WC()->payment_gateways()->get_available_payment_gateways();
		$gateway  = $gateways[ PaypercutGateway::ID ] ?? null;

		return $gateway instanceof PaypercutGateway ? $gateway : null;
	}

	/**
	 * Extract data object from a flexible payload structure.
	 *
	 * @param array<string,mixed> $payload Raw webhook payload.
	 *
	 * @return array<string,mixed>
	 */
	protected function extract_data_from_payload( array $payload ): array {
		return $payload['data']['object'] ?? $payload['data'] ?? $payload;
	}
}
