<?php
/**
 * AJAX handler for testing API connection.
 *
 * @package Paypercut\Payments\Http\Ajax
 */

declare(strict_types=1);

namespace Paypercut\Payments\Http\Ajax;

use Paypercut\Payments\Api\PaypercutApi;
use Paypercut\Payments\Gateway\PaypercutBnplGateway;
use Paypercut\Payments\Support\Logger;
use Throwable;
use function current_user_can;
use function check_ajax_referer;
use function sanitize_key;
use function sanitize_text_field;
use function wp_send_json_error;
use function wp_send_json_success;
use function wp_unslash;

final class TestConnection {

	/**
	 * Handle test connection request.
	 */
	public function handle(): void {
		check_ajax_referer( 'paypercut_test_connection', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error(
				array( 'message' => __( 'Insufficient permissions.', 'paypercut-payments-for-woocommerce' ) ),
				403
			);
		}

		$api_secret = isset( $_POST['api_client_secret'] )
			? sanitize_text_field( wp_unslash( (string) $_POST['api_client_secret'] ) )
			: '';

		if ( '' === $api_secret ) {
			wp_send_json_error(
				array( 'message' => __( 'API key secret is required.', 'paypercut-payments-for-woocommerce' ) ),
				400
			);
		}

		// Determine which gateway is being tested
		$gateway_section = isset( $_POST['gateway_section'] )
			? sanitize_key( wp_unslash( (string) $_POST['gateway_section'] ) )
			: '';

		$is_bnpl = PaypercutBnplGateway::ID === $gateway_section;

		// Get mode for BNPL (default to test)
		$mode = 'test';
		if ( $is_bnpl && isset( $_POST['mode'] ) ) {
			$mode = sanitize_key( wp_unslash( (string) $_POST['mode'] ) );
		}

		try {
			if ( $is_bnpl ) {
				// Use BNPL API for validation with correct mode
				$base_uri = 'live' === $mode
					? PaypercutApi::BNPL_LIVE_BASE_URI
					: PaypercutApi::BNPL_TEST_BASE_URI;
				$api = new PaypercutApi( $api_secret, null, $base_uri );

				$api->validate_bnpl_credentials();

				Logger::info( 'Test connection: BNPL API credential validation successful' );
				wp_send_json_success(
					array(
						'message' => __( 'Connection successful! Your BNPL API key secret is valid.', 'paypercut-payments-for-woocommerce' ),
					)
				);
			} else {
				// Use standard API for validation
				$api = new PaypercutApi( $api_secret );
				$api->validate_credentials();
				Logger::info( 'Test connection: API credential validation successful' );
				
				wp_send_json_success(
					array(
						'message' => __( 'Connection successful! Your API key secret is valid.', 'paypercut-payments-for-woocommerce' ),
					)
				);
			}
		} catch ( Throwable $exception ) {
			Logger::error(
				'Test connection: API credential validation failed',
				array(
					'exception' => get_class( $exception ),
					'gateway'   => $is_bnpl ? 'bnpl' : 'standard',
				)
			);
			
			wp_send_json_error(
				array(
					'message' => sprintf(
						/* translators: %s: Exception error message */
						__( 'Connection failed: %s', 'paypercut-payments-for-woocommerce' ),
						esc_html( $exception->getMessage() )
					),
				),
				400
			);
		}
	}
}
