<?php
/**
 * AJAX handler for setup wizard operations.
 *
 * @package Paypercut\Payments\Http\Ajax
 */

declare(strict_types=1);

namespace Paypercut\Payments\Http\Ajax;

use Paypercut\Payments\Api\PaypercutApi;
use Paypercut\Payments\Gateway\PaypercutGateway;
use Paypercut\Payments\Gateway\PaypercutBnplGateway;
use Paypercut\Payments\Services\WebhookPayloadFactory;
use Paypercut\Payments\Support\Logger;
use Throwable;
use function add_query_arg;
use function admin_url;
use function current_user_can;
use function check_ajax_referer;
use function get_option;
use function sanitize_key;
use function sanitize_text_field;
use function update_option;
use function wp_send_json_error;
use function wp_send_json_success;
use function wp_unslash;

final class SetupWizard {

	/**
	 * Handle setup wizard form submission.
	 */
	public function handle(): void {
		check_ajax_referer( 'paypercut_setup_wizard', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error(
				array( 'message' => __( 'Insufficient permissions.', 'paypercut-payments-for-woocommerce' ) ),
				403
			);
		}

		$api_secret = isset( $_POST['api_client_secret'] )
			? sanitize_text_field( wp_unslash( (string) $_POST['api_client_secret'] ) )
			: '';

		if ( '' === $api_secret ) {
			wp_send_json_error(
				array( 'message' => __( 'API key secret is required.', 'paypercut-payments-for-woocommerce' ) ),
				400
			);
		}

		$section = isset( $_POST['gateway_section'] )
			? sanitize_key( wp_unslash( (string) $_POST['gateway_section'] ) )
			: PaypercutGateway::ID;

		$is_bnpl = PaypercutBnplGateway::ID === $section;

		$mode = 'test';
		if ( $is_bnpl && isset( $_POST['mode'] ) ) {
			$mode = sanitize_key( wp_unslash( (string) $_POST['mode'] ) );
		}

		try {
			if ( $is_bnpl ) {
				$base_uri = 'live' === $mode
					? PaypercutApi::BNPL_LIVE_BASE_URI
					: PaypercutApi::BNPL_TEST_BASE_URI;
				$api = new PaypercutApi( $api_secret, null, $base_uri );
				$api->validate_bnpl_credentials();
				Logger::info( 'Setup wizard: BNPL API credential validation successful' );
			} else {
				$api = new PaypercutApi( $api_secret );
				$api->validate_credentials();
				Logger::info( 'Setup wizard: API credential validation successful' );
			}
		} catch ( Throwable $exception ) {
			Logger::error(
				'Setup wizard: API credential validation failed',
				array(
					'exception' => get_class( $exception ),
				)
			);
			wp_send_json_error(
				array(
					'message' => sprintf(
						/* translators: %s: Exception error message */
						__( 'Invalid API key secret: %s', 'paypercut-payments-for-woocommerce' ),
						esc_html( $exception->getMessage() )
					),
				),
				400
			);
		}

		$webhook_secret = '';
		$webhook_error  = null;

		if ( ! $is_bnpl ) {
			$webhook_factory = new WebhookPayloadFactory();
			$webhook_url     = $webhook_factory->get_webhook_url();
			$webhook_payload = $webhook_factory->build( $webhook_url, array( 'checkout_session.completed' ), 'Checkout Session' );

			try {
				$webhook_response = $api->create_webhook( $webhook_payload );
				$webhook_secret   = isset( $webhook_response['secret'] ) ? (string) $webhook_response['secret'] : '';
				Logger::info( 'Setup wizard: webhook created successfully' );
			} catch ( Throwable $exception ) {
				$error_message = $exception->getMessage();
				
				if ( strpos( $error_message, 'already_exists' ) !== false || strpos( $error_message, 'already exists' ) !== false ) {
					$webhook_error = __( 'A webhook for this URL already exists. You can manage existing webhooks in the Webhooks section below.', 'paypercut-payments-for-woocommerce' );
				} else {
					$webhook_error = sprintf(
						__( 'Webhook creation failed: %s', 'paypercut-payments-for-woocommerce' ),
						esc_html( $error_message )
					);
				}

				Logger::error(
					'Setup wizard: webhook creation failed',
					array(
						'exception' => get_class( $exception ),
						'message'   => $error_message,
					)
				);
			}
		}

		$gateway_id = 'woocommerce_' . $section . '_settings';
		$settings   = get_option( $gateway_id, array() );
		$settings['api_client_secret'] = $api_secret;
		$settings['enabled']            = 'yes';
		
		if ( ! empty( $webhook_secret ) ) {
			$settings['webhook_secret'] = $webhook_secret;
		}

		if ( $is_bnpl ) {
			$settings['mode'] = $mode;
		}

		update_option( $gateway_id, $settings );

		Logger::info(
			'Setup wizard completed successfully',
			array( 'gateway' => $section )
		);

		$redirect_url = admin_url( 'admin.php?page=wc-settings&tab=checkout&section=' . $section );
		$redirect_url = add_query_arg( 'paypercut_setup_complete', '1', $redirect_url );
		$redirect_url = add_query_arg( '_wpnonce', wp_create_nonce( 'paypercut_setup_complete' ), $redirect_url );

		$response_data = array(
			'message'    => __( 'Setup complete! Gateway enabled and configured.', 'paypercut-payments-for-woocommerce' ),
			'reload_url' => $redirect_url,
		);

		if ( null !== $webhook_error ) {
			$response_data['webhook_warning'] = $webhook_error;
		}

		wp_send_json_success( $response_data );
	}
}
