<?php
/**
 * AJAX handler for deleting webhooks.
 *
 * @package Paypercut\Payments\Http\Ajax
 */

declare(strict_types=1);

namespace Paypercut\Payments\Http\Ajax;

use Paypercut\Payments\Api\PaypercutApi;
use Paypercut\Payments\Gateway\PaypercutGateway;
use Paypercut\Payments\Support\Logger;
use Throwable;
use function check_ajax_referer;
use function current_user_can;
use function get_option;
use function sanitize_text_field;
use function update_option;
use function wp_send_json_error;
use function wp_send_json_success;
use function wp_unslash;

final class DeleteWebhook {

	/**
	 * Handle delete webhook request.
	 */
	public function handle(): void {
		check_ajax_referer( 'paypercut_delete_webhook', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error(
				array( 'message' => __( 'Insufficient permissions.', 'paypercut-payments-for-woocommerce' ) ),
				403
			);
		}

		$webhook_id = isset( $_POST['webhook_id'] )
			? sanitize_text_field( wp_unslash( (string) $_POST['webhook_id'] ) )
			: '';

		if ( '' === $webhook_id ) {
			wp_send_json_error(
				array( 'message' => __( 'Webhook ID is required.', 'paypercut-payments-for-woocommerce' ) ),
				400
			);
		}

		$gateway_id = 'woocommerce_' . PaypercutGateway::ID . '_settings';
		$settings   = get_option( $gateway_id, array() );
		$api_secret = isset( $settings['api_client_secret'] ) ? (string) $settings['api_client_secret'] : '';

		if ( '' === $api_secret ) {
			wp_send_json_error(
				array( 'message' => __( 'API credentials not configured.', 'paypercut-payments-for-woocommerce' ) ),
				400
			);
		}

		try {
			$api = new PaypercutApi( $api_secret );
			$api->delete_webhook( $webhook_id );
			Logger::info(
				'Webhook deleted successfully',
				array( 'webhook_id' => $webhook_id )
			);

			wp_send_json_success(
				array(
					'message' => __( 'Webhook deleted successfully.', 'paypercut-payments-for-woocommerce' ),
				)
			);
		} catch ( Throwable $exception ) {
			Logger::error(
				'Failed to delete webhook',
				array(
					'webhook_id' => $webhook_id,
					'exception'  => get_class( $exception ),
					'message'    => $exception->getMessage(),
				)
			);

			wp_send_json_error(
				array(
					'message' => sprintf(
						/* translators: %s: Exception error message */
						__( 'Failed to delete webhook: %s', 'paypercut-payments-for-woocommerce' ),
						esc_html( $exception->getMessage() )
					),
				),
				400
			);
		}
	}
}
