<?php
/**
 * AJAX handler responsible for creating checkout sessions.
 *
 * @package Paypercut\Payments\Http\Ajax
 */

declare(strict_types=1);

namespace Paypercut\Payments\Http\Ajax;

use Paypercut\Payments\Gateway\PaypercutGateway;
use Paypercut\Payments\Services\CheckoutPayloadFactory;
use Paypercut\Payments\Services\PaypercutCheckoutClient;
use Throwable;
use WC_Cart;
use WC_Order;
use function __;
use function absint;
use function check_ajax_referer;
use function sanitize_text_field;
use function wc_get_order;
use function wp_generate_uuid4;
use function wp_send_json_error;
use function wp_send_json_success;
use function wp_unslash;

final class CreateCheckoutSession {
	private CheckoutPayloadFactory $factory;

	public function __construct( CheckoutPayloadFactory $factory ) {
		$this->factory = $factory;
	}

	public function handle(): void {
		check_ajax_referer( 'paypercut_checkout_session', 'nonce' );

		$requested_context = isset( $_POST['context'] )
			? sanitize_text_field( wp_unslash( (string) $_POST['context'] ) )
			: 'checkout';

		$gateway = null;
		$payload = null;

		if ( 'setup' === $requested_context ) {
			$gateway = $this->resolve_gateway();

			if ( null === $gateway ) {
				wp_send_json_error( array( 'message' => __( 'Paypercut gateway is not available.', 'paypercut-payments-for-woocommerce' ) ), 400 );
			}

			$payload = $this->factory->build_setup( $gateway );
		} else {
			$order = $this->resolve_order_pay_context();

			if ( null !== $order ) {
				$gateway = $this->resolve_gateway();

				if ( null === $gateway ) {
					wp_send_json_error( array( 'message' => __( 'Paypercut gateway is not available.', 'paypercut-payments-for-woocommerce' ) ), 400 );
				}

				$payload = $this->factory->build_from_order( $order, $gateway );
			} else {
				$context = $this->resolve_context();

				if ( null === $context ) {
					wp_send_json_error( array( 'message' => __( 'Paypercut gateway is not available.', 'paypercut-payments-for-woocommerce' ) ), 400 );
				}

				list( $cart, $gateway ) = $context;
				$payload = $this->factory->build( $cart, $gateway );
			}
		}

		try {
			$client = PaypercutCheckoutClient::forPayments( $gateway->get_api_client_secret() );

				$response = $client->create(
					$payload,
					array( 'idempotency_key' => wp_generate_uuid4() )
				);

			$session_id        = $this->extract_identifier( $response, array( 'id' ) );
			$payment_intent_id = $this->extract_identifier( $response, array( 'payment_intent' ) );

			if ( '' === $session_id ) {
				wp_send_json_error( array( 'message' => __( 'Paypercut did not return a session identifier.', 'paypercut-payments-for-woocommerce' ) ), 500 );
			}

			$this->store_session_value( 'paypercut_checkout_session_id', $session_id );
			$this->store_session_value( 'paypercut_checkout_payment_id', $session_id );
			$this->store_session_value( 'paypercut_checkout_payment_intent_id', $payment_intent_id );

			wp_send_json_success(
				array(
					'sessionId'       => $session_id,
					'paymentId'       => $session_id,
					'paymentIntentId' => $payment_intent_id,
				)
			);
		} catch ( Throwable $exception ) {
			wp_send_json_error( array( 'message' => __( 'Unable to create a Paypercut checkout session.', 'paypercut-payments-for-woocommerce' ) ), 500 );
		}
	}

	/**
	 * Resolve order from order pay page context.
	 *
	 * @return WC_Order|null
	 */
	private function resolve_order_pay_context() {
		$order_id = isset( $_POST['order_id'] )
			? absint( wp_unslash( (string) $_POST['order_id'] ) )
			: 0;

		if ( ! $order_id ) {
			return null;
		}

		$order = wc_get_order( $order_id );
		if ( ! $order ) {
			return null;
		}

		$order_key = isset( $_GET['key'] ) ? sanitize_text_field( wp_unslash( (string) $_GET['key'] ) ) : '';
		if ( $order_key && ! hash_equals( $order->get_order_key(), $order_key ) ) {
			return null;
		}

		if ( ! $order->needs_payment() ) {
			return null;
		}

		return $order;
	}

	/**
	 * @return array{0: WC_Cart, 1: PaypercutGateway}|null
	 */
	private function resolve_context(): ?array {
		if ( ! function_exists( 'WC' ) ) {
			return null;
		}

		$woocommerce = WC();

		if ( ! $woocommerce || ! $woocommerce->cart || $woocommerce->cart->is_empty() ) {
			return null;
		}

		$gateway = $this->resolve_gateway();

		if ( null === $gateway ) {
			return null;
		}

		return array( $woocommerce->cart, $gateway );
	}

	private function resolve_gateway(): ?PaypercutGateway {
		if ( ! function_exists( 'WC' ) ) {
			return null;
		}

		$woocommerce = WC();

		if ( ! $woocommerce ) {
			return null;
		}

		$manager = $woocommerce->payment_gateways();

		if ( ! $manager ) {
			return null;
		}

		$gateways = $manager->get_available_payment_gateways();

		if ( empty( $gateways[ PaypercutGateway::ID ] ) || ! $gateways[ PaypercutGateway::ID ] instanceof PaypercutGateway ) {
			return null;
		}

		return $gateways[ PaypercutGateway::ID ];
	}

	/**
	 * @param array<string, mixed> $response Response payload from the API.
	 * @param array<int, string>   $keys      Ordered list of keys to inspect.
	 */
	private function extract_identifier( array $response, array $keys ): string {
		foreach ( $keys as $key ) {
			if ( isset( $response[ $key ] ) && is_string( $response[ $key ] ) ) {
				return $response[ $key ];
			}
		}

		return '';
	}

	private function store_session_value( string $key, string $value ): void {
		if ( ! function_exists( 'WC' ) || ! WC()->session ) {
			return;
		}

		WC()->session->set( $key, $value ?: null );
	}
}
