<?php
/**
 * Paypercut Payments gateway definition.
 *
 * @package Paypercut\Payments\Gateway
 */

declare(strict_types=1);

namespace Paypercut\Payments\Gateway;

use Paypercut\Payments\Admin\SetupWizard;
use Paypercut\Payments\Api\PaypercutApi;
use Paypercut\Payments\Gateway\PaypercutPaymentToken;
use Paypercut\Payments\Services\CheckoutPayloadFactory;
use Paypercut\Payments\Services\OrderStatusUpdater;
use Paypercut\Payments\Services\RefundClient;
use Paypercut\Payments\Services\PaypercutCheckoutClient;
use Paypercut\Payments\Support\Assets;
use Paypercut\Payments\Support\Logger;
use Throwable;
use WC_Order;
use WC_Payment_Gateway;
use WC_Payment_Tokens;
use WP_Error;
use function esc_attr;
use function esc_url;
use function esc_html__;
use function esc_html;
use function sanitize_text_field;
use function is_string;
use function wc_add_notice;
use function wc_get_order;
use function wc_get_price_decimals;
use function wc_price;
use function wp_unslash;
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Main gateway implementation for Paypercut Payments.
 */
final class PaypercutGateway extends WC_Payment_Gateway {
	public const ID                           = 'paypercut_payments';
	public const META_SESSION_ID              = '_paypercut_session_id';
	public const META_PAYMENT_INTENT_ID       = '_paypercut_payment_intent_id';
	public const META_VERIFICATION_ATTEMPTED  = '_paypercut_verification_attempted';
	/**
	 * API secret.
	 *
	 * @var string
	 */
	private string $api_client_secret = '';

	/**
	 * Webhook secret for signature verification.
	 *
	 * @var string
	 */
	private string $webhook_secret = '';

	/**
	 * Selected checkout mode.
	 *
	 * @var string
	 */
	private string $checkout_mode = 'hosted';

	/**
	 * Whether checkout logo should be hidden.
	 *
	 * @var bool
	 */
	private bool $hide_logo = false;

	private string $appearance_preset = 'inline';

	private string $appearance_theme = 'light';

	private string $appearance_labels = 'above';

	private bool $saved_payment_methods_enabled = true;

	private const APPEARANCE_PRESET_OPTIONS = array( 'inline', 'modal' );

	private const APPEARANCE_THEME_OPTIONS = array( 'light', 'dark' );

	private const APPEARANCE_LABEL_OPTIONS = array( 'above', 'inline', 'floating' );

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->id                 = self::ID;
		$this->icon               = Assets::url( 'images/icon-128x128.png' );
		$this->method_title       = __( 'Paypercut Payments', 'paypercut-payments-for-woocommerce' );
		$this->method_description = __( 'Accept Paypercut one-off and tokenised payments.', 'paypercut-payments-for-woocommerce' );
		$this->supports           = array(
			'products',
			'refunds',
			'subscriptions',
			'subscription_cancellation',
			'subscription_suspension',
			'subscription_reactivation',
			'subscription_date_changes',
			'subscription_amount_changes',
			'tokenization',
			'add_payment_method',
		);

		add_filter( 'woocommerce_payment_methods_list_item', array( $this, 'format_payment_method_display' ), 10, 2 );

		$this->init_form_fields();
		$this->init_settings();

		$this->enabled         = $this->get_option( 'enabled', 'no' );
		$this->title           = $this->get_option( 'title', __( 'Pay with Paypercut', 'paypercut-payments-for-woocommerce' ) );
		$this->description     = $this->get_option( 'description', '' );
		$this->api_client_secret        = $this->get_option( 'api_client_secret', '' );
		$this->webhook_secret  = $this->get_option( 'webhook_secret', '' );
		$this->checkout_mode     = $this->get_option( 'checkout_mode', 'hosted' );
		$this->hide_logo         = 'yes' === $this->get_option( 'hide_logo', 'no' );
		$this->appearance_preset = (string) $this->get_option( 'appearance_preset', 'inline' );
		if ( ! in_array( $this->appearance_preset, self::APPEARANCE_PRESET_OPTIONS, true ) ) {
			$this->appearance_preset = 'inline';
		}

		$this->appearance_theme = (string) $this->get_option( 'appearance_theme', 'light' );
		if ( ! in_array( $this->appearance_theme, self::APPEARANCE_THEME_OPTIONS, true ) ) {
			$this->appearance_theme = 'light';
		}

		$this->appearance_labels = (string) $this->get_option( 'appearance_labels', 'above' );
		if ( ! in_array( $this->appearance_labels, self::APPEARANCE_LABEL_OPTIONS, true ) ) {
			$this->appearance_labels = 'above';
		}

		$this->saved_payment_methods_enabled = 'yes' === $this->get_option( 'saved_payment_methods', 'yes' );

		$force_embedded_for_add_payment = function_exists( 'is_add_payment_method_page' ) && is_add_payment_method_page();
		$this->has_fields                = 'embedded' === $this->checkout_mode || $force_embedded_for_add_payment;

		add_action(
			'woocommerce_update_options_payment_gateways_' . $this->id,
			function (): void {
				$this->process_admin_options();
			}
		);
	}

	/**
	 * Initialise gateway settings fields.
	 */
	public function init_form_fields(): void {
		$this->form_fields = array(
			'enabled'           => array(
				'title'   => __( 'Enable/Disable', 'paypercut-payments-for-woocommerce' ),
				'label'   => __( 'Enable Paypercut Payments', 'paypercut-payments-for-woocommerce' ),
				'type'    => 'checkbox',
				'default' => 'no',
			),
			'title'             => array(
				'title'       => __( 'Title', 'paypercut-payments-for-woocommerce' ),
				'type'        => 'text',
				'default'     => __( 'Pay with Paypercut', 'paypercut-payments-for-woocommerce' ),
				'description' => __( 'Displayed to customers during checkout.', 'paypercut-payments-for-woocommerce' ),
			),
			'description'       => array(
				'title'       => __( 'Description', 'paypercut-payments-for-woocommerce' ),
				'type'        => 'textarea',
				'default'     => '',
				'description' => __( 'Payment method description that the customer will see on your checkout.', 'paypercut-payments-for-woocommerce' ),
			),
			'api_client_secret' => array(
				'title'       => __( 'API Key Secret', 'paypercut-payments-for-woocommerce' ),
				'type'        => 'api_client_secret',
				'description' => __( 'Bearer token for Paypercut API requests.', 'paypercut-payments-for-woocommerce' ),
			),
			'logging'           => array(
				'title'       => __( 'Logging', 'paypercut-payments-for-woocommerce' ),
				'label'       => __( 'Log debug messages', 'paypercut-payments-for-woocommerce' ),
				'type'        => 'checkbox',
				'description' => __( 'Writes gateway events to WooCommerce Status → Logs.', 'paypercut-payments-for-woocommerce' ),
				'default'     => 'yes',
			),
			'checkout_mode'     => array(
				'title'   => __( 'Checkout Mode', 'paypercut-payments-for-woocommerce' ),
				'type'    => 'select',
				'options' => array(
					'hosted'    => __( 'Hosted redirect checkout', 'paypercut-payments-for-woocommerce' ),
					'embedded'  => __( 'Embedded (iframe/modal) checkout', 'paypercut-payments-for-woocommerce' ),
				),
				'default' => 'hosted',
			),
			'appearance_preset' => array(
				'title'       => __( 'Embedded experience', 'paypercut-payments-for-woocommerce' ),
				'type'        => 'select',
				'description' => __( 'Choose whether the embedded checkout appears inline or inside a modal.', 'paypercut-payments-for-woocommerce' ),
				'options'     => array(
					'inline' => __( 'Inline form', 'paypercut-payments-for-woocommerce' ),
					'modal'  => __( 'Modal overlay', 'paypercut-payments-for-woocommerce' ),
				),
				'default'     => 'inline',
			),
			'appearance_theme'  => array(
				'title'       => __( 'Form theme', 'paypercut-payments-for-woocommerce' ),
				'type'        => 'select',
				'description' => __( 'Match the embedded Paypercut form with your storefront theme.', 'paypercut-payments-for-woocommerce' ),
				'options'     => array(
					'light' => __( 'Light', 'paypercut-payments-for-woocommerce' ),
					'dark'  => __( 'Dark', 'paypercut-payments-for-woocommerce' ),
				),
				'default'     => 'light',
			),
			'appearance_labels' => array(
				'title'       => __( 'Field labels', 'paypercut-payments-for-woocommerce' ),
				'type'        => 'select',
				'description' => __( 'Control how Paypercut displays field labels inside the embedded form.', 'paypercut-payments-for-woocommerce' ),
				'options'     => array(
					'above'    => __( 'Above inputs', 'paypercut-payments-for-woocommerce' ),
					'inline'   => __( 'Inline (placeholder)', 'paypercut-payments-for-woocommerce' ),
					'floating' => __( 'Floating labels', 'paypercut-payments-for-woocommerce' ),
				),
				'default'     => 'above',
			),
			'saved_payment_methods' => array(
				'title'       => __( 'Saved payment methods', 'paypercut-payments-for-woocommerce' ),
				'label'       => __( 'Allow customers to save cards for future purchases', 'paypercut-payments-for-woocommerce' ),
				'type'        => 'checkbox',
				'description' => __( 'When enabled, Paypercut will display the option to store a payment method for reuse. Disable if you do not offer saved cards.', 'paypercut-payments-for-woocommerce' ),
				'default'     => 'yes',
			),
			'hide_logo'         => array(
				'title'   => __( 'Hide logo', 'paypercut-payments-for-woocommerce' ),
				'label'   => __( 'Disable Paypercut branding on checkout', 'paypercut-payments-for-woocommerce' ),
				'type'    => 'checkbox',
				'default' => 'no',
			),
		);
	}

	/**
	 * Render the gateway settings page with custom styling.
	 */
	public function admin_options(): void {
		if ( ! $this->is_setup_complete() ) {
			( new SetupWizard() )->render();
			return;
		}

		$banner_large = Assets::url( 'images/banner-1544x500.png' );
		$banner_small = Assets::url( 'images/banner-772x250.png' );
		$icon         = Assets::url( 'images/icon-128x128.png' );
		$sections     = array(
			'general'     => array(
				'title' => __( 'Display', 'paypercut-payments-for-woocommerce' ),
				'keys'  => array( 'enabled', 'title', 'description', 'hide_logo' ),
			),
			'credentials' => array(
				'title' => __( 'API Credentials', 'paypercut-payments-for-woocommerce' ),
				'keys'  => array( 'api_client_secret' ),
			),
			'checkout'    => array(
				'title' => __( 'Checkout Configuration', 'paypercut-payments-for-woocommerce' ),
				'keys'  => array(
					'checkout_mode',
					'saved_payment_methods',
					'logging',
				),
			),
			'webhooks'    => array(
				'title' => __( 'Webhooks', 'paypercut-payments-for-woocommerce' ),
			),
			'advanced'    => array(
				'title' => __( 'Advanced', 'paypercut-payments-for-woocommerce' ),
				'description' => __( 'These options control the appearance of the embedded experience. The defaults are recommended—change them only if Paypercut support asks you to.', 'paypercut-payments-for-woocommerce' ),
				'keys'  => array(
					'appearance_preset',
					'appearance_theme',
					'appearance_labels',
				),
			),
		);
		$form_fields  = $this->get_form_fields();

		?>
		<div class="paypercut-settings">
			<div class="paypercut-settings__hero">
				<picture>
					<source media="(max-width: 782px)" srcset="<?php echo esc_url( $banner_small ); ?>">
					<img src="<?php echo esc_url( $banner_large ); ?>" alt="<?php esc_attr_e( 'Paypercut banner', 'paypercut-payments-for-woocommerce' ); ?>">
				</picture>
			</div>
			<div class="paypercut-settings__header">
				<img src="<?php echo esc_url( $icon ); ?>" alt="<?php esc_attr_e( 'Paypercut logo', 'paypercut-payments-for-woocommerce' ); ?>">
				<div>
					<h2><?php echo esc_html( $this->method_title ); ?></h2>
					<p><?php echo esc_html__( 'Cut transaction friction with Paypercut’s embedded or hosted checkout experiences. Configure the basics below.', 'paypercut-payments-for-woocommerce' ); ?></p>
				</div>
			</div>
			<?php foreach ( $sections as $section_key => $section ) : ?>
				<div class="paypercut-settings__section">
					<h3 class="paypercut-settings__section-title"><?php echo esc_html( $section['title'] ); ?></h3>
					<?php if ( ! empty( $section['description'] ) ) : ?>
						<div class="notice notice-warning inline" style="margin:14px;">
							<p><?php echo esc_html( $section['description'] ); ?></p>
						</div>
					<?php endif; ?>
					<?php if ( 'webhooks' === $section_key ) : ?>
						<?php $this->render_webhooks_section(); ?>
					<?php else : ?>
						<table class="form-table paypercut-settings__table">
							<?php $this->render_section_fields( $section['keys'] ?? array(), $form_fields ); ?>
						</table>
					<?php endif; ?>
				</div>
			<?php endforeach; ?>
		</div>
		<?php
	}

	/**
	 * Add payment method via account screen.
	 *
	 * @return array{result: string, redirect: string}
	 */
	public function add_payment_method(): array {
		if ( ! is_user_logged_in() ) {
			wc_add_notice( __( 'Please log in to add a payment method.', 'paypercut-payments-for-woocommerce' ), 'error' );
			return array(
				'result'   => 'failure',
				'redirect' => wc_get_account_endpoint_url( 'payment-methods' ),
			);
		}

		$identifiers = $this->extract_transaction_identifiers_from_request();
		$session_id = $identifiers['session'];

		if ( empty( $session_id ) ) {
			Logger::error( 'Add payment method: Missing session ID' );
			wc_add_notice( __( 'Payment method setup failed. Please try again.', 'paypercut-payments-for-woocommerce' ), 'error' );
			return array(
				'result'   => 'failure',
				'redirect' => wc_get_account_endpoint_url( 'payment-methods' ),
			);
		}

		$api_secret = $this->get_api_client_secret();
		if ( empty( $api_secret ) ) {
			Logger::error( 'Add payment method: API secret not found' );
			wc_add_notice( __( 'Payment gateway configuration error. Please contact support.', 'paypercut-payments-for-woocommerce' ), 'error' );
			return array(
				'result'   => 'failure',
				'redirect' => wc_get_account_endpoint_url( 'payment-methods' ),
			);
		}

		try {
			$api = new PaypercutApi( $api_secret );
			$session = $api->get_checkout_session(
				$session_id,
				array( 'expand' => 'setup_intent.payment_method' )
			);

			Logger::info(
				'Add payment method: Retrieved session',
				array(
					'session_id'   => $session_id,
					'session_mode' => $session['mode'] ?? 'unknown',
					'status'       => $session['status'] ?? 'unknown',
				)
			);

			$session_mode = (string) ( $session['mode'] ?? '' );

			if ( 'setup' !== $session_mode ) {
				Logger::error(
					'Add payment method: Invalid session mode',
					array(
						'session_id'   => $session_id,
						'session_mode' => $session_mode,
					)
				);
				wc_add_notice( __( 'Invalid payment method setup session.', 'paypercut-payments-for-woocommerce' ), 'error' );
				return array(
					'result'   => 'failure',
					'redirect' => wc_get_account_endpoint_url( 'payment-methods' ),
				);
			}

			$setup_status = (string) ( $session['setup_intent']['status'] ?? '' );
			if ( 'succeeded' !== $setup_status ) {
				Logger::error(
					'Add payment method: Setup intent not completed',
					array(
						'session_id'   => $session_id,
						'setup_status' => $setup_status,
					)
				);
				wc_add_notice( __( 'Payment method setup was not completed. Please try again.', 'paypercut-payments-for-woocommerce' ), 'error' );
				return array(
					'result'   => 'failure',
					'redirect' => wc_get_account_endpoint_url( 'payment-methods' ),
				);
			}

			$token = $this->save_payment_method_from_session( $session, get_current_user_id() );

			if ( $token ) {
				Logger::info(
					'Payment method added successfully',
					array(
						'user_id'  => get_current_user_id(),
						'token_id' => $token->get_token(),
					)
				);
				return array(
					'result'   => 'success',
					'redirect' => wc_get_account_endpoint_url( 'payment-methods' ),
				);
			}

			Logger::error( 'Add payment method: Failed to save token' );
			wc_add_notice( __( 'Unable to add payment method. Please try again.', 'paypercut-payments-for-woocommerce' ), 'error' );
			return array(
				'result'   => 'failure',
				'redirect' => wc_get_account_endpoint_url( 'payment-methods' ),
			);

		} catch ( Throwable $exception ) {
			Logger::error(
				'Add payment method: Exception',
				array(
					'session_id' => $session_id,
					'error'      => $exception->getMessage(),
					'file'       => $exception->getFile(),
					'line'       => $exception->getLine(),
				)
			);
			wc_add_notice( __( 'Unable to add payment method. Please try again.', 'paypercut-payments-for-woocommerce' ), 'error' );
			return array(
				'result'   => 'failure',
				'redirect' => wc_get_account_endpoint_url( 'payment-methods' ),
			);
		}
	}

	/**
	 * Process payments.
	 *
	 * @param int $order_id Order identifier.
	 *
	 * @return array{result: string, redirect: string}
	 */
	public function process_payment( $order_id ): array {
		Logger::info(
			'Paypercut process_payment called',
			array(
				'order_id'     => $order_id,
				'checkout_mode' => $this->checkout_mode,
			)
		);

		$order = wc_get_order( $order_id );

		if ( ! $order instanceof WC_Order ) {
			Logger::error(
				sprintf(
					'Failed to initiate payment for order #%d: WooCommerce order not found.',
					$order_id
				),
				array(
					'order_id' => $order_id,
				)
			);
			return array(
				'result'   => 'failure',
				'redirect' => '',
			);
		}

		Logger::info(
			'Paypercut order found, processing payment',
			array(
				'order_id'   => $order->get_id(),
				'order_status' => $order->get_status(),
				'order_total' => $order->get_total(),
			)
		);

		if ( 'hosted' === $this->checkout_mode ) {
			return $this->process_hosted_checkout( $order );
		}

		return $this->process_embedded_checkout( $order );
	}

	private function process_embedded_checkout( WC_Order $order ): array {
		Logger::info(
			'Processing embedded Paypercut payment',
			array(
				'order_id' => $order->get_id(),
				'amount'   => $order->get_total(),
				'currency' => $order->get_currency(),
			)
		);

		$identifiers = $this->extract_transaction_identifiers_from_request();
		$session_id = $identifiers['session'];

		if ( empty( $session_id ) ) {
			Logger::error(
				'Embedded checkout: Missing session ID',
				array( 'order_id' => $order->get_id() )
			);
			wc_add_notice( __( 'Payment verification failed. Please contact support.', 'paypercut-payments-for-woocommerce' ), 'error' );
			return array(
				'result'   => 'failure',
				'redirect' => '',
			);
		}

		$api_secret = $this->get_api_client_secret();
		if ( empty( $api_secret ) ) {
			Logger::error(
				'Embedded checkout: API secret not found',
				array( 'order_id' => $order->get_id() )
			);
			wc_add_notice( __( 'Payment gateway configuration error. Please contact support.', 'paypercut-payments-for-woocommerce' ), 'error' );
			return array(
				'result'   => 'failure',
				'redirect' => '',
			);
		}

		$this->persist_transaction_identifiers(
			$order,
			$identifiers['session'],
			$identifiers['payment'],
			$identifiers['payment_intent']
		);

		$order->save();
		$this->clear_cached_checkout_identifiers();

		Logger::info(
			'Embedded checkout: Order created, payment verification will happen via webhook/thank you page',
			array(
				'order_id'   => $order->get_id(),
				'session_id' => $session_id,
			)
		);

		return array(
			'result'   => 'success',
			'redirect' => $this->get_return_url( $order ),
		);
	}

	private function process_hosted_checkout( WC_Order $order ): array {
		$factory = new CheckoutPayloadFactory();
		$client  = PaypercutCheckoutClient::forPayments( $this->get_api_client_secret() );

		try {
			$payload = $factory->build_from_order( $order, $this );

			Logger::info(
				'Paypercut hosted checkout payload',
				array(
					'order_id' => $order->get_id(),
				)
			);

			$response = $client->create(
				$payload,
				array(
					'idempotency_key' => $this->build_idempotency_key( $order ),
				)
			);

			$identifiers = $this->extract_transaction_identifiers_from_response( $response );
			$session_id = $identifiers['session'] ?? '';
			Logger::info(
				'Paypercut hosted checkout response',
				array(
					'order_id' => $order->get_id(),
					'session_id' => $session_id,
				)
			);
		} catch ( Throwable $exception ) {
			Logger::error(
				'Hosted checkout session failed',
				array(
					'order_id' => $order->get_id(),
				)
			);

			wc_add_notice( __( 'Unable to start Paypercut checkout. Please try again.', 'paypercut-payments-for-woocommerce' ), 'error' );

			return array(
				'result'   => 'failure',
				'redirect' => '',
			);
		}

		$redirect_url = $this->extract_hosted_redirect_url( $response );

		if ( '' === $redirect_url ) {
			$identifiers = $this->extract_transaction_identifiers_from_response( $response );
			$session_id = $identifiers['session'] ?? '';
			Logger::error(
				'Hosted checkout response missing redirect URL',
				array(
					'order_id' => $order->get_id(),
					'session_id' => $session_id,
				)
			);

			wc_add_notice( __( 'Unable to start Paypercut checkout. Please try again.', 'paypercut-payments-for-woocommerce' ), 'error' );

			return array(
				'result'   => 'failure',
				'redirect' => '',
			);
		}

		$identifiers = $this->extract_transaction_identifiers_from_response( $response );

		$this->persist_transaction_identifiers(
			$order,
			$identifiers['session'],
			$identifiers['payment'],
			$identifiers['payment_intent']
		);

		$order->update_status(
			'pending',
			__( 'Awaiting completion of Paypercut hosted checkout.', 'paypercut-payments-for-woocommerce' )
		);

		$order->save();

		Logger::info(
			'Paypercut hosted checkout order saved, redirecting',
			array(
				'order_id'     => $order->get_id(),
				'order_status' => $order->get_status(),
				'redirect_url' => $redirect_url,
			)
		);

		return array(
			'result'   => 'success',
			'redirect' => $redirect_url,
		);
	}

	public function process_refund( $order_id, $amount = null, $reason = '' ) {
		$order = wc_get_order( $order_id );

		if ( ! $order instanceof WC_Order ) {
			return new WP_Error( 'paypercut_invalid_order', __( 'Unable to locate order for refund.', 'paypercut-payments-for-woocommerce' ) );
		}

		$payment_intent_id = (string) $order->get_meta( self::META_PAYMENT_INTENT_ID, true );

		if ( '' === $payment_intent_id ) {
			return new WP_Error( 'paypercut_missing_transaction', __( 'No Paypercut payment reference is stored for this order.', 'paypercut-payments-for-woocommerce' ) );
		}

		$amount_to_refund = null === $amount ? (float) $order->get_total() : (float) $amount;

		if ( $amount_to_refund <= 0 ) {
			return new WP_Error( 'paypercut_invalid_amount', __( 'Refund amount must be greater than zero.', 'paypercut-payments-for-woocommerce' ) );
		}

		$payload = array(
			'amount' => $this->convert_amount_to_minor_units( $amount_to_refund ),
		);

		if ( '' !== $payment_intent_id ) {
			$payload['payment_intent'] = $payment_intent_id;
		}

		if ( '' !== $reason ) {
			$payload['reason'] = $reason;
		}

		try {
			$client   = new RefundClient( $this->get_api_client_secret() );
			$response = $client->create( $payload );
		} catch ( Throwable $exception ) {
			Logger::error(
				sprintf(
					'Paypercut refund failed for order #%1$d: %2$s',
					$order->get_id(),
					$exception->getMessage()
				)
			);

			return new WP_Error( 'paypercut_refund_failed', __( 'Unable to issue a Paypercut refund at this time. Please try again.', 'paypercut-payments-for-woocommerce' ) );
		}

		$refund_id = (string) ( $response['id'] ?? '' );

		$order->add_order_note(
			sprintf(
				/* translators: %1$s: Refund ID (optional), %2$s: Formatted refund amount */
				__( 'Paypercut refund created%1$s for %2$s.', 'paypercut-payments-for-woocommerce' ),
				$refund_id ? sprintf( ' (ID: %s)', esc_html( $refund_id ) ) : '',
				wc_price(
					$amount_to_refund,
					array(
						'currency' => $order->get_currency(),
					)
				)
			)
		);

		return true;
	}

	private function build_idempotency_key( WC_Order $order ): string {
		return sprintf( 'wc_paypercut_%d_%s', $order->get_id(), $order->get_order_key() );
	}

	private function extract_hosted_redirect_url( array $response ): string {
		if ( ! empty( $response['url'] ) && is_string( $response['url'] ) ) {
			return $response['url'];
		}

		return '';
	}

	private function render_section_fields( array $keys, array $form_fields ): void {
		if ( empty( $keys ) ) {
			return;
		}

		$fields_subset = array();

		foreach ( $keys as $key ) {
			if ( isset( $form_fields[ $key ] ) ) {
				$fields_subset[ $key ] = $form_fields[ $key ];
			}
		}

		if ( empty( $fields_subset ) ) {
			return;
		}

		$this->generate_settings_html( $fields_subset );
	}

	/**
	 * Render webhooks section with list of all webhooks.
	 *
	 * @return void
	 */
	private function render_webhooks_section(): void {
		$api_secret = $this->get_api_client_secret();

		if ( empty( $api_secret ) ) {
			echo '<p>' . esc_html__( 'Please configure your API credentials first to view webhooks.', 'paypercut-payments-for-woocommerce' ) . '</p>';
			return;
		}

		try {
			$api = new PaypercutApi( $api_secret );
			$response = $api->get_webhooks();
			$webhooks = isset( $response['items'] ) && is_array( $response['items'] ) ? $response['items'] : array();
		} catch ( Throwable $exception ) {
			Logger::error(
				'Failed to fetch webhooks',
				array(
					'exception' => get_class( $exception ),
				)
			);
			echo '<div class="notice notice-error inline"><p>' . esc_html__( 'Unable to load webhooks. Please check your API credentials.', 'paypercut-payments-for-woocommerce' ) . '</p></div>';
			return;
		}

		if ( empty( $webhooks ) ) {
			?>
			<div class="paypercut-webhooks__empty">
				<span class="dashicons dashicons-admin-links paypercut-webhooks__empty-icon"></span>
				<p class="paypercut-webhooks__empty-text"><?php esc_html_e( 'No webhooks found.', 'paypercut-payments-for-woocommerce' ); ?></p>
			</div>
			<?php
			return;
		}

		?>
		<div class="paypercut-webhooks">
			<table class="paypercut-webhooks__table wp-list-table widefat fixed striped">
			<thead>
				<tr>
					<th scope="col"><?php esc_html_e( 'ID', 'paypercut-payments-for-woocommerce' ); ?></th>
					<th scope="col"><?php esc_html_e( 'Events', 'paypercut-payments-for-woocommerce' ); ?></th>
					<th scope="col"><?php esc_html_e( 'Status', 'paypercut-payments-for-woocommerce' ); ?></th>
					<th scope="col"><?php esc_html_e( 'Actions', 'paypercut-payments-for-woocommerce' ); ?></th>
				</tr>
			</thead>
			<tbody>
				<?php foreach ( $webhooks as $webhook ) : ?>
					<?php
					$webhook_id = isset( $webhook['id'] ) ? esc_html( (string) $webhook['id'] ) : '—';
					$webhook_events = isset( $webhook['enabled_events'] ) && is_array( $webhook['enabled_events'] )
						? implode( ', ', array_map( 'esc_html', $webhook['enabled_events'] ) )
						: '—';
					$webhook_status = isset( $webhook['status'] ) ? (string) $webhook['status'] : '';
					$is_active = 'active' === strtolower( $webhook_status );
					?>
					<tr>
						<td><code><?php echo esc_html( $webhook_id ); ?></code></td>
						<td><span class="paypercut-webhooks__events"><?php echo esc_html( $webhook_events ); ?></span></td>
						<td>
							<?php if ( $is_active ) : ?>
								<span class="paypercut-webhooks__status paypercut-webhooks__status--active">
									<span class="dashicons dashicons-yes-alt" aria-label="<?php esc_attr_e( 'Active', 'paypercut-payments-for-woocommerce' ); ?>"></span>
									<?php esc_html_e( 'Active', 'paypercut-payments-for-woocommerce' ); ?>
								</span>
							<?php else : ?>
								<span class="paypercut-webhooks__status paypercut-webhooks__status--inactive">
									<span class="dashicons dashicons-dismiss" aria-label="<?php esc_attr_e( 'Inactive', 'paypercut-payments-for-woocommerce' ); ?>"></span>
									<?php echo esc_html( $webhook_status ?: __( 'Inactive', 'paypercut-payments-for-woocommerce' ) ); ?>
								</span>
							<?php endif; ?>
						</td>
						<td>
							<button 
								type="button" 
								class="button button-secondary paypercut-delete-webhook" 
								data-webhook-id="<?php echo esc_attr( $webhook_id ); ?>"
								data-nonce="<?php echo esc_attr( wp_create_nonce( 'paypercut_delete_webhook' ) ); ?>"
							>
								<?php esc_html_e( 'Delete', 'paypercut-payments-for-woocommerce' ); ?>
							</button>
						</td>
					</tr>
				<?php endforeach; ?>
			</tbody>
		</table>
		</div>
		<?php
	}

	public function payment_fields(): void {
		parent::payment_fields();

		$force_embedded = function_exists( 'is_add_payment_method_page' ) && is_add_payment_method_page();

	if ( 'embedded' !== $this->checkout_mode && ! $force_embedded ) {
		return;
	}

	echo '<div class="paypercut-checkout__container" aria-live="polite" aria-busy="true">';
	echo '<div id="paypercut-checkout-container"></div>';
	echo '</div>';

		$hidden_fields = array(
			'paypercut_session_id',
			'paypercut_payment_intent_id',
		);

		foreach ( $hidden_fields as $field_name ) {
			printf(
				'<input type="hidden" name="%1$s" id="%1$s" value="" />',
				esc_attr( $field_name )
			);
		}
	}

	public function get_checkout_mode(): string {
		return $this->checkout_mode;
	}

	public function get_icon(): string {
		$brands = $this->render_brand_icons();

		if ( $this->hide_logo ) {
			return $brands;
		}

		return parent::get_icon() . $brands;
	}

	private function render_brand_icons(): string {
		$assets = array(
			array(
				'src'    => Assets::url( 'images/visa.svg' ),
				'alt'    => __( 'Visa', 'paypercut-payments-for-woocommerce' ),
				'height' => 22,
			),
			array(
				'src'    => Assets::url( 'images/mastercard.svg' ),
				'alt'    => __( 'Mastercard', 'paypercut-payments-for-woocommerce' ),
				'height' => 22,
			),
			array(
				'src'    => Assets::url( 'images/apple.svg' ),
				'alt'    => __( 'Apple Pay', 'paypercut-payments-for-woocommerce' ),
				'height' => 22,
			),
			array(
				'src'    => Assets::url( 'images/google.svg' ),
				'alt'    => __( 'Google Pay', 'paypercut-payments-for-woocommerce' ),
				'height' => 22,
			),
		);

		$icons = array();

		foreach ( $assets as $asset ) {
			if ( empty( $asset['src'] ) ) {
				continue;
			}

			$height = isset( $asset['height'] ) ? (int) $asset['height'] : 22;

			$icons[] = sprintf(
				'<img src="%1$s" alt="%2$s" class="paypercut-card-brands__icon" loading="lazy" decoding="async" height="%3$d" />',
				esc_url( $asset['src'] ),
				esc_attr( $asset['alt'] ),
				$height
			);
		}

		if ( empty( $icons ) ) {
			return '';
		}

		return sprintf(
			'<span class="paypercut-card-brands" aria-hidden="true">%s</span>',
			implode( '', $icons )
		);
	}

	/**
	 * Store Paypercut identifiers on the order and add descriptive notes.
	 */
	private function persist_transaction_identifiers( WC_Order $order, string $session_id, string $payment_id, string $payment_intent_id ): void {
		$this->store_identifier(
			$order,
			self::META_SESSION_ID,
			$session_id,
			/* translators: %s: Session ID */
			__( 'Paypercut checkout session ID: %s', 'paypercut-payments-for-woocommerce' )
		);


		$this->store_identifier(
			$order,
			self::META_PAYMENT_INTENT_ID,
			$payment_intent_id,
			/* translators: %s: Payment Intent ID */
			__( 'Paypercut payment intent ID: %s', 'paypercut-payments-for-woocommerce' )
		);
	}

	private function store_identifier( WC_Order $order, string $meta_key, string $value, string $note_template ): void {
		if ( '' === $value ) {
			return;
		}

		$current_value = (string) $order->get_meta( $meta_key, true );

		if ( $current_value === $value ) {
			return;
		}

		$order->update_meta_data( $meta_key, $value );
		$order->add_order_note( sprintf( $note_template, esc_html( $value ) ) );
	}

	/**
	 * Extract identifiers from the current POST request or WooCommerce session.
	 *
	 * Note: This method is called during WooCommerce checkout processing,
	 * which has its own nonce verification via WooCommerce's form handling.
	 *
	 * @return array{session: string, payment: string, payment_intent: string}
	 */
	private function extract_transaction_identifiers_from_request(): array {
		$identifiers = array(
			'session'        => $this->get_post_value( 'paypercut_session_id' ),
			'payment'        => $this->get_post_value( 'paypercut_payment_id' ),
			'payment_intent' => $this->get_post_value( 'paypercut_payment_intent_id' ),
		);

		if ( function_exists( 'WC' ) && WC()->session ) {
			$identifiers['session']        = $identifiers['session'] ?: (string) WC()->session->get( 'paypercut_checkout_session_id', '' );
			$identifiers['payment']        = $identifiers['payment'] ?: (string) WC()->session->get( 'paypercut_checkout_payment_id', '' );
			$identifiers['payment_intent'] = $identifiers['payment_intent'] ?: (string) WC()->session->get( 'paypercut_checkout_payment_intent_id', '' );
		}

		return $identifiers;
	}

	/**
	 * Extract identifiers from a Paypercut API response.
	 *
	 * @param array<string, mixed> $response Response payload.
	 *
	 * @return array{session: string, payment: string, payment_intent: string}
	 */
	private function extract_transaction_identifiers_from_response( array $response ): array {
		return array(
			'session'        => $this->extract_identifier_from_source( $response, array( 'id', 'sessionId' ) ),
			'payment'        => $this->extract_identifier_from_source( $response, array( 'payment', 'paymentId' ) ),
			'payment_intent' => $this->extract_identifier_from_source( $response, array( 'payment_intent', 'paymentIntent', 'paymentIntentId' ) ),
		);
	}

	/**
	 * @param array<string, mixed> $source Source array.
	 * @param array<int, string>   $keys   Keys to attempt in order.
	 */
	private function extract_identifier_from_source( array $source, array $keys ): string {
		foreach ( $keys as $key ) {
			if ( isset( $source[ $key ] ) && is_string( $source[ $key ] ) ) {
				return $source[ $key ];
			}
		}

		return '';
	}

	/**
	 * Get sanitized POST value.
	 *
	 * Note: This method verifies the WooCommerce checkout nonce before accessing POST data.
	 *
	 * @param string $key POST key.
	 * @return string Sanitized value.
	 */
	private function get_post_value( string $key ): string {
		// Verify WooCommerce checkout nonce if available.
		if ( isset( $_REQUEST['woocommerce-process-checkout-nonce'] ) ) {
			$nonce_value = sanitize_text_field( wp_unslash( $_REQUEST['woocommerce-process-checkout-nonce'] ) );
			if ( ! wp_verify_nonce( $nonce_value, 'woocommerce-process_checkout' ) ) {
				return '';
			}
		}

		if ( ! isset( $_POST[ $key ] ) ) {
			return '';
		}

		return sanitize_text_field( wp_unslash( (string) $_POST[ $key ] ) );
	}

	private function clear_cached_checkout_identifiers(): void {
		if ( ! function_exists( 'WC' ) || ! WC()->session ) {
			return;
		}

		foreach ( array(
			'paypercut_checkout_session_id',
			'paypercut_checkout_payment_intent_id',
		) as $session_key ) {
			WC()->session->set( $session_key, null );
		}
	}

	private function convert_amount_to_minor_units( float $amount ): int {
		$decimals = (int) wc_get_price_decimals();

		return (int) round( $amount * pow( 10, max( 0, $decimals ) ) );
	}

	/**
	 * Generate API Key Secret field HTML with test connection button.
	 *
	 * @param string $key  Field key.
	 * @param array  $data Field data.
	 *
	 * @return string
	 */
	public function generate_api_client_secret_html( $key, $data ): string {
		$field_key = $this->get_field_key( $key );
		$defaults  = array(
			'title'             => '',
			'disabled'          => false,
			'class'             => '',
			'css'               => '',
			'placeholder'       => '',
			'type'              => 'password',
			'desc_tip'          => false,
			'description'       => '',
			'custom_attributes' => array(),
		);

		$data = wp_parse_args( $data, $defaults );

		ob_start();
		?>
		<tr valign="top">
			<th scope="row" class="titledesc">
				<label for="<?php echo esc_attr( $field_key ); ?>"><?php echo wp_kses_post( $data['title'] ); ?> <?php echo wp_kses_post( $this->get_tooltip_html( $data ) ); ?></label>
			</th>
			<td class="forminp">
				<fieldset>
					<legend class="screen-reader-text"><span><?php echo wp_kses_post( $data['title'] ); ?></span></legend>
					<input
						class="input-text regular-input <?php echo esc_attr( $data['class'] ); ?>"
						type="password"
						name="<?php echo esc_attr( $field_key ); ?>"
						id="<?php echo esc_attr( $field_key ); ?>"
						style="<?php echo esc_attr( $data['css'] ); ?>"
						value="<?php echo esc_attr( $this->get_option( $key ) ); ?>"
						placeholder="<?php echo esc_attr( $data['placeholder'] ); ?>"
						<?php disabled( $data['disabled'], true ); ?>
						<?php echo wp_kses_post( $this->get_custom_attribute_html( $data ) ); ?>
					/>
					<button type="button" id="paypercut-test-connection" class="button" style="margin-left: 10px;">
						<?php esc_html_e( 'Test Connection', 'paypercut-payments-for-woocommerce' ); ?>
					</button>
					<div id="paypercut-connection-status" style="margin-top: 10px; display: none; border-radius: 4px;"></div>
					<?php echo wp_kses_post( $this->get_description_html( $data ) ); ?>
				</fieldset>
			</td>
		</tr>
		<?php
		return ob_get_clean();
	}

	/**
	 * Retrieve the configured API client secret.
	 */
	public function get_api_client_secret(): string {
		return $this->api_client_secret;
	}

	/**
	 * Retrieve the configured webhook secret.
	 */
	public function get_webhook_secret(): string {
		return $this->webhook_secret;
	}

	public function get_appearance_settings(): array {
		return array(
			'preset' => $this->appearance_preset ?: 'inline',
			'theme'  => $this->appearance_theme ?: 'light',
			'labels' => $this->appearance_labels ?: 'above',
		);
	}

	public function allows_saved_payment_methods(): bool {
		return $this->saved_payment_methods_enabled;
	}

	/**
	 * Register custom payment token class.
	 *
	 * @param string $class Token class name.
	 * @param string $type  Token type.
	 * @return string
	 */

	/**
	 * Format payment method display for My Account page.
	 *
	 * @param array<string, mixed> $item          Payment method item.
	 * @param object               $payment_token Payment token object.
	 * @return array<string, mixed>
	 */
	public function format_payment_method_display( array $item, $payment_token ): array {
		if ( self::ID !== $payment_token->get_gateway_id() ) {
			return $item;
		}

		if ( method_exists( $payment_token, 'get_card_type' ) && method_exists( $payment_token, 'get_last4' ) ) {
			$item['method']['brand'] = $payment_token->get_card_type();
			$item['method']['last4'] = $payment_token->get_last4();
		}

		if ( method_exists( $payment_token, 'get_expiry_month' ) && method_exists( $payment_token, 'get_expiry_year' ) ) {
			$expiry_month = $payment_token->get_expiry_month();
			$expiry_year  = $payment_token->get_expiry_year();

			if ( $expiry_month && $expiry_year ) {
				$item['expires'] = sprintf(
					'%02d/%s',
					(int) $expiry_month,
					substr( $expiry_year, -2 )
				);
			}
		}

		return $item;
	}

	/**
	 * Check if the gateway is fully configured and ready to use.
	 *
	 * @return bool True if setup is complete, false otherwise.
	 */
	public function is_setup_complete(): bool {
		return ! empty( $this->api_client_secret );
	}

	/**
	 * Save a payment method token from a checkout session.
	 *
	 * @param array<string, mixed> $session_data The checkout session data with expanded payment method.
	 * @param int                  $user_id      The WordPress user ID.
	 * @return PaypercutPaymentToken|null The saved token or null on failure.
	 */
	private function save_payment_method_from_session( array $session_data, int $user_id ): ?PaypercutPaymentToken {
		if ( ! isset( $session_data['setup_intent']['payment_method'] ) ) {
			return null;
		}

		$payment_method = $session_data['setup_intent']['payment_method'];

		if ( ! isset( $payment_method['id'] ) || ! isset( $payment_method['card'] ) ) {
			Logger::error( 'Invalid payment method data in session' );
			return null;
		}

		if ( ! $user_id || $user_id <= 0 ) {
			Logger::error( 'Cannot save payment method: No customer ID' );
			return null;
		}

		$existing_tokens = WC_Payment_Tokens::get_customer_tokens( $user_id, self::ID );

		foreach ( $existing_tokens as $existing_token ) {
			if ( $existing_token->get_token() === $payment_method['id'] ) {
				Logger::info( 'Payment method token already exists', array( 'token_id' => $payment_method['id'] ) );
				return $existing_token;
			}
		}

		$token = new PaypercutPaymentToken();
		$token->set_token( $payment_method['id'] );
		$token->set_gateway_id( self::ID );
		$token->set_user_id( $user_id );

		$card = $payment_method['card'];
		$token->set_card_type( $card['brand'] ?? 'card' );
		$token->set_last4( $card['last4'] ?? '' );
		$token->set_expiry_month( (string) ( $card['exp_month'] ?? '' ) );
		$token->set_expiry_year( (string) ( $card['exp_year'] ?? '' ) );

		if ( $token->save() ) {
			Logger::info(
				'Payment method token saved',
				array(
					'token_id' => $payment_method['id'],
					'user_id'  => $user_id,
					'last4'    => $token->get_last4(),
				)
			);

			return $token;
		}

		Logger::error( 'Failed to save payment method token' );
		return null;
	}
}
