<?php
/**
 * Paypercut BNPL gateway definition.
 *
 * @package Paypercut\Payments\Gateway
 */

declare(strict_types=1);

namespace Paypercut\Payments\Gateway;

use Paypercut\Payments\Admin\SetupWizard;
use Paypercut\Payments\Api\PaypercutApi;
use Paypercut\Payments\Services\CheckoutPayloadFactory;
use Paypercut\Payments\Services\PaypercutCheckoutClient;
use Paypercut\Payments\Support\Assets;
use Paypercut\Payments\Support\Logger;
use Throwable;
use WC_Order;
use WC_Payment_Gateway;
use WP_Error;
use function __;
use function esc_attr;
use function esc_html__;
use function esc_html;
use function esc_url;
use function is_string;
use function wc_add_notice;
use function wc_get_order;
use function wc_price;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Simple BNPL gateway implementation for Paypercut.
 *
 * Uses hosted checkout with BNPL as the payment method type.
 */
final class PaypercutBnplGateway extends WC_Payment_Gateway {
	public const ID                   = 'paypercut_bnpl';
	public const META_BNPL_ATTEMPT_ID = '_paypercut_bnpl_attempt_id';

	/**
	 * API secret.
	 *
	 * @var string
	 */
	private string $api_client_secret = '';

	/**
	 * Mode (live or test).
	 *
	 * @var string
	 */
	private string $mode = 'test';

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->id                 = self::ID;
		$this->icon               = Assets::url( 'images/icon-128x128.png' );
		$this->method_title       = __( 'Paypercut BNPL', 'paypercut-payments-for-woocommerce' );
		$this->method_description = __( 'Offer Paypercut Buy Now, Pay Later via hosted checkout.', 'paypercut-payments-for-woocommerce' );
		$this->supports           = array(
			'products',
		);

		$this->init_form_fields();
		$this->init_settings();

		$this->enabled           = $this->get_option( 'enabled', 'no' );
		$this->title             = $this->get_option( 'title', __( 'Pay later with Paypercut', 'paypercut-payments-for-woocommerce' ) );
		$this->description       = $this->get_option( 'description', '' );
		$this->api_client_secret = $this->get_option( 'api_client_secret', '' );
		$this->mode              = $this->get_option( 'mode', 'test' );

		$this->has_fields = false;

		add_action(
			'woocommerce_update_options_payment_gateways_' . $this->id,
			function (): void {
				$this->process_admin_options();
			}
		);
	}

	/**
	 * Initialise BNPL gateway settings fields.
	 *
	 * Mirrors the main gateway basics but without checkout configuration
	 * or embedded/appearance options. Advanced section will only expose logging.
	 */
	public function init_form_fields(): void {
		$this->form_fields = array(
			'enabled'           => array(
				'title'   => __( 'Enable/Disable', 'paypercut-payments-for-woocommerce' ),
				'label'   => __( 'Enable Paypercut BNPL', 'paypercut-payments-for-woocommerce' ),
				'type'    => 'checkbox',
				'default' => 'no',
			),
			'title'             => array(
				'title'       => __( 'Title', 'paypercut-payments-for-woocommerce' ),
				'type'        => 'text',
				'default'     => __( 'Pay later with Paypercut', 'paypercut-payments-for-woocommerce' ),
				'description' => __( 'Displayed to customers during checkout.', 'paypercut-payments-for-woocommerce' ),
			),
			'description'       => array(
				'title'       => __( 'Description', 'paypercut-payments-for-woocommerce' ),
				'type'        => 'textarea',
				'default'     => '',
				'description' => __( 'Payment method description that the customer will see on your checkout.', 'paypercut-payments-for-woocommerce' ),
			),
			'mode'              => array(
				'title'       => __( 'Mode', 'paypercut-payments-for-woocommerce' ),
				'type'        => 'select',
				'description' => __( 'Select Live for production or Test for development.', 'paypercut-payments-for-woocommerce' ),
				'default'     => 'test',
				'options'     => array(
					'test' => __( 'Test', 'paypercut-payments-for-woocommerce' ),
					'live' => __( 'Live', 'paypercut-payments-for-woocommerce' ),
				),
			),
			'api_client_secret' => array(
				'title'       => __( 'API Key Secret', 'paypercut-payments-for-woocommerce' ),
				'type'        => 'api_client_secret',
				'description' => __( 'Bearer token for Paypercut API requests.', 'paypercut-payments-for-woocommerce' ),
			),
			'logging'           => array(
				'title'       => __( 'Logging', 'paypercut-payments-for-woocommerce' ),
				'label'       => __( 'Log debug messages', 'paypercut-payments-for-woocommerce' ),
				'type'        => 'checkbox',
				'description' => __( 'Writes gateway events to WooCommerce Status → Logs.', 'paypercut-payments-for-woocommerce' ),
				'default'     => 'yes',
			),
		);
	}

	/**
	 * Render the BNPL gateway settings page with minimal sections.
	 */
	public function admin_options(): void {
		if ( ! $this->is_setup_complete() ) {
			( new SetupWizard() )->render();
			return;
		}

		$banner_large = Assets::url( 'images/banner-1544x500.png' );
		$banner_small = Assets::url( 'images/banner-772x250.png' );
		$icon         = Assets::url( 'images/icon-128x128.png' );

		$webhook_url = rest_url( 'paypercut/v1/webhook/bnpl' );

		$sections = array(
			'general'     => array(
				'title' => __( 'Display', 'paypercut-payments-for-woocommerce' ),
				'keys'  => array( 'enabled', 'title', 'description' ),
			),
			'credentials' => array(
				'title' => __( 'API Credentials', 'paypercut-payments-for-woocommerce' ),
				'keys'  => array( 'api_client_secret', 'mode' ),
			),
			'webhooks'    => array(
				'title'       => __( 'Webhooks', 'paypercut-payments-for-woocommerce' ),
				'description' => '',
				'keys'        => array(),
				'webhook_url' => $webhook_url,
			),
			'advanced'    => array(
				'title'       => __( 'Advanced', 'paypercut-payments-for-woocommerce' ),
				'description' => __( 'Enable or disable debug logging for BNPL payments.', 'paypercut-payments-for-woocommerce' ),
				'keys'        => array( 'logging' ),
			),
		);

		$form_fields = $this->get_form_fields();

		?>
		<div class="paypercut-settings">
			<div class="paypercut-settings__hero">
				<picture>
					<source media="(max-width: 782px)" srcset="<?php echo esc_url( $banner_small ); ?>">
					<img src="<?php echo esc_url( $banner_large ); ?>" alt="<?php esc_attr_e( 'Paypercut banner', 'paypercut-payments-for-woocommerce' ); ?>">
				</picture>
			</div>
			<div class="paypercut-settings__header">
				<img src="<?php echo esc_url( $icon ); ?>" alt="<?php esc_attr_e( 'Paypercut logo', 'paypercut-payments-for-woocommerce' ); ?>">
				<div>
					<h2><?php echo esc_html( $this->method_title ); ?></h2>
					<p><?php echo esc_html__( 'Configure Paypercut BNPL hosted checkout settings.', 'paypercut-payments-for-woocommerce' ); ?></p>
				</div>
			</div>
			<?php foreach ( $sections as $section_key => $section ) : ?>
				<div class="paypercut-settings__section">
					<h3 class="paypercut-settings__section-title"><?php echo esc_html( $section['title'] ); ?></h3>
					<?php if ( ! empty( $section['description'] ) ) : ?>
						<div class="notice notice-warning inline" style="margin:14px;">
							<p><?php echo esc_html( $section['description'] ); ?></p>
						</div>
					<?php endif; ?>
					<?php if ( 'webhooks' === $section_key ) : ?>
						<div class="notice notice-warning inline" style="margin: 14px;">
							<p>
								<?php esc_html_e( 'Webhook configuration is required for automatic order status updates. Please contact Paypercut support to configure webhooks using the URL below.', 'paypercut-payments-for-woocommerce' ); ?>
							</p>
						</div>
						<table class="form-table paypercut-settings__table">
							<tbody>
								<tr valign="top">
									<th scope="row" class="titledesc">
										<label for="paypercut-bnpl-webhook-url"><?php esc_html_e( 'Webhook URL', 'paypercut-payments-for-woocommerce' ); ?></label>
									</th>
									<td class="forminp">
										<fieldset>
											<legend class="screen-reader-text"><span><?php esc_html_e( 'Webhook URL', 'paypercut-payments-for-woocommerce' ); ?></span></legend>
											<input 
												class="input-text regular-input" 
												type="text" 
												readonly 
												value="<?php echo esc_attr( $section['webhook_url'] ); ?>" 
												id="paypercut-bnpl-webhook-url"
												onclick="this.select();"
											/>
											<button 
												type="button" 
												class="button" 
												style="margin-left: 10px;"
												onclick="navigator.clipboard.writeText(document.getElementById('paypercut-bnpl-webhook-url').value).then(function(){alert('<?php echo esc_js( __( 'Webhook URL copied to clipboard!', 'paypercut-payments-for-woocommerce' ) ); ?>');})"
											>
												<?php esc_html_e( 'Copy', 'paypercut-payments-for-woocommerce' ); ?>
											</button>
											<p class="description">
												<?php esc_html_e( 'Provide this URL to Paypercut when setting up your webhooks.', 'paypercut-payments-for-woocommerce' ); ?>
											</p>
										</fieldset>
									</td>
								</tr>
							</tbody>
						</table>
					<?php else : ?>
						<table class="form-table paypercut-settings__table">
							<?php $this->render_section_fields( $section['keys'] ?? array(), $form_fields ); ?>
						</table>
					<?php endif; ?>
				</div>
			<?php endforeach; ?>
		</div>
		<?php
	}

	/**
	 * Render a subset of form fields inside a section.
	 *
	 * @param array $keys        Keys to render.
	 * @param array $form_fields All form fields.
	 *
	 * @return void
	 */
	private function render_section_fields( array $keys, array $form_fields ): void {
		if ( empty( $keys ) ) {
			return;
		}

		$fields_subset = array();

		foreach ( $keys as $key ) {
			if ( isset( $form_fields[ $key ] ) ) {
				$fields_subset[ $key ] = $form_fields[ $key ];
			}
		}

		if ( empty( $fields_subset ) ) {
			return;
		}

		$this->generate_settings_html( $fields_subset );
	}

	/**
	 * Process BNPL payments via hosted checkout.
	 *
	 * @param int $order_id Order identifier.
	 *
	 * @return array{result: string, redirect: string}|WP_Error
	 */
	public function process_payment( $order_id ) {
		Logger::info(
			'Paypercut BNPL process_payment called',
			array(
				'order_id' => $order_id,
			)
		);

		$order = wc_get_order( $order_id );

		if ( ! $order instanceof WC_Order ) {
			Logger::error(
				sprintf(
					'Failed to initiate BNPL payment for order #%d: WooCommerce order not found.',
					$order_id
				),
				array(
					'order_id' => $order_id,
				)
			);

			return new WP_Error( 'paypercut_bnpl_invalid_order', __( 'Unable to start Paypercut BNPL checkout. Please try again.', 'paypercut-payments-for-woocommerce' ) );
		}

		if ( empty( $this->get_api_client_secret() ) ) {
			Logger::error(
				'BNPL checkout: API secret not configured',
				array( 'order_id' => $order->get_id() )
			);

			wc_add_notice( __( 'Payment gateway configuration error. Please contact support.', 'paypercut-payments-for-woocommerce' ), 'error' );

			return array(
				'result'   => 'failure',
				'redirect' => '',
			);
		}

		return $this->process_hosted_checkout( $order );
	}

	/**
	 * Hosted checkout implementation for BNPL.
	 */
	private function process_hosted_checkout( WC_Order $order ): array {
		$factory = new CheckoutPayloadFactory();
		$client  = PaypercutCheckoutClient::forBnpl( $this->get_api_client_secret(), $this->get_mode() );

		try {
			$payload = $factory->build_bnpl_from_order( $order, $this );

			Logger::info(
				'Paypercut BNPL hosted checkout payload prepared',
				array(
					'order_id' => $order->get_id(),
				)
			);

			$response = $client->create_bnpl_attempt(
				$payload,
				array(
					'idempotency_key' => $this->build_idempotency_key( $order ),
				)
			);

			$attempt_id = isset( $response['attempt_id'] ) && is_string( $response['attempt_id'] )
				? $response['attempt_id']
				: '';

			$this->store_identifier(
				$order,
				self::META_BNPL_ATTEMPT_ID,
				$attempt_id,
				/* translators: %s: BNPL attempt ID */
				__( 'Paypercut BNPL attempt ID: %s', 'paypercut-payments-for-woocommerce' )
			);

			Logger::info(
				'Paypercut BNPL hosted checkout response',
				array(
					'order_id'   => $order->get_id(),
					'attempt_id' => $attempt_id,
				)
			);
		} catch ( Throwable $exception ) {
			Logger::error(
				'BNPL hosted checkout session failed',
				array(
					'order_id' => $order->get_id(),
					'error'    => $exception->getMessage(),
				)
			);

			wc_add_notice( __( 'Unable to start Paypercut BNPL checkout. Please try again.', 'paypercut-payments-for-woocommerce' ), 'error' );

			return array(
				'result'   => 'failure',
				'redirect' => '',
			);
		}

		$redirect_url = $this->extract_hosted_redirect_url( $response );

		if ( '' === $redirect_url ) {
			Logger::error(
				'BNPL hosted checkout response missing redirect URL',
				array(
					'order_id' => $order->get_id(),
				)
			);

			wc_add_notice( __( 'Unable to start Paypercut BNPL checkout. Please try again.', 'paypercut-payments-for-woocommerce' ), 'error' );

			return array(
				'result'   => 'failure',
				'redirect' => '',
			);
		}

		$order->update_status(
			'pending',
			__( 'Awaiting completion of Paypercut BNPL hosted checkout.', 'paypercut-payments-for-woocommerce' )
		);

		$order->save();

		Logger::info(
			'Paypercut BNPL hosted checkout order saved, redirecting',
			array(
				'order_id'     => $order->get_id(),
				'order_status' => $order->get_status(),
				'redirect_url' => $redirect_url,
			)
		);

		return array(
			'result'   => 'success',
			'redirect' => $redirect_url,
		);
	}

	private function build_idempotency_key( WC_Order $order ): string {
		return sprintf( 'wc_paypercut_bnpl_%d_%s', $order->get_id(), $order->get_order_key() );
	}

	private function extract_hosted_redirect_url( array $response ): string {
		if ( ! empty( $response['redirect_url'] ) && is_string( $response['redirect_url'] ) ) {
			return $response['redirect_url'];
		}

		return '';
	}

	private function store_identifier( WC_Order $order, string $meta_key, string $value, string $note_template ): void {
		if ( '' === $value ) {
			return;
		}

		$current_value = (string) $order->get_meta( $meta_key, true );

		if ( $current_value === $value ) {
			return;
		}

		$order->update_meta_data( $meta_key, $value );
		$order->add_order_note( sprintf( $note_template, esc_html( $value ) ) );
	}


	/**
	 * Retrieve the configured API client secret.
	 */
	public function get_api_client_secret(): string {
		return $this->api_client_secret;
	}

	/**
	 * Get the mode (live or test).
	 *
	 * @return string
	 */
	public function get_mode(): string {
		return $this->mode;
	}

	/**
	 * Generate API Key Secret field HTML with test connection button.
	 *
	 * @param string $key  Field key.
	 * @param array  $data Field data.
	 *
	 * @return string
	 */
	public function generate_api_client_secret_html( $key, $data ): string {
		$field_key = $this->get_field_key( $key );
		$defaults  = array(
			'title'             => '',
			'disabled'          => false,
			'class'             => '',
			'css'               => '',
			'placeholder'       => '',
			'type'              => 'password',
			'desc_tip'          => false,
			'description'       => '',
			'custom_attributes' => array(),
		);

		$data = wp_parse_args( $data, $defaults );

		ob_start();
		?>
		<tr valign="top">
			<th scope="row" class="titledesc">
				<label for="<?php echo esc_attr( $field_key ); ?>"><?php echo wp_kses_post( $data['title'] ); ?> <?php echo wp_kses_post( $this->get_tooltip_html( $data ) ); ?></label>
			</th>
			<td class="forminp">
				<fieldset>
					<legend class="screen-reader-text"><span><?php echo wp_kses_post( $data['title'] ); ?></span></legend>
					<input
						class="input-text regular-input <?php echo esc_attr( $data['class'] ); ?>"
						type="password"
						name="<?php echo esc_attr( $field_key ); ?>"
						id="<?php echo esc_attr( $field_key ); ?>"
						style="<?php echo esc_attr( $data['css'] ); ?>"
						value="<?php echo esc_attr( $this->get_option( $key ) ); ?>"
						placeholder="<?php echo esc_attr( $data['placeholder'] ); ?>"
						<?php disabled( $data['disabled'], true ); ?>
						<?php echo wp_kses_post( $this->get_custom_attribute_html( $data ) ); ?>
					/>
					<button type="button" id="paypercut-test-connection" class="button" style="margin-left: 10px;">
						<?php esc_html_e( 'Test Connection', 'paypercut-payments-for-woocommerce' ); ?>
					</button>
					<div id="paypercut-connection-status" style="margin-top: 10px; display: none; border-radius: 4px;"></div>
					<?php echo wp_kses_post( $this->get_description_html( $data ) ); ?>
				</fieldset>
			</td>
		</tr>
		<?php
		return ob_get_clean();
	}

	/**
	 * Check if the gateway is fully configured and ready to use.
	 */
	public function is_setup_complete(): bool {
		return ! empty( $this->api_client_secret );
	}
}


