<?php
/**
 * Checkout block payment method integration.
 *
 * @package Paypercut\Payments\Blocks
 */

declare(strict_types=1);

namespace Paypercut\Payments\Blocks;

use Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType;
use Paypercut\Payments\Gateway\PaypercutGateway;
use Paypercut\Payments\Support\Assets;
use function __;
use function admin_url;
use function get_option;
use function wp_create_nonce;
use function wp_register_script;

if ( ! class_exists( AbstractPaymentMethodType::class ) ) {
	return;
}

final class PaypercutBlocksSupport extends AbstractPaymentMethodType {
	protected $name = PaypercutGateway::ID;

	protected $settings = array();

	private ?PaypercutGateway $gateway = null;

	public function initialize() {
		$this->settings = (array) get_option( 'woocommerce_' . $this->name . '_settings', array() );
	}

	public function is_active() {
		return ( $this->settings['enabled'] ?? 'no' ) === 'yes';
	}

	public function get_payment_method_script_handles() {
		$asset_path = Assets::path( 'build/index.asset.php' );

		$asset = array(
			'dependencies' => array( 'wc-blocks-registry', 'wc-settings', 'wp-element', 'wp-html-entities' ),
			'version'      => PAYPERCUT_PAYMENTS_VERSION,
		);

		if ( file_exists( $asset_path ) ) {
			$resolved = require $asset_path;

			if ( is_array( $resolved ) ) {
				$asset = array_merge( $asset, $resolved );
			}
		}

		wp_register_script(
			'paypercut-blocks',
			Assets::url( 'build/index.js' ),
			$asset['dependencies'],
			$asset['version'],
			true
		);

		return array( 'paypercut-blocks' );
	}

	public function get_payment_method_script_handles_for_admin() {
		return $this->get_payment_method_script_handles();
	}

	public function get_supported_features() {
		$gateway = $this->resolve_gateway();
		if ( ! $gateway ) {
			return array( 'products' );
		}
		
		$features = array_filter( $gateway->supports, array( $gateway, 'supports' ) );
		
		return $features;
	}

	public function get_payment_method_data() {
		$gateway = $this->resolve_gateway();
		$appearance = $gateway && method_exists( $gateway, 'get_appearance_settings' ) ? $gateway->get_appearance_settings() : array();
		$checkout_mode = $gateway && method_exists( $gateway, 'get_checkout_mode' ) ? $gateway->get_checkout_mode() : 'hosted';

		return array(
			'gatewayId'              => PaypercutGateway::ID,
			'title'                  => $gateway ? $gateway->get_title() : __( 'Pay with Paypercut', 'paypercut-payments-for-woocommerce' ),
			'description'            => $gateway && method_exists( $gateway, 'get_description' ) ? $gateway->get_description() : '',
			'containerId'            => '#paypercut-checkout-container',
			'brandIcons'             => $this->get_brand_icons(),
			'ajaxUrl'                => admin_url( 'admin-ajax.php' ),
			'nonce'                  => wp_create_nonce( 'paypercut_checkout_session' ),
			'appearance'             => $appearance,
			'checkoutMode'           => $checkout_mode,
			'validationErrorMessage' => __( 'Please fill all required fields and accept the terms and conditions.', 'paypercut-payments-for-woocommerce' ),
			'sdkNotLoadedMessage'    => __( 'Paypercut SDK not loaded', 'paypercut-payments-for-woocommerce' ),
			'paymentInitFailedMessage' => __( 'Payment initialization failed', 'paypercut-payments-for-woocommerce' ),
			'initFailedMessage'      => __( 'Failed to initialize Paypercut', 'paypercut-payments-for-woocommerce' ),
			'sessionFailedMessage'   => __( 'Failed to create checkout session', 'paypercut-payments-for-woocommerce' ),
			'networkErrorMessage'    => __( 'Network error', 'paypercut-payments-for-woocommerce' ),
			'notInitializedMessage'  => __( 'Paypercut not initialized', 'paypercut-payments-for-woocommerce' ),
			'paymentFailedMessage'   => __( 'Payment could not be processed. Please try again.', 'paypercut-payments-for-woocommerce' ),
			'submissionFailedMessage' => __( 'Payment submission failed. Please try again.', 'paypercut-payments-for-woocommerce' ),
			'supports'               => array(
				'features' => $this->get_supported_features(),
			),
		);
	}

	private function get_brand_icons(): array {
		$icons = array();

		$brands = array(
			array(
				'src' => Assets::url( 'images/visa.svg' ),
				'alt' => __( 'Visa', 'paypercut-payments-for-woocommerce' ),
			),
			array(
				'src' => Assets::url( 'images/mastercard.svg' ),
				'alt' => __( 'Mastercard', 'paypercut-payments-for-woocommerce' ),
			),
			array(
				'src' => Assets::url( 'images/apple.svg' ),
				'alt' => __( 'Apple Pay', 'paypercut-payments-for-woocommerce' ),
			),
			array(
				'src' => Assets::url( 'images/google.svg' ),
				'alt' => __( 'Google Pay', 'paypercut-payments-for-woocommerce' ),
			),
		);

		foreach ( $brands as $brand ) {
			if ( ! empty( $brand['src'] ) ) {
				$icons[] = $brand;
			}
		}

		return $icons;
	}

	private function resolve_gateway(): ?PaypercutGateway {
		if ( $this->gateway instanceof PaypercutGateway ) {
			return $this->gateway;
		}

		if ( ! function_exists( 'WC' ) ) {
			return null;
		}

		$manager = WC()->payment_gateways();

		if ( ! $manager ) {
			return null;
		}

		$gateways = $manager->get_available_payment_gateways();

		if ( empty( $gateways[ PaypercutGateway::ID ] ) || ! $gateways[ PaypercutGateway::ID ] instanceof PaypercutGateway ) {
			return null;
		}

		$this->gateway = $gateways[ PaypercutGateway::ID ];

		return $this->gateway;
	}
}

