<?php
/**
 * Frontend assets helper.
 *
 * @package Paypercut\Payments\Assets
 */

declare(strict_types=1);

namespace Paypercut\Payments\Assets;

use Paypercut\Payments\Gateway\PaypercutGateway;
use Paypercut\Payments\Support\Assets;
use function __;
use function admin_url;
use function get_query_var;
use function has_block;
use function is_add_payment_method_page;
use function is_cart;
use function is_checkout;
use function is_order_received_page;
use function is_wc_endpoint_url;
use function wp_create_nonce;
use function wp_enqueue_script;
use function wp_enqueue_style;
use function wp_localize_script;
use function wp_register_script;

final class FrontendAssets {
	public function enqueue(): void {
		$is_checkout_page          = is_checkout() && ! is_order_received_page();
		$is_add_payment_method_page = function_exists( 'is_add_payment_method_page' ) && is_add_payment_method_page();
		$is_order_pay_page         = function_exists( 'is_wc_endpoint_url' ) && is_wc_endpoint_url( 'order-pay' );

		if ( ! $is_checkout_page && ! $is_add_payment_method_page && ! $is_order_pay_page ) {
			return;
		}

		if ( ! function_exists( 'WC' ) ) {
			return;
		}

		$woocommerce = WC();

		if ( ! $woocommerce ) {
			return;
		}

		$manager = $woocommerce->payment_gateways();

		if ( ! $manager ) {
			return;
		}

		$gateways = $manager->get_available_payment_gateways();

		if ( empty( $gateways[ PaypercutGateway::ID ] ) || ! $gateways[ PaypercutGateway::ID ] instanceof PaypercutGateway ) {
			return;
		}

		$gateway = $gateways[ PaypercutGateway::ID ];

		wp_enqueue_style(
			'paypercut-checkout-style',
			Assets::url( 'public/css/checkout.css' ),
			array(),
			PAYPERCUT_PAYMENTS_VERSION
		);

		$force_embedded = $is_add_payment_method_page;

		if ( 'embedded' !== $gateway->get_checkout_mode() && ! $force_embedded ) {
			return;
		}

		wp_register_script(
			'paypercut-checkout-sdk',
			Assets::url( 'public/js/vendor/paypercut-checkout.iife.min.js' ),
			array(),
			PAYPERCUT_PAYMENTS_VERSION,
			true
		);

		wp_register_script(
			'paypercut-checkout',
			Assets::url( 'public/js/paypercut-checkout.js' ),
			array( 'paypercut-checkout-sdk' ),
			PAYPERCUT_PAYMENTS_VERSION,
			true
		);

		$session_id = '';

		if ( $woocommerce->session ) {
			$stored = $woocommerce->session->get( 'paypercut_checkout_session_id' );

			if ( is_string( $stored ) ) {
				$session_id = $stored;
			}
		}

		$page_context = $is_add_payment_method_page ? 'add_payment_method' : 'checkout';
		$ajax_context = $is_add_payment_method_page ? 'setup' : 'checkout';

		$order_id = 0;
		if ( $is_order_pay_page && function_exists( 'get_query_var' ) ) {
			$order_id = absint( get_query_var( 'order-pay' ) );
		}

		$is_blocks_checkout = false;
		if ( $is_checkout_page && ! $is_order_pay_page && function_exists( 'has_block' ) && ( is_checkout() || is_cart() ) ) {
			$is_blocks_checkout = has_block( 'woocommerce/checkout' ) || has_block( 'woocommerce/cart' );
		}

		wp_enqueue_script( 'paypercut-checkout-sdk' );

		if ( ! $is_blocks_checkout ) {
			$localize_data = array(
				'gatewayId'               => PaypercutGateway::ID,
				'containerId'             => '#paypercut-checkout-container',
				'sessionId'               => $session_id,
				'ajaxUrl'                 => admin_url( 'admin-ajax.php' ),
				'nonce'                   => wp_create_nonce( 'paypercut_checkout_session' ),
				'loadingMessage'          => __( 'Setting up secure Paypercut checkout…', 'paypercut-payments-for-woocommerce' ),
				'errorMessage'            => __( 'Unable to initialise Paypercut checkout. Please try again.', 'paypercut-payments-for-woocommerce' ),
				'validationErrorMessage'  => __( 'Please fill all required fields to proceed and complete your order.', 'paypercut-payments-for-woocommerce' ),
				'appearance'              => method_exists( $gateway, 'get_appearance_settings' ) ? $gateway->get_appearance_settings() : array(),
				'context'                 => $page_context,
				'ajaxContext'             => $ajax_context,
				'addPaymentFormSelector'  => '#add_payment_method',
				'addPaymentButtonSelector'=> '#add_payment_method button[type="submit"]',
			);

			if ( $order_id > 0 ) {
				$localize_data['orderId'] = $order_id;
			}

			wp_localize_script(
				'paypercut-checkout',
				'paypercutCheckoutData',
				$localize_data
			);

			wp_enqueue_script( 'paypercut-checkout' );
		}
	}
}
