<?php
/**
 * Admin assets helper.
 *
 * @package Paypercut\Payments\Assets
 */

declare(strict_types=1);

namespace Paypercut\Payments\Assets;

use Paypercut\Payments\Gateway\PaypercutGateway;
use Paypercut\Payments\Gateway\PaypercutBnplGateway;
use Paypercut\Payments\Support\Assets;
use WC_Order;
use function admin_url;
use function current_user_can;
use function filter_input;
use function get_option;
use function is_string;
use function sanitize_key;
use function wc_get_order;
use function wp_create_nonce;
use function wp_enqueue_script;
use function wp_enqueue_style;
use function wp_localize_script;
use function wp_unslash;

final class AdminAssets {
	public function enqueue( string $hook ): void {
		$this->enqueue_settings_assets( $hook );
		$this->enqueue_setup_wizard_assets( $hook );
		$this->enqueue_refund_helper( $hook );
	}

	private function enqueue_settings_assets( string $hook ): void {
		if ( 'woocommerce_page_wc-settings' !== $hook ) {
			return;
		}

		$raw_tab     	= filter_input( INPUT_GET, 'tab', FILTER_UNSAFE_RAW );
		$raw_section 	= filter_input( INPUT_GET, 'section', FILTER_UNSAFE_RAW );
		$tab     		= sanitize_key( is_string( $raw_tab ) ? wp_unslash( $raw_tab ) : '' );
		$section 		= sanitize_key( is_string( $raw_section ) ? wp_unslash( $raw_section ) : '' );

		if ( 'checkout' !== $tab || ! in_array( $section, array( PaypercutGateway::ID, PaypercutBnplGateway::ID ), true ) ) {
			return;
		}

		wp_enqueue_style(
			'paypercut-payments-admin',
			Assets::url( 'admin/css/settings.css' ),
			array(),
			PAYPERCUT_PAYMENTS_VERSION
		);

		wp_enqueue_script(
			'paypercut-test-connection',
			Assets::url( 'admin/js/test-connection.js' ),
			array(),
			PAYPERCUT_PAYMENTS_VERSION,
			true
		);

		wp_localize_script(
			'paypercut-test-connection',
			'paypercutTestConnection',
			array(
				'ajaxurl' => admin_url( 'admin-ajax.php' ),
				'nonce'   => wp_create_nonce( 'paypercut_test_connection' ),
				'i18n'    => array(
					'apiSecretRequired'   	=> __( 'Please enter an API key secret before testing.', 'paypercut-payments-for-woocommerce' ),
					'testing'             	=> __( 'Testing...', 'paypercut-payments-for-woocommerce' ),
					'connectionSuccessful' 	=> __( 'Connection successful!', 'paypercut-payments-for-woocommerce' ),
					'connectionFailed'     	=> __( 'Connection failed. Please check your API key secret.', 'paypercut-payments-for-woocommerce' ),
					'networkError'         	=> __( 'Network error. Please try again.', 'paypercut-payments-for-woocommerce' ),
					'testConnection'       	=> __( 'Test Connection', 'paypercut-payments-for-woocommerce' ),
				),
			)
		);

		wp_enqueue_script(
			'paypercut-delete-webhook',
			Assets::url( 'admin/js/delete-webhook.js' ),
			array(),
			PAYPERCUT_PAYMENTS_VERSION,
			true
		);

		wp_localize_script(
			'paypercut-delete-webhook',
			'paypercutAdmin',
			array(
				'ajaxurl' => admin_url( 'admin-ajax.php' ),
				'i18n'    => array(
					'missingData'    => __( 'Missing required data.', 'paypercut-payments-for-woocommerce' ),
					'confirmDelete'  => __( 'Are you sure you want to delete this webhook? This action cannot be undone.', 'paypercut-payments-for-woocommerce' ),
					'deleting'       => __( 'Deleting...', 'paypercut-payments-for-woocommerce' ),
					'deleteFailed'   => __( 'Failed to delete webhook.', 'paypercut-payments-for-woocommerce' ),
					'networkError'   => __( 'Network error. Please try again.', 'paypercut-payments-for-woocommerce' ),
				),
			)
		);
	}

	private function enqueue_setup_wizard_assets( string $hook ): void {
		if ( 'woocommerce_page_wc-settings' !== $hook ) {
			return;
		}

		$raw_tab     	= filter_input( INPUT_GET, 'tab', FILTER_UNSAFE_RAW );
		$raw_section 	= filter_input( INPUT_GET, 'section', FILTER_UNSAFE_RAW );
		$tab     		= sanitize_key( is_string( $raw_tab ) ? wp_unslash( $raw_tab ) : '' );
		$section 		= sanitize_key( is_string( $raw_section ) ? wp_unslash( $raw_section ) : '' );

		if ( 'checkout' !== $tab || ! in_array( $section, array( PaypercutGateway::ID, PaypercutBnplGateway::ID ), true ) ) {
			return;
		}

		wp_enqueue_script(
			'paypercut-setup-wizard',
			Assets::url( 'admin/js/setup-wizard.js' ),
			array(),
			PAYPERCUT_PAYMENTS_VERSION,
			false
		);

		$setup_complete = false;
		if ( is_admin() && isset( $_GET['paypercut_setup_complete'] ) ) {
			if ( isset( $_GET['_wpnonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'paypercut_setup_complete' ) ) {
				$setup_complete = '1' === sanitize_text_field( wp_unslash( $_GET['paypercut_setup_complete'] ) );
			}
		}

		wp_localize_script(
			'paypercut-setup-wizard',
			'paypercutSetupWizard',
			array(
				'ajaxurl' 		=> admin_url( 'admin-ajax.php' ),
				'setupComplete' => $setup_complete,
				'i18n'    		=> array(
					'setupComplete'       	=> __( 'Setup complete! Redirecting...', 'paypercut-payments-for-woocommerce' ),
					'errorOccurred'       	=> __( 'An error occurred. Please try again.', 'paypercut-payments-for-woocommerce' ),
					'networkError'        	=> __( 'Network error. Please try again.', 'paypercut-payments-for-woocommerce' ),
					'fieldNotFound'       	=> __( 'API key secret field not found. Please refresh the page.', 'paypercut-payments-for-woocommerce' ),
					'apiSecretRequired'   	=> __( 'API key secret is required.', 'paypercut-payments-for-woocommerce' ),
					'securityTokenMissing' 	=> __( 'Security token missing. Please refresh the page.', 'paypercut-payments-for-woocommerce' ),
				),
			)
		);

		// Check the correct gateway settings based on current section
		$gateway_id = PaypercutBnplGateway::ID === $section
			? PaypercutBnplGateway::ID
			: PaypercutGateway::ID;

		$settings = get_option( 'woocommerce_' . $gateway_id . '_settings', array() );
		$api_secret = $settings['api_client_secret'] ?? '';

		if ( empty( $api_secret ) ) {
			wp_enqueue_style(
				'paypercut-setup-wizard',
				Assets::url( 'admin/css/setup-wizard.css' ),
				array(),
				PAYPERCUT_PAYMENTS_VERSION
			);
		}
	}

	private function enqueue_refund_helper( string $hook ): void {
		$order = $this->resolve_current_order();

		if ( ! $order instanceof WC_Order ) {
			return;
		}

		wp_enqueue_script(
			'paypercut-refund-helper',
			Assets::url( 'admin/js/refund-helper.js' ),
			array(),
			PAYPERCUT_PAYMENTS_VERSION,
			true
		);

		wp_localize_script(
			'paypercut-refund-helper',
			'paypercutRefundHelper',
			array(
				'helperText' 	=> __( 'Paypercut recommends using the literal values duplicate, fraudulent, or requested_by_customer.', 'paypercut-payments-for-woocommerce' ),
				'placeholder' 	=> __( 'Select a Paypercut refund reason', 'paypercut-payments-for-woocommerce' ),
				'options'    	=> array(
					'duplicate' 			=> __( 'Duplicate', 'paypercut-payments-for-woocommerce' ),
					'fraudulent' 			=> __( 'Fraudulent', 'paypercut-payments-for-woocommerce' ),
					'requested_by_customer' => __( 'Requested by customer', 'paypercut-payments-for-woocommerce' ),
				),
				'isPaypercutGateway' => $order->get_payment_method() === PaypercutGateway::ID,
			)
		);
	}

	private function resolve_current_order(): ?WC_Order {
		if ( ! current_user_can( 'edit_shop_orders' ) ) {
			return null;
		}

		$order_id = filter_input( INPUT_GET, 'post', FILTER_VALIDATE_INT );

		if ( ! $order_id ) {
			$order_id = filter_input( INPUT_POST, 'post_ID', FILTER_VALIDATE_INT );
		}

		if ( ! $order_id ) {
			$order_id = filter_input( INPUT_GET, 'id', FILTER_VALIDATE_INT );
		}

		if ( ! $order_id ) {
			return null;
		}

		$order = wc_get_order( (int) $order_id );

		if ( ! $order instanceof WC_Order ) {
			return null;
		}

		if ( ! current_user_can( 'view_order', $order->get_id() ) ) {
			return null;
		}

		return $order;
	}
}
