<?php
/**
 * Thin API client for interacting with Paypercut REST endpoints.
 *
 * @package Paypercut\Payments\Api
 */

declare(strict_types=1);

namespace Paypercut\Payments\Api;

use GuzzleHttp\Client;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\Exception\GuzzleException;
use Paypercut\Payments\Support\Logger;
use RuntimeException;

use function array_key_exists;
use function esc_html;
use function is_array;
use function json_decode;
use function sanitize_text_field;
use function sprintf;
use function wp_json_encode;

/**
 * Handles checkout session requests against Paypercut.
 */
final class PaypercutApi {
	/**
	 * Default Paypercut production base URI.
	 */
	public const DEFAULT_BASE_URI = 'https://api.paypercut.io/';

	/**
	 * BNPL gateway test base URI.
	 */
	public const BNPL_TEST_BASE_URI = 'https://bnpl-gw.bender.paypercut.net/';

	/**
	 * BNPL gateway live base URI.
	 */
	public const BNPL_LIVE_BASE_URI = 'http://api.bnpl.paypercut.io/';

	/**
	 * BNPL gateway base URI (deprecated, use BNPL_TEST_BASE_URI or BNPL_LIVE_BASE_URI).
	 *
	 * @deprecated Use BNPL_TEST_BASE_URI or BNPL_LIVE_BASE_URI instead.
	 */
	public const BNPL_BASE_URI = self::BNPL_TEST_BASE_URI;

	/**
	 * HTTP client.
	 *
	 * @var ClientInterface
	 */
	private ClientInterface $http_client;

	/**
	 * Paypercut API key secret.
	 *
	 * @var string
	 */
	private string $client_secret;

	/**
	 * Construct the API wrapper.
	 *
	 * @param string               $client_secret Paypercut API key secret.
	 * @param ClientInterface|null $http_client   Optional HTTP client for testing/DI.
	 * @param string|null          $base_uri      Optional base URI override.
	 */
	public function __construct( string $client_secret, ?ClientInterface $http_client = null, ?string $base_uri = null ) {
		$this->client_secret = $client_secret;
		$this->http_client   = $http_client ?? new Client(
			array(
				'base_uri' => $base_uri ?? self::DEFAULT_BASE_URI,
				'timeout'  => 10,
			)
		);
	}

	/**
	 * Create a checkout session using the stored API secret as bearer token.
	 *
	 * @param array<string, mixed> $payload Request body following Paypercut schema.
	 * @param array<string, mixed> $options Additional request options (idempotency keys etc.).
	 *
	 * @throws RuntimeException When the API returns an error.
	 *
	 * @return array<string, mixed>
	 */
	public function create_checkout_session( array $payload, array $options = array() ): array {
		return $this->post_json( 'v1/checkouts', $payload, $options, 'checkout session' );
	}

	/**
	 * Create a BNPL attempt using the stored API secret as bearer token.
	 *
	 * @param array<string, mixed> $payload Request body following Paypercut BNPL schema.
	 * @param array<string, mixed> $options Additional request options (idempotency keys etc.).
	 *
	 * @throws RuntimeException When the API returns an error.
	 *
	 * @return array<string, mixed>
	 */
	public function create_bnpl_attempt( array $payload, array $options = array() ): array {
		return $this->post_json( 'v1/bnpl/attempt', $payload, $options, 'BNPL attempt' );
	}

	public function create_refund( array $payload, array $options = array() ): array {
		return $this->post_json( 'v1/refunds', $payload, $options, 'refund' );
	}

	/**
	 * Validate API secret by making a test request.
	 *
	 * @throws RuntimeException When the API secret is invalid.
	 *
	 * @return array<string, mixed>
	 */
	public function validate_credentials(): array {
		return $this->get_json( 'v1/account', 'credentials validation' );
	}

	/**
	 * Validate BNPL API secret by making a test request to /v1/bnpl/attempt.
	 * 401 = invalid credentials, 400 = valid credentials.
	 *
	 * @throws RuntimeException When the API secret is invalid (401 response) or network error occurs.
	 *
	 * @return bool True if credentials are valid.
	 */
	public function validate_bnpl_credentials(): bool {
		try {
			$response = $this->http_client->request(
				'POST',
				'v1/bnpl/attempt',
				array(
					'headers' => array(
						'Authorization' => 'Bearer ' . $this->client_secret,
						'Content-Type'  => 'application/json',
					),
					'body'    => wp_json_encode( (object) array() ),
				)
			);

			// Unexpected success response
			throw new RuntimeException( 'Unexpected response from BNPL API during validation.' );
		} catch ( ClientException $exception ) {
			$response = $exception->getResponse();
			$status_code = $response ? $response->getStatusCode() : 0;

			// 401 = invalid credentials
			if ( 401 === $status_code ) {
				throw new RuntimeException( 'Invalid BNPL API credentials (401 Unauthorized).' );
			}

			// 400 = valid credentials
			if ( 400 === $status_code ) {
				return true;
			}

			throw new RuntimeException(
				sprintf( 'Unexpected response from BNPL API: HTTP %d', $status_code )
			);
		} catch ( GuzzleException $exception ) {
			throw new RuntimeException(
				sprintf( 'Unable to connect to BNPL API: %s', $exception->getMessage() )
			);
		}
	}

	/**
	 * Create a webhook using the provided payload.
	 *
	 * @param array<string, mixed> $payload Webhook creation payload with name, url, and enabled_events.
	 *
	 * @throws RuntimeException When the webhook creation fails.
	 *
	 * @return array<string, mixed>
	 */
	public function create_webhook( array $payload ): array {
		return $this->post_json( 'v1/webhooks', $payload, array(), 'webhook' );
	}

	/**
	 * Retrieve all webhooks for the account.
	 *
	 * @throws RuntimeException When the webhook retrieval fails.
	 *
	 * @return array<string, mixed>
	 */
	public function get_webhooks(): array {
		return $this->get_json( 'v1/webhooks', 'webhook listing' );
	}

	/**
	 * Delete a webhook by ID.
	 *
	 * @param string $webhook_id The webhook ID to delete.
	 *
	 * @throws RuntimeException When the webhook deletion fails.
	 *
	 * @return array<string, mixed>
	 */
	public function delete_webhook( string $webhook_id ): array {
		return $this->delete_json( 'v1/webhooks/' . $webhook_id, 'webhook deletion' );
	}

	/**
	 * Make a DELETE request to the API.
	 *
	 * @param string $path    API endpoint path.
	 * @param string $context Context for error messages.
	 *
	 * @throws RuntimeException When the API returns an error.
	 *
	 * @return array<string, mixed>
	 */
	private function delete_json( string $path, string $context ): array {
		$headers = array(
			'Authorization' => 'Bearer ' . $this->client_secret,
			'Content-Type'  => 'application/json',
		);

		Logger::debug(
			sprintf( 'API Request: DELETE %s', $path ),
			array(
				'method'  => 'DELETE',
				'path'    => $path,
				'headers' => $this->sanitize_headers_for_logging( $headers ),
			)
		);

		try {
			$response = $this->http_client->request(
				'DELETE',
				ltrim( $path, '/' ),
				array(
					'headers' => $headers,
				)
			);

			$response_body = (string) $response->getBody();
			$response_data = json_decode( $response_body, true );

			Logger::debug(
				sprintf( 'API Response: DELETE %s', $path ),
				array(
					'status_code' => $response->getStatusCode(),
					'body'        => $this->sanitize_response_body_for_logging( $response_data ?: $response_body, $path ),
				)
			);

			if ( empty( $response_body ) || $response->getStatusCode() === 204 ) {
				return array( 'deleted' => true );
			}

			return $this->decode_response_body( $response_body );
		} catch ( GuzzleException $exception ) {
			Logger::error( sprintf( 'Failed %s: %s', $context, $exception->getMessage() ) );
			throw new RuntimeException( 'Unable to complete Paypercut API request.' );
		}
	}

	/**
	 * Retrieve a checkout session by ID.
	 *
	 * @param string               $session_id The checkout session ID.
	 * @param array<string, mixed> $params     Optional query parameters (e.g., expand).
	 *
	 * @throws RuntimeException When the session retrieval fails.
	 *
	 * @return array<string, mixed>
	 */
	public function get_checkout_session( string $session_id, array $params = array() ): array {
		$path = 'v1/checkouts/' . $session_id;

		if ( ! empty( $params ) ) {
			$query_string = http_build_query( $params );
			$path .= '?' . $query_string;
		}

		return $this->get_json( $path, 'checkout session retrieval' );
	}

	/**
	 * Retrieve BNPL attempt status by attempt ID.
	 *
	 * @param string $attempt_id The BNPL attempt ID.
	 *
	 * @throws RuntimeException When the attempt retrieval fails.
	 *
	 * @return array<string, mixed>
	 */
	public function get_bnpl_attempt_status( string $attempt_id ): array {
		$path = 'v1/bnpl/attempt/' . rawurlencode( $attempt_id ) . '/status';

		return $this->get_json( $path, 'BNPL attempt status retrieval' );
	}


	/**
	 * Make a GET request to the API.
	 *
	 * @param string $path    API endpoint path.
	 * @param string $context Context for error messages.
	 *
	 * @throws RuntimeException When the API returns an error.
	 *
	 * @return array<string, mixed>
	 */
	private function get_json( string $path, string $context ): array {
		$headers = array(
			'Authorization' => 'Bearer ' . $this->client_secret,
			'Content-Type'  => 'application/json',
		);

		Logger::debug(
			sprintf( 'API Request: GET %s', $path ),
			array(
				'method'  => 'GET',
				'path'    => $path,
				'headers' => $this->sanitize_headers_for_logging( $headers ),
			)
		);

		try {
			$response = $this->http_client->request(
				'GET',
				ltrim( $path, '/' ),
				array(
					'headers' => $headers,
				)
			);

			$response_body = (string) $response->getBody();
			$response_data = json_decode( $response_body, true );

			Logger::debug(
				sprintf( 'API Response: GET %s', $path ),
				array(
					'status_code' => $response->getStatusCode(),
					'body'        => $this->sanitize_response_body_for_logging( $response_data ?: $response_body, $path ),
				)
			);

			return $this->decode_response_body( $response_body );
		} catch ( GuzzleException $exception ) {
			Logger::error( sprintf( 'Failed %s: %s', $context, $exception->getMessage() ) );
			throw new RuntimeException( 'Unable to complete Paypercut API request.' );
		}
	}

	private function post_json( string $path, array $payload, array $options, string $context ): array {
		$headers = array(
			'Authorization' => 'Bearer ' . $this->client_secret,
			'Content-Type'  => 'application/json',
		);

		if ( isset( $options['idempotency_key'] ) ) {
			$headers['Idempotency-Key'] = (string) $options['idempotency_key'];
		}

		$request_body = wp_json_encode( $payload );

		Logger::debug(
			sprintf( 'API Request: POST %s', $path ),
			array(
				'method'  => 'POST',
				'path'    => $path,
				'headers' => $this->sanitize_headers_for_logging( $headers ),
				'body'    => $payload,
			)
		);

		try {
			$response = $this->http_client->request(
				'POST',
				ltrim( $path, '/' ),
				array(
					'headers' => $headers,
					'body'    => $request_body,
				)
			);

			$response_body = (string) $response->getBody();
			$response_data = json_decode( $response_body, true );

			Logger::debug(
				sprintf( 'API Response: POST %s', $path ),
				array(
					'status_code' => $response->getStatusCode(),
					'body'        => $this->sanitize_response_body_for_logging( $response_data ?: $response_body, $path ),
				)
			);

			return $this->decode_response_body( $response_body );
		} catch ( GuzzleException $exception ) {
			Logger::error( sprintf( 'Failed creating Paypercut %s: %s', $context, $exception->getMessage() ) );
			throw new RuntimeException( 'Unable to complete Paypercut API request.' );
		}
	}

	/**
	 * Decode an HTTP response body and normalise JSON errors.
	 *
	 * @param string $response_body Response body as string.
	 *
	 * @throws RuntimeException When the response body is not valid JSON or contains an error.
	 *
	 * @return array<string, mixed>
	 */
	private function decode_response_body( string $response_body ): array {
		$data = json_decode( $response_body, true );

		if ( ! is_array( $data ) ) {
			throw new RuntimeException( 'Unexpected response from Paypercut API.' );
		}

		if ( array_key_exists( 'status_code', $data ) && $data['status_code'] >= 400 ) {
			$error_message = $data['message'] ?? 'Unknown error';
			Logger::error( sprintf( 'Paypercut API returned an error: %s', $error_message ) );
			throw new RuntimeException( 'Paypercut API returned an error.' );
		}

		return $data;
	}

	/**
	 * Sanitize headers for logging by redacting sensitive information.
	 *
	 * @param array<string, string> $headers Headers to sanitize.
	 *
	 * @return array<string, string>
	 */
	private function sanitize_headers_for_logging( array $headers ): array {
		$sanitized = $headers;

		if ( isset( $sanitized['Authorization'] ) ) {
			$sanitized['Authorization'] = 'Bearer [REDACTED]';
		}

		return $sanitized;
	}

	/**
	 * Sanitize response body for logging by redacting sensitive fields like secrets.
	 *
	 * @param array<string, mixed>|string $body Response body to sanitize.
	 * @param string                       $path API endpoint path.
	 *
	 * @return array<string, mixed>|string
	 */
	private function sanitize_response_body_for_logging( $body, string $path ) {
		if ( ! is_array( $body ) ) {
			return $body;
		}

		$sanitized = $body;

		// Redact 'secret' field in webhook responses
		if ( strpos( $path, 'webhooks' ) !== false ) {
			if ( isset( $sanitized['secret'] ) && ! empty( $sanitized['secret'] ) ) {
				$sanitized['secret'] = '[REDACTED]';
			}

			// Handle webhook lists with items array
			if ( isset( $sanitized['items'] ) && is_array( $sanitized['items'] ) ) {
				foreach ( $sanitized['items'] as $key => $item ) {
					if ( is_array( $item ) && isset( $item['secret'] ) && ! empty( $item['secret'] ) ) {
						$sanitized['items'][ $key ]['secret'] = '[REDACTED]';
					}
				}
			}
		}

		return $sanitized;
	}
}
