<?php
/**
 * Setup wizard for initial Paypercut configuration.
 *
 * @package Paypercut\Payments\Admin
 */

declare(strict_types=1);

namespace Paypercut\Payments\Admin;

use Paypercut\Payments\Support\Assets;
use function esc_attr;
use function esc_url;
use function filter_input;
use function sanitize_key;
use function wp_create_nonce;
use function wp_unslash;

final class SetupWizard {

	/**
	 * Render the setup wizard form.
	 */
	public function render(): void {
		$banner_large = Assets::url( 'images/banner-1544x500.png' );
		$banner_small = Assets::url( 'images/banner-772x250.png' );
		$icon         = Assets::url( 'images/icon-128x128.png' );
		$nonce        = wp_create_nonce( 'paypercut_setup_wizard' );

		// Get the current gateway section from URL
		$raw_section = filter_input( INPUT_GET, 'section', FILTER_UNSAFE_RAW );
		$section     = sanitize_key( is_string( $raw_section ) ? wp_unslash( $raw_section ) : '' );
		
		// Check if this is BNPL gateway
		$is_bnpl = 'paypercut_bnpl' === $section;

		?>
		<div class="paypercut-setup-wizard">
			<div class="paypercut-setup-wizard__hero">
				<picture>
					<source media="(max-width: 782px)" srcset="<?php echo esc_url( $banner_small ); ?>">
					<img src="<?php echo esc_url( $banner_large ); ?>" alt="<?php esc_attr_e( 'Paypercut banner', 'paypercut-payments-for-woocommerce' ); ?>">
				</picture>
			</div>
			<div class="paypercut-setup-wizard__container">
				<div class="paypercut-setup-wizard__header">
					<img src="<?php echo esc_url( $icon ); ?>" alt="<?php esc_attr_e( 'Paypercut logo', 'paypercut-payments-for-woocommerce' ); ?>">
					<div>
						<h2><?php esc_html_e( 'Welcome to Paypercut Payments', 'paypercut-payments-for-woocommerce' ); ?></h2>
						<p><?php esc_html_e( 'Let\'s get you set up in just a few steps. Enter your API key secret to automatically configure webhooks and enable the gateway.', 'paypercut-payments-for-woocommerce' ); ?></p>
					</div>
				</div>

				<div id="paypercut-setup-form" class="paypercut-setup-wizard__form">
					<input type="hidden" id="paypercut-nonce" value="<?php echo esc_attr( $nonce ); ?>">
					<input type="hidden" id="paypercut-gateway-section" name="gateway_section" value="<?php echo esc_attr( $section ); ?>">
					
					<?php if ( $is_bnpl ) : ?>
					<div class="paypercut-setup-wizard__field">
						<label for="mode">
							<?php esc_html_e( 'Mode', 'paypercut-payments-for-woocommerce' ); ?>
							<span class="required">*</span>
						</label>
						<select id="mode" name="mode" class="regular-text" required>
							<option value="test"><?php esc_html_e( 'Test', 'paypercut-payments-for-woocommerce' ); ?></option>
							<option value="live"><?php esc_html_e( 'Live', 'paypercut-payments-for-woocommerce' ); ?></option>
						</select>
						<p class="description">
							<?php esc_html_e( 'Select Live for production or Test for development.', 'paypercut-payments-for-woocommerce' ); ?>
						</p>
					</div>
					<?php endif; ?>
					
					<div class="paypercut-setup-wizard__field">
						<label for="api_client_secret">
							<?php esc_html_e( 'API Key Secret', 'paypercut-payments-for-woocommerce' ); ?>
							<span class="required">*</span>
						</label>
						<input 
							type="password" 
							id="api_client_secret" 
							name="api_client_secret" 
							class="regular-text" 
							required
							placeholder="<?php esc_attr_e( 'Enter your Paypercut API key secret', 'paypercut-payments-for-woocommerce' ); ?>"
						>
						<p class="description">
							<?php esc_html_e( 'Your API key secret from your Paypercut dashboard.', 'paypercut-payments-for-woocommerce' ); ?>
						</p>
					</div>

					<div id="paypercut-setup-messages" class="paypercut-setup-wizard__messages" style="display: none;"></div>

					<div class="paypercut-setup-wizard__actions">
						<button type="button" id="paypercut-setup-submit" class="button button-primary button-large">
							<span class="button-text"><?php esc_html_e( 'Connect & Setup', 'paypercut-payments-for-woocommerce' ); ?></span>
							<span class="button-spinner" style="display: none;"><?php esc_html_e( 'Setting up...', 'paypercut-payments-for-woocommerce' ); ?></span>
						</button>
					</div>
				</div>
			</div>
		</div>
		<?php
	}
}
