/**
 * Paypercut Checkout Controller – classic (shortcode) checkout.
 *
 * Responsibilities in this file:
 * - Watch which payment method is selected on the classic checkout page.
 * - Create / refresh Paypercut checkout sessions via admin‑ajax.php.
 * - Render the Paypercut iframe and keep it in sync with the cart.
 * - Run a very small amount of client‑side validation before we ask Paypercut to charge.
 * - Once Paypercut reports a successful payment, submit the WooCommerce form so the order is created.
 *
 * The important detail is that WooCommerce classic checkout is still a normal <form>
 * submission. This controller never creates orders directly – it only decides
 * when it is safe to let WooCommerce continue with its own form POST.
 */

(function () {
	'use strict';

	// Headers used for all session create calls against admin‑ajax.php.
	const DEFAULT_HEADERS = {
		'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8',
		Accept: 'application/json',
	};

	// Name of the payment method radio field used by WooCommerce.
	const PAYMENT_METHOD_NAME = 'payment_method';

	// AJAX action used on the PHP side to create a new Paypercut session.
	const AJAX_ACTION = 'paypercut_create_checkout_session';

	const DEFAULT_APPEARANCE = Object.freeze({
		preset: 'inline',
		theme: 'light',
		labels: 'above',
	});

	const TRANSACTION_FIELD_SELECTORS = Object.freeze({
		session: 'input[name="paypercut_session_id"]',
		payment: 'input[name="paypercut_payment_id"]',
		paymentIntent: 'input[name="paypercut_payment_intent_id"]',
	});

	/**
	 * Main controller used on classic checkout / add payment method screens.
	 *
	 * This class does not know anything about WooCommerce internals – it works
	 * entirely through DOM events, admin‑ajax and the Paypercut SDK.
	 */
	class PaypercutCheckoutController {
		constructor(config) {
			this.config = {
				gatewayId: 'paypercut_payments',
				containerId: '#paypercut-checkout-container',
				ajaxUrl: '',
				nonce: '',
				loadingMessage: 'Setting up secure checkout…',
				errorMessage: 'Something went wrong while loading Paypercut. Please try again.',
				...config,
			};
			const appearanceConfig =
				typeof config?.appearance === 'object' && config.appearance !== null
					? config.appearance
					: {};
			this.config.appearance = { ...DEFAULT_APPEARANCE, ...appearanceConfig };

			this.config.context = typeof config?.context === 'string' ? config.context : 'checkout';
			this.config.addPaymentFormSelector =
				config?.addPaymentFormSelector || '#add_payment_method';
			this.config.addPaymentButtonSelector =
				config?.addPaymentButtonSelector || '#add_payment_method button[type="submit"]';

			// Runtime flags and state.
			this.checkoutInstance = null;   // Active PaypercutCheckout instance.
			this.currentSessionId = null;   // ID of the currently rendered Paypercut session.
			this.pendingRequest = null;     // In‑flight AJAX request for a new session.
			this.isCheckoutLoaded = false;  // Becomes true once the iframe has finished loading.
			this.isProcessing = false;      // Guards against double‑submits.
			this.paymentSuccess = false;    // Tracks whether the last Paypercut payment succeeded.
			this.allowNativeSubmit = false; // When true we let WooCommerce submit the form itself.
		}

		// Public entry point. Called once on DOMContentLoaded.
		init() {
			this.setupEventListeners();
			this.checkAndRender();
		}

		/**
		 * Wire up all high‑level listeners needed for the checkout page.
		 *
		 * The idea here is:
		 * - When Paypercut is selected we ensure a session exists and the iframe is rendered.
		 * - When totals change we refresh the session.
		 * - When the user clicks "Place order", we decide whether the WooCommerce form
		 *   may submit or we have to run the Paypercut flow first.
		 */
		setupEventListeners() {
		this.setupPaymentMethodListener();

		if (this.config.context === 'add_payment_method') {
			this.setupAddPaymentMethodListener();
			return;
		}

		this.setupCheckoutUpdateListeners();
		this.setupClassicCheckoutValidation();
	}

		/**
		 * Watch for changes to the payment method radio group and react when
		 * Paypercut is selected or deselected.
		 */
		setupPaymentMethodListener() {
			document.addEventListener('change', (event) => {
				const target = event.target;

				if (!(target instanceof HTMLInputElement) || target.type !== 'radio') {
					return;
				}

				if (target.name !== PAYMENT_METHOD_NAME) {
					return;
				}

				const selectedMethod = target.value;
				if (selectedMethod) {
					this.handlePaymentMethodChange(selectedMethod);
				}
			});
		}

		/**
		 * When WooCommerce recalculates the checkout (shipping, coupons, etc.),
		 * we need to recreate the Paypercut session so that the amount and cart
		 * contents match.
		 */
		setupCheckoutUpdateListeners() {
		if (!window.jQuery) {
			return;
		}

		window.jQuery(document.body).on('updated_checkout', () => {
			if (this.isPaypercutSelected()) {
				this.refreshSession();
			}
		});
	}

		/**
		 * Special handling for the "Add payment method" screen in My Account.
		 *
		 * The flow here is almost identical to the normal checkout, but it runs
		 * inside a different form and button selectors.
		 */
		setupAddPaymentMethodListener() {
			const formSelector = this.config.addPaymentFormSelector || '#add_payment_method';
			const buttonSelector =
				this.config.addPaymentButtonSelector || '#add_payment_method button[type="submit"]';

			document.addEventListener(
				'submit',
				(event) => {
					if (!(event.target instanceof HTMLFormElement) || !event.target.matches(formSelector) || !this.isPaypercutSelected()) {
						return;
					}

					if (this.shouldAllowNativeSubmit()) {
						return;
					}

					if (!this.isCheckoutFormValid()) {
						return;
					}

					event.preventDefault();
					event.stopPropagation();
					this.triggerCheckoutSubmit();
				},
				true
			);

			document.addEventListener(
				'click',
				(event) => {
					if (!this.isPaypercutSelected()) {
						return;
					}

					const button =
						event.target instanceof HTMLElement ? event.target.closest(buttonSelector) : null;

					if (!button) {
						return;
					}

					const form = button.closest(formSelector);

					if (!form) {
						return;
					}

					if (this.shouldAllowNativeSubmit()) {
						return;
					}

					event.preventDefault();
					event.stopPropagation();
					this.triggerCheckoutSubmit();
				},
				true
			);
		}

		/**
		 * Hook into WooCommerce's classic checkout JS so we can run our own
		 * validation + Paypercut flow before the form is actually submitted.
		 *
		 * We deliberately keep this logic small and rely on WooCommerce to
		 * display its own per‑field error messages.
		 */
		setupClassicCheckoutValidation() {
		const $ = window.jQuery;
		if (!$ || !$.fn) {
			return;
		}

		const self = this;

		if (this.config.context === 'checkout') {
			$('form.checkout').on('checkout_place_order', function(e, data) {
				if (!self.isPaypercutSelected()) {
					return true;
				}
				
				if (self.shouldAllowNativeSubmit()) {
					return true;
				}

				if (self.isProcessing && !self.paymentSuccess) {
					return false;
				}

				if (data && data.skipPaypercutHandler) {
					return true;
				}

				if (!self.isCheckoutFormValid()) {
					return true;
				}
				
				self.triggerCheckoutSubmit();
				return false;
			});
		}

		if (this.config.orderId) {
			const orderPayForm = document.querySelector('form#order_review');
			
			if (orderPayForm) {
				orderPayForm.addEventListener('submit', function(e) {
					if (!self.isPaypercutSelected()) {
						return true;
					}

					if (self.shouldAllowNativeSubmit()) {
						return true;
					}

					e.preventDefault();
					e.stopImmediatePropagation();

					if (self.isProcessing && !self.paymentSuccess) {
						return false;
					}

					if (!self.isOrderPayFormValid(orderPayForm)) {
						return false;
					}

					if (!self.checkoutInstance || !self.isCheckoutLoaded) {
						self.showError(self.config.errorMessage || 'Paypercut checkout is not ready. Please try again.');
						return false;
					}

					const submitResult = self.triggerCheckoutSubmit();
					
					if (submitResult) {
						return false;
					}

					self.showError(self.config.errorMessage || 'Unable to process payment. Please try again.');
					return false;
				}, true);
			}
		}
	}

		/**
		 * Simple, deterministic validation used by both "Place order" and
		 * Paypercut wallet flows.
		 *
		 * Rules:
		 * - If WooCommerce already marked any field aria‑invalid="true", we
		 *   treat the form as invalid.
		 * - For each .validate-required row we check the corresponding field:
		 *   - required checkboxes/radios must be checked;
		 *   - other inputs/selects/textareas must have a non‑empty value.
		 *
		 * This mirrors WooCommerce's behaviour without copying every rule.
		 */
		isCheckoutFormValid() {
		const form = document.querySelector('form.checkout');
		if (!form) {
			return false;
		}

		const invalidFields = form.querySelectorAll('input[aria-invalid="true"], select[aria-invalid="true"], textarea[aria-invalid="true"]');
		if (invalidFields.length > 0) {
			return false;
		}

		const requiredRows = form.querySelectorAll('.validate-required');
		for (const row of requiredRows) {
			const field = row.querySelector('input, select, textarea');
			if (!field || field.disabled) {
				continue;
			}

			if (field instanceof HTMLInputElement) {
				if ((field.type === 'checkbox' || field.type === 'radio') && !field.checked) {
					return false;
				}
				if (field.type !== 'checkbox' && field.type !== 'radio' && field.type !== 'hidden' && !field.value.trim()) {
					return false;
				}
			} else if (field instanceof HTMLSelectElement || field instanceof HTMLTextAreaElement) {
				if (!field.value.trim()) {
					return false;
				}
			}
		}

		return true;
	}

		/**
		 * Validate order pay form (similar to checkout form but for order pay page).
		 *
		 * @param {HTMLFormElement} form The order pay form element.
		 * @return {boolean} True if form is valid, false otherwise.
		 */
		isOrderPayFormValid(form) {
		if (!form) {
			return false;
		}

		const requiredRows = form.querySelectorAll('.validate-required');
		
		for (const row of requiredRows) {
			let field = row.querySelector('input, select, textarea');
			
			if (!field) {
				field = row.querySelector('input[type="checkbox"]');
			}
			
			if (!field) {
				field = row.querySelector('input[name="terms"]');
			}
			
			if (!field) {
				continue;
			}
			
			if (field.disabled) {
				continue;
			}

			if (field instanceof HTMLInputElement) {
				if ((field.type === 'checkbox' || field.type === 'radio') && !field.checked) {
					this.showOrderPayValidationError(field);
					return false;
				}
				if (field.type !== 'checkbox' && field.type !== 'radio' && field.type !== 'hidden' && !field.value.trim()) {
					this.showOrderPayValidationError(field);
					return false;
				}
			} else if (field instanceof HTMLSelectElement || field instanceof HTMLTextAreaElement) {
				if (!field.value.trim()) {
					this.showOrderPayValidationError(field);
					return false;
				}
			}
		}

		return true;
	}

		/**
		 * Show validation error for order pay form fields.
		 *
		 * @param {HTMLElement} field The invalid field element.
		 */
		showOrderPayValidationError(field) {
		const $ = window.jQuery;
		if (!$ || !$.fn) {
			const row = field.closest('.form-row, .woocommerce-form-row');
			if (row) {
				row.classList.add('woocommerce-invalid');
				field.setAttribute('aria-invalid', 'true');
				
				field.scrollIntoView({ behavior: 'smooth', block: 'center' });
				field.focus();
			}
			return;
		}

		const row = field.closest('.form-row, .woocommerce-form-row');
		if (row) {
			$(row).addClass('woocommerce-invalid');
			field.setAttribute('aria-invalid', 'true');
			
			$('html, body').animate({
				scrollTop: $(field).offset().top - 100
			}, 500);
			field.focus();
			
			$(document.body).trigger('checkout_error', [{
				message: this.config.validationErrorMessage || 'Please fill all required fields.'
			}]);
		}
	}

		/**
		 * Show a single generic WooCommerce error notice using the message
		 * provided by the PHP side.
		 *
		 * This is used mainly for wallet payments – standard checkout will
		 * usually have its own per‑field notices as well.
		 */
		showValidationErrorMessage() {
		const $ = window.jQuery;
		if (!$ || !$.fn) {
			return;
		}

		const errorMessage = this.config.validationErrorMessage;
		if (!errorMessage) {
			return;
		}
		
		$('.woocommerce-NoticeGroup-checkout, .woocommerce-error').remove();

		const $list = $('<ul class="woocommerce-error" tabindex="-1">');
		$list.append($('<li>').text(errorMessage));

		const $alert = $('<div role="alert">').html($list);
		const $notice = $('<div class="woocommerce-NoticeGroup woocommerce-NoticeGroup-checkout">').html($alert);

		const $form = $('form.checkout');
		if ($form.length) {
			$form.prepend($notice);
			$('html, body').animate({
				scrollTop: $notice.offset().top - 100
			}, 500);
		}

		$(document.body).trigger('checkout_error', [{ message: errorMessage }]);
	}


		/**
		 * Ask the Paypercut SDK to process the payment for the current session.
		 *
		 * We only call this once we know the WooCommerce form is valid. If the
		 * SDK reports success, submitWooCommerceForm() will be called and the
		 * normal WooCommerce checkout flow continues.
		 */
		triggerCheckoutSubmit() {
		if (!this.checkoutInstance) {
			return false;
		}

		const isMounted = typeof this.checkoutInstance.isMounted === 'function' ? this.checkoutInstance.isMounted() : false;
		
		if (typeof this.checkoutInstance.isMounted === 'function' && !isMounted) {
			return false;
		}

		if (!this.isCheckoutLoaded) {
			return false;
		}

		this.paymentSuccess = false;
		this.blockProcessingUI();

		try {
			this.checkoutInstance.submit();
			return true;
		} catch (error) {
			this.paymentSuccess = false;
			this.unblockProcessingUI();
			return false;
		}
	}

		/**
		 * Put a light overlay over the page while the Paypercut SDK is running.
		 * This is purely cosmetic – it prevents accidental double‑clicks and
		 * gives the customer a clear "please wait" signal.
		 */
		blockProcessingUI() {
		if (this.isProcessing) {
			return;
		}

		this.isProcessing = true;

		const $ = window.jQuery;
		if ($?.fn?.block) {
			$(document.body).block({
				message: null,
				overlayCSS: {
					background: '#fff',
					opacity: 0.6,
					cursor: 'wait',
				},
			});
		}
	}

		/**
		 * Remove the overlay created by blockProcessingUI().
		 */
		unblockProcessingUI() {
		this.isProcessing = false;

		const $ = window.jQuery;
		if ($?.fn?.unblock) {
			$(document.body).unblock();
		}
	}

		/**
		 * Called every time WooCommerce switches the selected payment method.
		 * If the user chooses Paypercut we make sure a session exists; otherwise
		 * we tear down any existing iframe.
		 */
		handlePaymentMethodChange(selectedMethod) {
		if (selectedMethod === this.config.gatewayId) {
			this.checkAndRender();
		} else {
			this.destroy();
		}
	}

		/**
		 * Simple wrapper used when WooCommerce recalculates totals.
		 */
		refreshSession() {
		this.createSession(true);
	}

		/**
		 * Create or refresh a Paypercut session via admin‑ajax.php.
		 *
		 * The PHP handler is responsible for:
		 * - Building the Paypercut checkout session on the server.
		 * - Returning session/payment identifiers that we store in hidden fields.
		 */
		createSession(force = false) {
			if (!this.config.ajaxUrl || !this.config.nonce) {
				return;
			}

			if (this.pendingRequest && !force) {
				return;
			}

			const params = new URLSearchParams();
			params.append('action', AJAX_ACTION);
			params.append('nonce', this.config.nonce);
			params.append('context', this.config.ajaxContext);
			
			if (this.config.orderId) {
				params.append('order_id', this.config.orderId);
			}

			this.pendingRequest = fetch(this.config.ajaxUrl, {
				method: 'POST',
				headers: DEFAULT_HEADERS,
				credentials: 'same-origin',
				body: params.toString(),
			})
				.then((response) => {
					if (!response.ok) {
						throw new Error(`HTTP ${response.status}: ${response.statusText}`);
					}
					return response.json();
				})
				.then((data) => {
					if (data?.success && data?.data?.sessionId) {
						const payload = data.data;
						if (payload.paymentId) {
							this.setHiddenFieldValue('payment', payload.paymentId);
						}
						if (payload.paymentIntentId) {
							this.setHiddenFieldValue('paymentIntent', payload.paymentIntentId);
						}
						this.setSession(payload.sessionId);
					} else {
						const errorMessage = data?.data?.message || this.config.errorMessage;
						this.showError(errorMessage);
					}
				})
				.catch((err) => {
					this.showError(this.config.errorMessage);
				})
				.finally(() => {
					this.pendingRequest = null;
				});
		}

		/**
		 * Store the new session ID and trigger a re‑render of the iframe.
		 */
		setSession(sessionId) {
			if (!sessionId || sessionId === this.currentSessionId) {
				return;
			}

			this.destroy();
			this.currentSessionId = sessionId;
			this.setHiddenFieldValue('session', sessionId);

			this.checkAndRender();
		}

		/**
		 * Decide whether the Paypercut iframe should be rendered right now,
		 * and if so which session should be used.
		 */
		checkAndRender() {
		if (!this.isPaypercutSelected()) {
			return;
		}

		const container = this.getContainer();
		if (!container) {
			return;
		}

		if (!this.isSdkAvailable()) {
			this.showError(this.config.errorMessage);
			return;
		}

		if (this.currentSessionId) {
			this.renderCheckout();
			return;
		}

		this.showLoading();
		this.createSession();
	}

		/**
		 * Create a new Paypercut SDK instance and render the iframe.
		 *
		 * The SDK takes over everything related to card data / wallet handling.
		 * We only listen for a handful of events to keep WooCommerce in sync.
		 */
		renderCheckout() {
			if (this.checkoutInstance) {
				this.destroy();
			}

			if (!this.currentSessionId) {
				return;
			}

			const container = this.getContainer();
			if (!container) {
				return;
			}

			container.setAttribute('aria-busy', 'true');
			this.clearContainer();

			try {
				this.checkoutInstance = window.PaypercutCheckout({
					id: this.currentSessionId,
					containerId: this.config.containerId,
					form_only: true,
					appearance: this.config.appearance,
					validate_form: true
				});

		this.checkoutInstance.on('success', (data) => {
			this.paymentSuccess = true;
			this.unblockProcessingUI();
			this.persistSuccessPayload(data);
			this.submitWooCommerceForm();
		});
		this.checkoutInstance.on('error', (error) => {
			this.paymentSuccess = false;
			this.unblockProcessingUI();
		});
		this.checkoutInstance.on('loaded', () => {
			this.isCheckoutLoaded = true;
			const containerEl = this.getContainer();
			if (containerEl) {
				containerEl.setAttribute('aria-busy', 'false');
			}
		});

			this.setupThreeDSListeners();
			this.setupWalletFormValidation();

			this.checkoutInstance.render();
			} catch (err) {
				this.showError(this.config.errorMessage);
			}
		}

		/**
		 * Wire up the optional 3‑D Secure callbacks from the Paypercut SDK.
		 * At the moment we only toggle the processing overlay.
		 */
		setupThreeDSListeners() {
		if (!this.checkoutInstance) {
			return;
		}

		this.checkoutInstance.on('threeds_started', () => {
			this.unblockProcessingUI();
		});

		this.checkoutInstance.on('threeds_complete', () => {
			this.blockProcessingUI();
		});

		this.checkoutInstance.on('threeds_canceled', () => {
			this.unblockProcessingUI();
		});

		this.checkoutInstance.on('threeds_error', () => {
			this.unblockProcessingUI();
		});
	}

		/**
		 * Integrate the Paypercut wallet flow ("Pay" buttons rendered by the
		 * SDK) with WooCommerce's notion of a valid checkout form.
		 *
		 * The SDK will call this handler before it talks to the wallet
		 * provider – we run the same validation that the normal checkout uses
		 * and either allow the wallet flow to continue or block it with a
		 * generic error message.
		 */
		setupWalletFormValidation() {
		if (!this.checkoutInstance) {
			return;
		}

		if (this.config.context !== 'checkout') {
			return;
		}

		this.checkoutInstance.on('form_validation', ({ wallet }) => {
			if (this.isCheckoutFormValid()) {
				this.checkoutInstance.completeFormValidation(wallet);
			} else {
				this.showValidationErrorMessage();
				this.checkoutInstance.failFormValidation(wallet, [{ code: 'validation_failed', message: this.config.validationErrorMessage || 'Please fill all required fields.' }]);
			}
		});
	}

		/**
		 * Tear down the current Paypercut instance and reset all local state.
		 * Called when the gateway is deselected or the session changes.
		 */
		destroy() {
			if (this.checkoutInstance?.destroy) {
				this.checkoutInstance.destroy();
			}

			this.checkoutInstance = null;
			this.isCheckoutLoaded = false;
			this.currentSessionId = null;
			this.clearTransactionFields();
			this.clearContainer();
			this.unblockProcessingUI();
		}

		/**
		 * Render a very small placeholder while we are waiting for the first
		 * session to be created.
		 */
		showLoading() {
			const container = this.getContainer();
			if (!container) {
				return;
			}

			container.setAttribute('aria-busy', 'true');
			this.clearContainer();

			const placeholder = document.createElement('div');
			placeholder.className = 'paypercut-checkout__placeholder';
			placeholder.textContent = this.config.loadingMessage;
			container.appendChild(placeholder);
		}

		/**
		 * Render a simple error message inside the Paypercut container. This
		 * does not use WooCommerce notices on purpose – it is only about the
		 * iframe area failing to initialise.
		 */
		showError(message) {
			const container = this.getContainer();
			if (!container) {
				return;
			}

			container.setAttribute('aria-busy', 'false');
			this.clearContainer();

			const errorEl = document.createElement('div');
			errorEl.className = 'paypercut-checkout__error';
			errorEl.textContent = message;
			container.appendChild(errorEl);
		}

		/**
		 * Helper that clears any previous iframe / placeholder content.
		 */
		clearContainer() {
			const container = this.getContainer();
			if (container) {
				container.innerHTML = '';
			}
		}

		/**
		 * Return the DOM element that should host the Paypercut iframe.
		 */
		getContainer() {
			return document.querySelector(this.config.containerId);
		}

		/**
		 * Look up the hidden input that stores a given transaction identifier
		 * (session / payment / paymentIntent).
		 */
		getTransactionField(field) {
			const selector = TRANSACTION_FIELD_SELECTORS[field];
			if (!selector) {
				return null;
			}

			const input = document.querySelector(selector);
			return input instanceof HTMLInputElement ? input : null;
		}

		/**
		 * Store a given transaction identifier in its hidden field, if present.
		 */
		setHiddenFieldValue(field, value) {
		const input = this.getTransactionField(field);
		if (input) {
			input.value = typeof value === 'string' ? value : '';
		}
	}

		/**
		 * Reset all hidden fields that belong to the Paypercut transaction.
		 */
		clearTransactionFields() {
			this.setHiddenFieldValue('session', '');
			this.setHiddenFieldValue('payment', '');
			this.setHiddenFieldValue('paymentIntent', '');
		}

		/**
		 * Persist IDs from the Paypercut success payload into hidden fields so
		 * that the PHP side can attach them to the order.
		 */
	persistSuccessPayload(data) {
		if (!data || typeof data !== 'object') {
			return;
		}


		if (typeof data.paymentIntentId === 'string') {
			this.setHiddenFieldValue('paymentIntent', data.paymentIntentId);
		}
	}
	
		/**
		 * Return the currently selected WooCommerce payment method ID.
		 */
	getSelectedPaymentMethod() {
		const input = document.querySelector(`input[name="${PAYMENT_METHOD_NAME}"]:checked`);
		return input instanceof HTMLInputElement ? input.value : '';
	}

		/**
		 * Convenience wrapper used throughout the controller.
		 */
	isPaypercutSelected() {
		return this.getSelectedPaymentMethod() === this.config.gatewayId;
	}

		/**
		 * Check whether the Paypercut SDK has been loaded on the page.
		 */
	isSdkAvailable() {
		return typeof window.PaypercutCheckout === 'function';
	}

		/**
		 * Small flag used to hand control back to WooCommerce for a single
		 * submit. This avoids endless loops where our own event handlers keep
		 * intercepting the form submission that we just triggered.
		 */
	shouldAllowNativeSubmit() {
			if (!this.allowNativeSubmit) {
				return false;
			}

			this.allowNativeSubmit = false;
			return true;
		}

		/**
		 * After Paypercut has confirmed that the payment succeeded, submit the
		 * relevant WooCommerce form so that the order (or payment method) is
		 * actually created.
		 *
		 * For normal checkout this is `form.checkout`. On the "Add payment
		 * method" screen it is the dedicated add‑payment form.
		 */
	submitWooCommerceForm() {
		if (!this.paymentSuccess) {
			return;
		}

		if (this.config.context === 'add_payment_method') {
			const addPaymentForm = document.querySelector(
				this.config.addPaymentFormSelector || '#add_payment_method'
			);

			if (addPaymentForm instanceof HTMLFormElement) {
				this.allowNativeSubmit = true;
				if (typeof addPaymentForm.requestSubmit === 'function') {
					addPaymentForm.requestSubmit();
				} else {
					addPaymentForm.submit();
				}
				return;
			}
		}

		if (this.config.orderId) {
			const orderPayForm = document.querySelector('form#order_review');
			if (orderPayForm instanceof HTMLFormElement) {
				this.allowNativeSubmit = true;
				
				const $ = window.jQuery;
				if ($ && $.fn) {
					$(orderPayForm).submit();
				} else {
					orderPayForm.submit();
				}
				return;
			}
		}

		const classicForm = document.querySelector('form.checkout');
		if (classicForm instanceof HTMLFormElement) {
			this.allowNativeSubmit = true;
			
			const $ = window.jQuery;
			if ($ && $.fn) {
				$(classicForm).submit();
			} else {
				classicForm.submit();
			}
		}
	}
	}

	document.addEventListener('DOMContentLoaded', () => {
		const config = window.paypercutCheckoutData || {};

		const controller = new PaypercutCheckoutController(config);
		controller.init();

		window.paypercutCheckoutController = controller;
	});
})();
