import { decodeEntities } from '@wordpress/html-entities';
import { createElement, useEffect, useState, useRef } from '@wordpress/element';

// This file wires the Paypercut gateway into WooCommerce Blocks.
//
// High‑level responsibilities:
// - Register Paypercut as a Blocks payment method.
// - Render the Paypercut iframe when the method is selected.
// - Keep the Paypercut session in sync with the cart totals.
// - Integrate the Paypercut SDK events (card + wallets) with Blocks'
//   payment lifecycle (onPaymentSetup) and its own form validation.

const MAX_ATTEMPTS = 20;
const RETRY_DELAY = 150;

// Resolve the two global objects that Woo Blocks exposes. Depending on the
// version these may live on window.wc or directly on window.
const resolveGlobals = () => ({
	registry: (window.wc && window.wc.wcBlocksRegistry) || window.wcBlocksRegistry || null,
	settingsApi: (window.wc && window.wc.wcSettings) || window.wcSettings || null,
});

// Entry point used at the bottom of the file. We retry a few times because
// Blocks may register its globals slightly after DOMContentLoaded.
const boot = (attempt = 0) => {
	const { registry, settingsApi } = resolveGlobals();

	if (!registry || !settingsApi) {
		if (attempt >= MAX_ATTEMPTS) {
			return;
		}
		setTimeout(() => boot(attempt + 1), RETRY_DELAY);
		return;
	}

	const { registerPaymentMethod } = registry;
	const { getSetting } = settingsApi;

	if (typeof registerPaymentMethod !== 'function' || typeof getSetting !== 'function') {
		return;
	}

	// Settings are provided by PHP via PaypercutBlocksSupport::get_payment_method_data().
	const settings = getSetting('paypercut_payments_data', {});

	if (!settings || !settings.gatewayId) {
		return;
	}

	const containerId = settings.containerId || '#paypercut-checkout-container';
	const elementId = containerId.replace('#', '');
	const label = settings.title ? settings.title.replace(/<[^>]*>/g, '') : '';
	const description = decodeEntities(settings.description || '');
	const ajaxUrl = settings.ajaxUrl || '/wp-admin/admin-ajax.php';
	const nonce = settings.nonce || '';
	const gatewayId = settings.gatewayId;
	const appearance = settings.appearance || { theme: 'light', labels: 'above' };
	const checkoutMode = settings.checkoutMode || 'hosted';
	const validationErrorMessage = settings.validationErrorMessage;
	const sdkNotLoadedMessage = settings.sdkNotLoadedMessage;
	const paymentInitFailedMessage = settings.paymentInitFailedMessage;
	const initFailedMessage = settings.initFailedMessage;
	const sessionFailedMessage = settings.sessionFailedMessage;
	const networkErrorMessage = settings.networkErrorMessage;
	const notInitializedMessage = settings.notInitializedMessage;
	const paymentFailedMessage = settings.paymentFailedMessage;
	const submissionFailedMessage = settings.submissionFailedMessage;

	/**
	 * React component rendered by Woo Blocks inside the Paypercut payment
	 * method panel.
	 *
	 * This is the heart of the integration:
	 * - It manages the Paypercut session lifecycle.
	 * - It renders / destroys the iframe as needed.
	 * - It ties the Paypercut SDK events into Blocks' onPaymentSetup flow.
	 */
	const Content = ({ eventRegistration, emitResponse, activePaymentMethod }) => {
		const { onPaymentSetup } = eventRegistration || {};
		const [checkoutInstance, setCheckoutInstance] = useState(null);
		const [sessionId, setSessionId] = useState('');
		const [isLoading, setIsLoading] = useState(true);
		const [error, setError] = useState('');
		const containerRef = useRef(null);
		const lastTotalsRef = useRef('');
		const checkoutInstanceRef = useRef(null);
		const sessionIdRef = useRef('');
		const paymentSuccessRef = useRef(false);
		const walletFlowRef = useRef(false);
		const isProcessingRef = useRef(false);

		// Tear down the current Paypercut instance (if any) and clear related state.
		const destroyInstance = () => {
			if (checkoutInstance?.destroy) {
				checkoutInstance.destroy();
				setCheckoutInstance(null);
				checkoutInstanceRef.current = null;
			}
		};

		// Show a light overlay while either Blocks or the Paypercut SDK is doing work.
		const blockProcessingUI = () => {
			if (isProcessingRef.current) {
				return;
			}

			isProcessingRef.current = true;

			const $ = window.jQuery;
			if ($?.fn?.block) {
				$(document.body).block({
					message: null,
					overlayCSS: {
						background: '#fff',
						opacity: 0.6,
						cursor: 'wait',
					},
				});
			}
		};

		// Remove the processing overlay when the Paypercut SDK is done with its work.
		const unblockProcessingUI = () => {
			isProcessingRef.current = false;

			const $ = window.jQuery;
			if ($?.fn?.unblock) {
				$(document.body).unblock();
			}
		};

		/**
		 * Render a Blocks‑style error banner at the top of the checkout form.
		 * This is our single generic validation message for both card and
		 * wallet flows.
		 */
		const showValidationError = (message) => {
			const checkoutForm = document.querySelector('.wc-block-checkout, .wp-block-woocommerce-checkout');
			if (!checkoutForm) {
				return;
			}

			const existingNotices = checkoutForm.querySelectorAll('.wc-block-components-notice-banner.is-error[data-paypercut-validation]');
			existingNotices.forEach(notice => notice.remove());

			const noticeBanner = document.createElement('div');
			noticeBanner.className = 'wc-block-components-notice-banner is-error';
			noticeBanner.setAttribute('role', 'alert');
			noticeBanner.setAttribute('tabindex', '-1');
			noticeBanner.setAttribute('data-paypercut-validation', 'true');
			
			noticeBanner.innerHTML = `
				<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="24" height="24" aria-hidden="true" focusable="false">
					<path d="M12 3.2c-4.8 0-8.8 3.9-8.8 8.8 0 4.8 3.9 8.8 8.8 8.8 4.8 0 8.8-3.9 8.8-8.8 0-4.8-4-8.8-8.8-8.8zm0 16c-4 0-7.2-3.3-7.2-7.2C4.8 8 8 4.8 12 4.8s7.2 3.3 7.2 7.2c0 4-3.2 7.2-7.2 7.2zM11 17h2v-6h-2v6zm0-8h2V7h-2v2z"></path>
				</svg>
				<div class="wc-block-components-notice-banner__content">
					${message}
				</div>
			`;

			const formStart = checkoutForm.querySelector('fieldset') || checkoutForm.firstElementChild;
			if (formStart && formStart.parentNode) {
				formStart.parentNode.insertBefore(noticeBanner, formStart);
			} else {
				checkoutForm.insertBefore(noticeBanner, checkoutForm.firstChild);
			}

			noticeBanner.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
			noticeBanner.focus();
		};

		/**
		 * Lightweight validation that relies on what Woo Blocks already knows.
		 *
		 * Rules:
		 * - If any field has aria‑invalid="true", we consider the form invalid.
		 * - If any required field is empty, we consider the form invalid.
		 * - If the terms checkbox exists and is not checked, we block payment.
		 */
		const isFormValid = () => {
			const checkoutForm = document.querySelector('.wc-block-checkout, .wp-block-woocommerce-checkout');
			if (!checkoutForm) {
				return true;
			}

			const invalidFields = checkoutForm.querySelectorAll('input[aria-invalid="true"], select[aria-invalid="true"], textarea[aria-invalid="true"]');
			if (invalidFields.length > 0) {
				return false;
			}

			const requiredFields = checkoutForm.querySelectorAll('input[required], select[required], textarea[required]');
			for (const field of requiredFields) {
				if (field.disabled || (field instanceof HTMLInputElement && field.type === 'hidden')) {
					continue;
				}

				if (field instanceof HTMLInputElement) {
					const type = field.type;

					if ((type === 'checkbox' || type === 'radio') && !field.checked) {
						return false;
					}

					if (type !== 'checkbox' && type !== 'radio') {
						const value = field.value ? field.value.trim() : '';
						if (!value) {
							return false;
						}
					}
				} else if (field instanceof HTMLSelectElement || field instanceof HTMLTextAreaElement) {
					const value = field.value ? field.value.trim() : '';
					if (!value) {
						return false;
					}
				}
			}

			const termsCheckbox = checkoutForm.querySelector('#terms-and-conditions');
			if (termsCheckbox && !termsCheckbox.checked) {
				return false;
			}

			return true;
		};



		/**
		 * Create a Paypercut checkout session on the server and render a
		 * PaypercutCheckout instance for it.
		 *
		 * The PHP side is responsible for building the session; here we only
		 * pass the session id to the SDK and listen to its events.
		 */
		const createSession = () => {
			if (!window.PaypercutCheckout) {
				setError(sdkNotLoadedMessage);
				setIsLoading(false);
				return;
			}

			if (!containerRef.current) {
				return;
			}

			paymentSuccessRef.current = false;
			setIsLoading(true);
			setError('');

			const formData = new FormData();
			formData.append('action', 'paypercut_create_checkout_session');
			formData.append('context', 'checkout');
			formData.append('nonce', nonce);

			fetch(ajaxUrl, {
				method: 'POST',
				body: formData,
			})
				.then(response => response.json())
				.then(data => {
					if (data.success && data.data?.sessionId) {
						const session = data.data.sessionId;
						
						destroyInstance();
						setSessionId(session);
						sessionIdRef.current = session;
						
						setTimeout(() => {
							try {
								const instance = window.PaypercutCheckout({
									id: session,
									containerId: `#${elementId}`,
									form_only: true,
									appearance: appearance,
									validate_form: true
								});

								let isLoaded = false;

								// The iframe finished loading – we can hide the "busy" state.
								instance.on('loaded', () => {
									isLoaded = true;
									setIsLoading(false);
									setError('');
								});

								// If the SDK reports an error before it has loaded once, treat
								// that as a hard initialisation failure.
								instance.on('error', () => {
									if (!isLoaded) {
										setError(paymentInitFailedMessage);
										setIsLoading(false);
									}
								});

								// 3‑D Secure events only influence the overlay, the rest of the
								// flow is handled by the SDK itself.
								instance.on('threeds_started', () => {
									unblockProcessingUI();
								});

								instance.on('threeds_complete', () => {
									blockProcessingUI();
								});

								instance.on('threeds_canceled', () => {
									unblockProcessingUI();
								});

								instance.on('threeds_error', () => {
									unblockProcessingUI();
								});

								// Paypercut asks us to validate the Woo checkout form before it
								// talks to e.g. Google Pay / Apple Pay.
								instance.on('form_validation', ({ wallet }) => {
									// Mark that we're in a wallet flow (e.g. G Pay / Apple Pay).
									walletFlowRef.current = !!wallet;

									if (!isFormValid()) {
										paymentSuccessRef.current = false;
										unblockProcessingUI();
										showValidationError(validationErrorMessage);
										instance.failFormValidation(wallet, [
											{ code: 'validation_failed', message: validationErrorMessage },
										]);
										return;
									}

									// Validation passed, let the SDK continue.
									instance.completeFormValidation(wallet);
								});

								// When Paypercut reports a successful payment for a wallet flow,
								// remember that and trigger the standard Blocks flow by clicking
								// the "Place order" button. onPaymentSetup will then see that
								// success already happened and simply resolve SUCCESS without
								// submitting to the SDK again.
								instance.on('success', () => {
									if (!walletFlowRef.current) {
										// Card flow goes through onPaymentSetup/submit, leave it alone.
										return;
									}

									paymentSuccessRef.current = true;

									const checkoutForm = document.querySelector('.wc-block-components-form.wc-block-checkout__form');
									const placeOrderButton = checkoutForm
										? checkoutForm.querySelector('.wc-block-components-checkout-place-order-button')
										: null;

									if (placeOrderButton instanceof HTMLButtonElement) {
										placeOrderButton.click();
									}
								});

								instance.render();
								setCheckoutInstance(instance);
								checkoutInstanceRef.current = instance;
							} catch (err) {
								setError(initFailedMessage);
								setIsLoading(false);
							}
						}, 100);
					} else {
						setError(sessionFailedMessage);
						setIsLoading(false);
					}
				})
				.catch(() => {
					setError(networkErrorMessage);
					setIsLoading(false);
				});
		};

		// Keep the Paypercut session in sync with cart totals. When something on
		// the cart changes (shipping, discounts, quantities, …) we regenerate
		// the session so the iframe always reflects the current order.
		useEffect(() => {
			if (checkoutMode === 'embedded') {
				createSession();
			} else {
				setIsLoading(false);
			}
			
			const originalFetch = window.fetch;
			window.fetch = function(...args) {
				const [url, options] = args;
				
				if (url && url.includes('/wc/store/v1/checkout') && options?.method === 'POST') {
					try {
						const body = JSON.parse(options.body);
						const activeMethod = window.wp?.data?.select('wc/store/payment')?.getActivePaymentMethod?.();
						
						if (!body.payment_method && activeMethod === gatewayId) {
							body.payment_method = gatewayId;
							options.body = JSON.stringify(body);
						}
					} catch (e) {
					}
				}
				
				return originalFetch.apply(this, args);
			};

			return () => {
				window.fetch = originalFetch;
				destroyInstance();
			};
		}, []);

		// Intercept the Store API checkout call and make sure the correct
		// payment_method is present in the payload when Paypercut is the
		// active method.
		useEffect(() => {
			if (checkoutMode !== 'embedded') {
				return;
			}

			if (!window.wp?.data?.subscribe) {
				return;
			}

			const unsubscribe = window.wp.data.subscribe(() => {
				try {
					const cartStore = window.wp.data.select('wc/store/cart');
					if (!cartStore?.getCartTotals) {
						return;
					}

					const totals = cartStore.getCartTotals();
					if (!totals) {
						return;
					}

					const totalsHash = `${totals.total_price || 0}|${totals.total_discount || 0}|${totals.total_shipping || 0}|${totals.total_tax || 0}|${totals.currency_code || ''}`;

					if (totalsHash !== lastTotalsRef.current && lastTotalsRef.current !== '') {
						lastTotalsRef.current = totalsHash;
						createSession();
					} else if (lastTotalsRef.current === '') {
						lastTotalsRef.current = totalsHash;
					}
				} catch (error) {
				}
			});

			return unsubscribe;
		}, [checkoutInstance]);

		/**
		 * Core integration point with Woo Blocks. This callback is invoked when
		 * the customer clicks the "Place order" button while Paypercut is the
		 * selected payment method.
		 *
		 * We return a promise that resolves with one of the values from
		 * emitResponse.responseTypes (SUCCESS / FAIL / ERROR). Woo Blocks then
		 * decides whether to send the Store API /checkout request or show an
		 * error to the customer.
		 */
		useEffect(() => {
			if (typeof onPaymentSetup !== 'function') {
				return;
			}

			const unsubscribe = onPaymentSetup(() => {
				return new Promise((resolve) => {
					if (checkoutMode !== 'embedded') {
						paymentSuccessRef.current = true;
						resolve({
							type: emitResponse?.responseTypes?.SUCCESS || 'success',
						});
						return;
					}

					// If a wallet flow has already completed successfully inside the
					// Paypercut iframe, don't submit again. Just tell Blocks that
					// payment setup succeeded so it can create the order.
					if (paymentSuccessRef.current && walletFlowRef.current) {
						paymentSuccessRef.current = false;
						walletFlowRef.current = false;

						resolve({
							type: emitResponse?.responseTypes?.SUCCESS || 'success',
						});
						return;
					}

					const currentInstance = checkoutInstanceRef.current;
					const currentSessionId = sessionIdRef.current;
					
					// No instance / session – tell Blocks that the gateway is not
					// ready. This should be rare and usually points to a config
					// or timing problem.
					if (!currentInstance || !currentSessionId) {
						paymentSuccessRef.current = false;
						resolve({
							type: emitResponse?.responseTypes?.ERROR || 'error',
							message: notInitializedMessage,
						});
						return;
					}

					// Let Woo Blocks handle the exact per‑field errors; we only
					// show a single generic message and avoid raising the generic
					// "There was a problem with your payment option." string.
					if (!isFormValid()) {
						paymentSuccessRef.current = false;
						unblockProcessingUI();
						showValidationError(validationErrorMessage);

						resolve({
							type: emitResponse?.responseTypes?.SUCCESS || 'success',
						});
						return;
					}

					paymentSuccessRef.current = false;
					blockProcessingUI();

					let resolved = false;
					
					// Called when the SDK reports that the payment completed
					// successfully. At this point we resolve SUCCESS and let
					// Woo Blocks create the order via the Store API.
					const successHandler = () => {
						if (resolved) {
							return;
						}
						resolved = true;
						paymentSuccessRef.current = true;
						unblockProcessingUI();
						resolve({
							type: emitResponse.responseTypes.SUCCESS,
						});
					};

					// Called when the SDK reports a failure. We pass the message
					// back to Blocks so the customer sees a specific reason
					// instead of a generic "payment failed" banner.
					const errorHandler = (error) => {
						if (resolved) {
							return;
						}
						resolved = true;
						paymentSuccessRef.current = false;
						unblockProcessingUI();
						
						resolve({
							type: emitResponse?.responseTypes?.FAIL || 'failure',
							message: error?.message || paymentFailedMessage,
						});
					};

					try {
						currentInstance.once('success', successHandler);
						currentInstance.once('error', errorHandler);
						currentInstance.submit();
					} catch (err) {
						if (!resolved) {
							resolved = true;
							paymentSuccessRef.current = false;
							unblockProcessingUI();
							resolve({
								type: emitResponse?.responseTypes?.FAIL || 'failure',
								message: submissionFailedMessage,
							});
						}
					}
				});
			});

			return unsubscribe;
		}, [onPaymentSetup]);

		if (checkoutMode !== 'embedded') {
			return null;
		}

		return createElement(
			'div',
			null,
			description ? createElement('div', {
				className: 'paypercut-blocks-description',
				dangerouslySetInnerHTML: { __html: description }
			}) : null,
			createElement(
				'div',
				{
					className: 'paypercut-checkout__container',
					'aria-live': 'polite',
					'aria-busy': isLoading ? 'true' : 'false',
				},
				error ? error : createElement('div', {
					ref: containerRef,
					id: elementId,
				})
			)
		);
	};

	const renderBrandIcons = (icons) => {
		if (!Array.isArray(icons) || icons.length === 0) {
			return null;
		}

		return createElement(
			'span',
			{ className: 'paypercut-card-brands' },
			icons.map((icon, index) =>
				createElement('img', {
					key: index,
					src: icon.src,
					alt: icon.alt || '',
					className: 'paypercut-card-brands__icon',
					loading: 'lazy',
				})
			)
		);
	};

	const Label = (props) => {
		const { PaymentMethodLabel } = props.components;

		return createElement(
			'span',
			{ className: 'paypercut-blocks-label' },
			createElement(PaymentMethodLabel, { text: label }),
			renderBrandIcons(settings.brandIcons)
		);
	};

	registerPaymentMethod({
		name: gatewayId,
		label: createElement(Label, null),
		ariaLabel: label,
		content: createElement(Content, null),
		edit: createElement(Content, null),
		canMakePayment: () => true,
		placeOrderButtonLabel: label,
		supports: settings.supports || { features: [] },
	});
};

if (document.readyState === 'loading') {
	document.addEventListener('DOMContentLoaded', () => boot());
} else {
	boot();
}
