/**
 * Paypercut Test Connection JavaScript
 *
 * Handles the "Test Connection" button functionality for Paypercut payment gateways.
 * Allows merchants to test their API credentials before saving the gateway settings.
 *
 * Features:
 * - Tests API credentials for both standard and BNPL gateways
 * - Supports test/live mode selection for BNPL gateway
 * - Shows success/error status messages
 * - Validates API secret input before testing
 *
 * @package Paypercut\Payments\Assets
 */

(function() {
	'use strict';

	const config = window.paypercutTestConnection || {};
	const i18n = config.i18n || {};

	const testButton = document.getElementById('paypercut-test-connection');
	if (!testButton) return;

	// Try to find the API secret field - could be for main gateway or BNPL gateway
	// Field names differ based on which gateway settings page we're on
	const fieldKeyMain = 'woocommerce_paypercut_payments_api_client_secret';
	const fieldKeyBnpl = 'woocommerce_paypercut_bnpl_api_client_secret';
	
	let apiSecretField = document.getElementById(fieldKeyMain) ||
		document.querySelector('input[name="' + fieldKeyMain + '"]');
	
	if (!apiSecretField) {
		apiSecretField = document.getElementById(fieldKeyBnpl) ||
			document.querySelector('input[name="' + fieldKeyBnpl + '"]');
	}

	const statusDiv = document.getElementById('paypercut-connection-status');

	if (!apiSecretField || !statusDiv) {
		return;
	}

	/**
	 * Get the current gateway section from the URL.
	 *
	 * Determines whether we're on the standard gateway or BNPL gateway settings page
	 * by checking the 'section' query parameter in the URL.
	 *
	 * @returns {string} The gateway section identifier (e.g., 'paypercut_payments' or 'paypercut_bnpl').
	 */
	function getGatewaySection() {
		const urlParams = new URLSearchParams(window.location.search);
		const section = urlParams.get('section') || '';
		return section;
	}

	/**
	 * Display a status message to the user.
	 *
	 * @param {string} message - The message text to display.
	 * @param {string} type - Message type: 'success' or 'error'.
	 */
	function showStatus(message, type) {
		statusDiv.style.display = 'block';
		statusDiv.className = type === 'success' ? 'notice notice-success inline' : 'notice notice-error inline';
		statusDiv.innerHTML = '<p>' + message + '</p>';
	}

	/**
	 * Hide the status message display.
	 */
	function hideStatus() {
		statusDiv.style.display = 'none';
		statusDiv.innerHTML = '';
	}

	/**
	 * Handle test connection button click.
	 *
	 * Validates the API secret, collects gateway configuration (including mode for BNPL),
	 * and sends an AJAX request to test the API connection.
	 *
	 * @param {Event} e - The click event.
	 */
	testButton.addEventListener('click', async function(e) {
		e.preventDefault();

		const apiSecret = apiSecretField.value ? apiSecretField.value.trim() : '';

		if (!apiSecret) {
			showStatus(i18n.apiSecretRequired, 'error');
			return;
		}

		// Update button state to show loading
		testButton.disabled = true;
		testButton.textContent = i18n.testing;
		hideStatus();

		// Prepare request data for AJAX submission
		const params = new URLSearchParams();
		params.append('action', 'paypercut_test_connection');
		params.append('api_client_secret', apiSecret);
		params.append('nonce', config.nonce);
		
		// Include gateway section to determine which gateway we're testing
		const gatewaySection = getGatewaySection();
		if (gatewaySection) {
			params.append('gateway_section', gatewaySection);
			
			// For BNPL gateway, include mode field (test/live) to use correct API endpoint
			if (gatewaySection === 'paypercut_bnpl') {
				const modeField = document.getElementById('woocommerce_paypercut_bnpl_mode') ||
					document.querySelector('select[name="woocommerce_paypercut_bnpl_mode"]');
				if (modeField && modeField.value) {
					params.append('mode', modeField.value);
				}
			}
		}

		try {
			const ajaxUrl = typeof ajaxurl !== 'undefined' ? ajaxurl : config.ajaxurl;

			const response = await fetch(ajaxUrl, {
				method: 'POST',
				headers: {
					'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8',
					'Accept': 'application/json',
				},
				body: params.toString()
			});

			const data = await response.json();

			if (data.success) {
				showStatus(data.data.message, 'success');
			} else {
				showStatus(data.data.message, 'error');
			}
		} catch (error) {
			showStatus(i18n.networkError, 'error');
		} finally {
			testButton.disabled = false;
			testButton.textContent = i18n.testConnection;
		}
	});
})();