/**
 * Paypercut Setup Wizard JavaScript
 *
 * Handles the initial setup wizard for Paypercut payment gateways.
 * This wizard is displayed when the gateway is not yet configured and allows
 * merchants to enter their API credentials and complete the initial setup.
 *
 * Features:
 * - Validates API secret input
 * - Submits setup data via AJAX
 * - Handles both standard and BNPL gateway setup
 * - Shows success/error messages
 * - Redirects to gateway settings after successful setup
 *
 * @package Paypercut\Payments\Assets
 */

(function() {
	'use strict';

	const config = window.paypercutSetupWizard || {};

	/**
	 * Clean up URL parameters if setup was already completed.
	 * Removes the 'paypercut_setup_complete' query parameter from the URL
	 * and clears the beforeunload warning.
	 */
	if (config.setupComplete === true) {
		window.onbeforeunload = null;
		const url = new URL(window.location.href);
		url.searchParams.delete('paypercut_setup_complete');
		window.history.replaceState({}, '', url);
	}

	/**
	 * Flag to track if setup has been completed in this session.
	 * @type {boolean}
	 */
	let setupComplete = false;

	/**
	 * Initialize the setup wizard.
	 *
	 * Sets up event listeners, prevents WooCommerce change events on form fields,
	 * and prepares the form for submission.
	 */
	function init() {
		document.body.classList.add('paypercut-setup-wizard-active');

		const form = document.getElementById('paypercut-setup-form');
		if (!form) {
			return;
		}

		// Prevent WooCommerce from triggering change events on setup wizard fields
		// This avoids conflicts with WooCommerce's settings save mechanism
		const setupWizardInputs = form.querySelectorAll('input, textarea, select');
		setupWizardInputs.forEach(function(input) {
			input.classList.add('wc-settings-prevent-change-event');
		});

		const config = window.paypercutSetupWizard || {};
		const i18n = config.i18n || {};

		const submitBtn = document.getElementById('paypercut-setup-submit');
		const buttonText = submitBtn ? submitBtn.querySelector('.button-text') : null;
		const buttonSpinner = submitBtn ? submitBtn.querySelector('.button-spinner') : null;
		const messagesDiv = document.getElementById('paypercut-setup-messages');

		/**
		 * Display a message to the user.
		 *
		 * @param {string} message - The message text to display.
		 * @param {string} type - Message type: 'success' or 'error'.
		 */
		function showMessage(message, type) {
			if (!messagesDiv) return;
			messagesDiv.textContent = message;
			messagesDiv.className = 'paypercut-setup-wizard__messages ' + type;
			messagesDiv.style.display = 'block';
		}

		/**
		 * Hide the message display area.
		 */
		function hideMessage() {
			if (messagesDiv) {
				messagesDiv.style.display = 'none';
			}
		}

		/**
		 * Set the loading state of the submit button.
		 *
		 * @param {boolean} loading - True to show loading state, false to hide it.
		 */
		function setLoading(loading) {
			if (!submitBtn || !buttonText || !buttonSpinner) return;
			if (loading) {
				submitBtn.disabled = true;
				buttonText.style.display = 'none';
				buttonSpinner.style.display = 'inline';
			} else {
				submitBtn.disabled = false;
				buttonText.style.display = 'inline';
				buttonSpinner.style.display = 'none';
			}
		}

		/**
		 * Handle form submission.
		 *
		 * Validates the form, collects data, and submits it via AJAX to complete
		 * the gateway setup. Supports both standard and BNPL gateway setup.
		 *
		 * @param {Event} e - The form submission event.
		 */
		async function handleSubmit(e) {
			if (e) {
				e.preventDefault();
				e.stopPropagation();
				e.stopImmediatePropagation();
			}

			hideMessage();
			setLoading(true);

			const formData = new FormData();
			const apiSecret = document.getElementById('api_client_secret');
			const nonceField = document.getElementById('paypercut-nonce');
			const gatewaySectionField = document.getElementById('paypercut-gateway-section');
			const modeField = document.getElementById('mode'); // Mode field for BNPL gateway (test/live)

			if (!apiSecret) {
				showMessage(i18n.fieldNotFound, 'error');
				setLoading(false);
				return;
			}

			const apiSecretValue = apiSecret.value ? apiSecret.value.trim() : '';
			if (!apiSecretValue) {
				showMessage(i18n.apiSecretRequired, 'error');
				setLoading(false);
				return;
			}

			if (!nonceField || !nonceField.value) {
				showMessage(i18n.securityTokenMissing, 'error');
				setLoading(false);
				return;
			}

			// Prepare form data for AJAX submission
			formData.append('api_client_secret', apiSecretValue);
			formData.append('nonce', nonceField.value);
			formData.append('action', 'paypercut_setup_wizard');
			
			// Include gateway section (paypercut_payments or paypercut_bnpl)
			if (gatewaySectionField && gatewaySectionField.value) {
				formData.append('gateway_section', gatewaySectionField.value);
			}
			
			// Include mode field if it exists (BNPL gateway: test/live)
			if (modeField && modeField.value) {
				formData.append('mode', modeField.value);
			}

			// Use WordPress ajaxurl if available, otherwise fall back to config
			const ajaxUrl = typeof ajaxurl !== 'undefined' ? ajaxurl : config.ajaxurl;

			try {
				const response = await fetch(ajaxUrl, {
					method: 'POST',
					body: formData
				});

				const data = await response.json();

				if (data.success) {
					let message = data.data.message;
					
					if (data.data.webhook_warning) {
						message += '\n\n' + data.data.webhook_warning;
						showMessage(message, 'warning');
					} else {
						showMessage(message, 'success');
					}
					
					setupComplete = true;

					// Redirect to gateway settings page after a short delay
					// Clear beforeunload handlers to prevent navigation warnings
					setTimeout(function() {
						window.onbeforeunload = null;
						if (window.jQuery) {
							window.jQuery(window).off('beforeunload');
						}
						window.location.href = data.data.reload_url;
					}, data.data.webhook_warning ? 3000 : 1500);
				} else {
					showMessage(data.data.message, 'error');
					setLoading(false);
				}
			} catch (error) {
				showMessage(i18n.networkError, 'error');
				setLoading(false);
			}
		}

		if (submitBtn) {
			submitBtn.addEventListener('click', function(e) {
				e.preventDefault();
				e.stopPropagation();
				e.stopImmediatePropagation();
				handleSubmit(e);
			});
		}
	}

	if (document.readyState === 'loading') {
		document.addEventListener('DOMContentLoaded', init);
	} else {
		init();
	}
})();