/**
 * Paypercut Refund Helper JavaScript
 *
 * Enhances the refund reason field in WooCommerce order edit pages for Paypercut
 * payment gateway orders. Replaces the standard text input with a dropdown select
 * populated with predefined refund reason options.
 *
 * Features:
 * - Detects Paypercut gateway orders
 * - Replaces refund reason text input with dropdown select
 * - Populates dropdown with predefined refund reason options
 * - Syncs dropdown selection with hidden input field
 * - Only activates for Paypercut gateway orders
 *
 * @package Paypercut\Payments\Assets
 */

(function (window, document) {
	'use strict';

	/**
	 * Configuration object passed from PHP.
	 * Expected properties:
	 * - placeholder: Default text for the dropdown placeholder
	 * - options: Object mapping refund reason keys to display labels
	 * - isPaypercutGateway: Boolean indicating if current order uses Paypercut gateway
	 */
	const config = window.paypercutRefundHelper || {};
	
	/**
	 * Placeholder text for the refund reason dropdown.
	 * @type {string}
	 */
	const placeholder =
		typeof config.placeholder === 'string' ? config.placeholder : 'Select refund reason';
	
	/**
	 * Refund reason options as key-value pairs.
	 * Keys are the refund reason codes, values are display labels.
	 * @type {Object<string, string>}
	 */
	const options =
		config.options && typeof config.options === 'object' ? config.options : {};
	
	/**
	 * Whether the current order uses a Paypercut payment gateway.
	 * @type {boolean}
	 */
	const isPaypercutGateway = Boolean(config.isPaypercutGateway);

	/**
	 * Early exit if not a Paypercut gateway order or no options available.
	 * Prevents unnecessary DOM manipulation for non-Paypercut orders.
	 */
	if (!isPaypercutGateway || !Object.keys(options).length) {
		return;
	}

	/**
	 * ID for the refund reason dropdown select element.
	 * @type {string}
	 */
	const selectId = 'paypercut-refund-reason-select';

	/**
	 * Enhance the refund reason field by replacing text input with dropdown.
	 *
	 * Finds the refund reason input field and replaces it with a dropdown select
	 * populated with predefined refund reason options. The original input is hidden
	 * but kept in the DOM to maintain compatibility with WooCommerce's refund form.
	 * The dropdown value is synced to the hidden input on change.
	 *
	 * @returns {boolean} True if enhancement was successful or already applied, false if field not found.
	 */
	function enhanceReasonField() {
		// Find the refund reason input field (standard WooCommerce field)
		const reasonInput = document.getElementById('refund_reason');
		if (!reasonInput) {
			return false;
		}

		// Find the parent container (td element) and check if already enhanced
		const container = reasonInput.closest('td');
		if (!container || reasonInput.dataset.paypercutEnhanced === '1') {
			return !!container;
		}

		// Mark as enhanced to prevent duplicate enhancement
		reasonInput.dataset.paypercutEnhanced = '1';
		
		if (isPaypercutGateway) {
			// Hide the original text input but keep it in DOM for form submission
			reasonInput.type = 'hidden';
			reasonInput.setAttribute('aria-hidden', 'true');
			reasonInput.style.display = 'none';

			// Create the dropdown select element
			const select = document.createElement('select');
			select.id = selectId;
			select.className = 'paypercut-refund-select';

			// Add placeholder option as first option
			const placeholderOption = document.createElement('option');
			placeholderOption.value = '';
			placeholderOption.textContent = placeholder;
			select.appendChild(placeholderOption);

			// Populate dropdown with refund reason options from config
			Object.keys(options).forEach((key) => {
				const value = key;
				const label = options[key];
				const option = document.createElement('option');
				option.value = value;
				option.textContent = label;
				select.appendChild(option);
			});

			// Sync dropdown selection to hidden input field
			// This ensures WooCommerce's refund form receives the selected value
			select.addEventListener('change', () => {
				reasonInput.value = select.value;
			});

			// Insert dropdown before the hidden input in the DOM
			container.insertBefore(select, reasonInput);
		}

		return true;
	}

	/**
	 * Initialize the refund reason field enhancement.
	 * Waits for DOM to be ready if necessary, otherwise runs immediately.
	 */
	if (document.readyState === 'loading') {
		document.addEventListener('DOMContentLoaded', enhanceReasonField);
	} else {
		enhanceReasonField();
	}
})(window, document);

