/**
 * Handles delete webhook functionality in the Paypercut gateway settings.
 *
 * @package Paypercut\Payments\Assets\Admin
 */

(function() {
	'use strict';

	const config = window.paypercutAdmin || {};
	const i18n = config.i18n || {};

	function init() {
		const deleteButtons = document.querySelectorAll('.paypercut-delete-webhook');
		
		if (deleteButtons.length === 0) {
			return;
		}

		deleteButtons.forEach(function(button) {
			button.addEventListener('click', async function(e) {
				e.preventDefault();
				
				const webhookId = button.getAttribute('data-webhook-id');
				const nonce = button.getAttribute('data-nonce');
				
				if (!webhookId || !nonce) {
					alert(i18n.missingData || 'Missing required data.');
					return;
				}

				if (!confirm(i18n.confirmDelete || 'Are you sure you want to delete this webhook? This action cannot be undone.')) {
					return;
				}

				const originalText = button.textContent;
				button.disabled = true;
				button.textContent = i18n.deleting || 'Deleting...';

				const formData = new URLSearchParams();
				formData.append('action', 'paypercut_delete_webhook');
				formData.append('nonce', nonce);
				formData.append('webhook_id', webhookId);

				const ajaxUrl = typeof ajaxurl !== 'undefined' ? ajaxurl : config.ajaxurl;

				try {
					const response = await fetch(ajaxUrl, {
						method: 'POST',
						headers: {
							'Content-Type': 'application/x-www-form-urlencoded',
							'Accept': 'application/json'
						},
						body: formData.toString()
					});

					if (!response.ok) {
						throw new Error(`HTTP error! status: ${response.status}`);
					}

					const data = await response.json();

					if (data.success) {
						const row = button.closest('tr');
						if (row) {
							row.style.transition = 'opacity 0.3s';
							row.style.opacity = '0';
							setTimeout(function() {
								row.remove();
								
								const tbody = row.closest('tbody');
								if (tbody && tbody.querySelectorAll('tr').length === 0) {
									window.location.reload();
								}
							}, 300);
						} else {
							window.location.reload();
						}
					} else {
						alert(data.data?.message || (i18n.deleteFailed || 'Failed to delete webhook.'));
						button.disabled = false;
						button.textContent = originalText;
					}
				} catch (error) {
					console.error('Delete webhook error:', error);
					alert(i18n.networkError || 'Network error. Please try again.');
					button.disabled = false;
					button.textContent = originalText;
				}
			});
		});
	}

	if (document.readyState === 'loading') {
		document.addEventListener('DOMContentLoaded', init);
	} else {
		init();
	}
})();
