<?php
/**
 * Paypercut BNPL Gateway Class
 *
 * @package Paypercut_BNPL
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Paypercut BNPL Gateway Class
 */
class Paypercut_BNPL_Gateway extends WC_Payment_Gateway {

    /**
     * API instance
     *
     * @var Paypercut_BNPL_API
     */
    private $api;

    /**
     * Constructor
     */
    public function __construct() {
        $this->id = 'paypercut_bnpl';
        // Prefer external PNG (user-provided); fallback to inline SVG if it fails
        $logo_url = PAYPERCUT_BNPL_PLUGIN_URL . 'assets/images/ppc-logo.png?v=1';
        $svg_icon = '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24"><rect width="24" height="24" fill="#111" rx="4"/><path fill="#fff" d="M8 18V6h6.2c2.1 0 3.8 1.6 3.8 3.6 0 2-1.7 3.6-3.8 3.6H10.2V18H8Zm2.2-6.2h4c.9 0 1.6-.7 1.6-1.6 0-.9-.7-1.6-1.6-1.6h-4v3.2Z"/></svg>';
        $this->icon = esc_url($logo_url);
        add_filter('woocommerce_gateway_icon', function($icon, $id) use ($logo_url, $svg_icon) {
            if ($id === $this->id) {
                $icon = '<img src="' . esc_url($logo_url) . '" alt="Paypercut" style="height:24px;" onerror="this.outerHTML=\'' . esc_attr($svg_icon) . '\';" />';
            }
            return $icon;
        }, 10, 2);
        $this->has_fields = true;
        $this->method_title = __('Paypercut BNPL', 'paypercut-bnpl-for-woocommerce');
        $this->method_description = __('Accept Buy Now Pay Later payments through Paypercut. Customers can split their purchases into manageable installments.', 'paypercut-bnpl-for-woocommerce');

        // Load the settings
        $this->init_form_fields();
        $this->init_settings();

        // Define user set variables from settings
        $this->title = __('Paypercut BNPL', 'paypercut-bnpl-for-woocommerce');
        $this->description = $this->get_option('description', __('Split your purchase into manageable installments with Paypercut BNPL', 'paypercut-bnpl-for-woocommerce'));
        $this->enabled = $this->get_option('enabled');
        $this->testmode = 'yes' === $this->get_option('testmode');
        $this->api_base_url = $this->get_option('api_base_url');
        $this->access_token = $this->get_option('access_token');
        $this->logging = 'yes' === $this->get_option('logging');
        $this->supported_currencies = $this->get_option('supported_currencies');
        // Fixed hard limits
        $this->min_amount = 100;
        $this->max_amount = 1500;

        // Initialize API - access token only
        $this->api = new Paypercut_BNPL_API('', '', $this->testmode, $this->api_base_url, $this->access_token);

        // Save settings
        add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));

        // Add order meta
        add_action('woocommerce_checkout_order_processed', array($this, 'add_order_meta'), 10, 1);

        // Handle webhooks and return URLs
        add_action('woocommerce_api_paypercut_bnpl_webhook', array($this, 'handle_webhook'));
        add_action('woocommerce_api_paypercut_bnpl_return', array($this, 'handle_return'));

        // Admin notices only for misconfiguration (no front-end debug)
        add_action('admin_notices', array($this, 'admin_notices'));

        // Enqueue checkout assets
        add_action('wp_enqueue_scripts', array($this, 'enqueue_checkout_assets'));
    }

    /**
     * Initialize Gateway Settings Form Fields
     */
    public function init_form_fields() {
        $this->form_fields = array(
            'enabled' => array(
                'title' => __('Enable/Disable', 'paypercut-bnpl-for-woocommerce'),
                'type' => 'checkbox',
                'label' => __('Enable Paypercut BNPL', 'paypercut-bnpl-for-woocommerce'),
                'default' => 'no',
            ),
            'description' => array(
                'title' => __('Description', 'paypercut-bnpl-for-woocommerce'),
                'type' => 'textarea',
                'description' => __('Payment method description that customers see during checkout.', 'paypercut-bnpl-for-woocommerce'),
                'default' => __('Split your purchase into manageable installments with Paypercut BNPL', 'paypercut-bnpl-for-woocommerce'),
                'desc_tip' => true,
            ),
            'testmode' => array(
                'title' => __('Test Mode', 'paypercut-bnpl-for-woocommerce'),
                'type' => 'checkbox',
                'label' => __('Enable Test Mode', 'paypercut-bnpl-for-woocommerce'),
                'default' => 'yes',
                'description' => __('Place the payment gateway in test mode using test API keys.', 'paypercut-bnpl-for-woocommerce'),
            ),
            'logging' => array(
                'title' => __('Logging', 'paypercut-bnpl-for-woocommerce'),
                'type' => 'checkbox',
                'label' => __('Log debug messages', 'paypercut-bnpl-for-woocommerce'),
                'description' => __('Enable logging for troubleshooting.', 'paypercut-bnpl-for-woocommerce'),
                'default' => 'yes',
            ),
            'api_base_url' => array(
                'title' => __('API Base URL (override)', 'paypercut-bnpl-for-woocommerce'),
                'type' => 'text',
                'description' => __('Optional. Override API base URL (e.g., https://bnpl-gw.bender.paypercut.net) if needed.', 'paypercut-bnpl-for-woocommerce'),
                'default' => '',
                'desc_tip' => true,
            ),
            'access_token' => array(
                'title' => __('Access Token (Bearer)', 'paypercut-bnpl-for-woocommerce'),
                'type' => 'password',
                'description' => __('Bearer token used for authorization. Not stored in code; entered here by admin.', 'paypercut-bnpl-for-woocommerce'),
                'default' => '',
                'desc_tip' => true,
            ),
            'supported_currencies' => array(
                'title' => __('Supported Currencies', 'paypercut-bnpl-for-woocommerce'),
                'type' => 'multiselect',
                'description' => __('Select currencies that support BNPL.', 'paypercut-bnpl-for-woocommerce'),
                'default' => array('BGN', 'CHF', 'CZK', 'EUR', 'GBP', 'HUF', 'PLN', 'RON', 'USD'),
                'options' => $this->get_currency_options(),
                'desc_tip' => true,
            ),
        );
    }

    /**
     * Get currency options
     *
     * @return array
     */
    private function get_currency_options() {
        $allowed = array('BGN', 'CHF', 'CZK', 'EUR', 'GBP', 'HUF', 'PLN', 'RON', 'USD');
        $wc_currencies = get_woocommerce_currencies();
        $currencies = array();

        foreach ($allowed as $code) {
            $name = isset($wc_currencies[$code]) ? $wc_currencies[$code] : $code;
            $currencies[$code] = $name . ' (' . get_woocommerce_currency_symbol($code) . ')';
        }

        return $currencies;
    }

    /**
     * Check if the gateway is available for use
     *
     * @return bool
     */
    public function is_available() {
        // Check if enabled
        if ('no' === $this->enabled) {
            return false;
        }

        // Check if access token is set
        if (empty($this->access_token)) {
            return false;
        }

        // Check currency support
        $current_currency = get_woocommerce_currency();
        if (!in_array($current_currency, $this->supported_currencies)) {
            return false;
        }

        // Check amount limits
        $cart_total = WC()->cart ? WC()->cart->get_total('raw') : 0;
        if ($cart_total < $this->min_amount || $cart_total > $this->max_amount) {
            return false;
        }

        return true;
    }

    /**
     * Payment fields on checkout page
     */
    public function payment_fields() {
        // Define logo variables
        $logo_url = PAYPERCUT_BNPL_PLUGIN_URL . 'assets/images/ppc-logo.png?v=1';
        $svg_icon = '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24"><rect width="24" height="24" fill="#111" rx="4"/><path fill="#fff" d="M8 18V6h6.2c2.1 0 3.8 1.6 3.8 3.6 0 2-1.7 3.6-3.8 3.6H10.2V18H8Zm2.2-6.2h4c.9 0 1.6-.7 1.6-1.6 0-.9-.7-1.6-1.6-1.6h-4v3.2Z"/></svg>';
        
        // Load fresh description from settings
        $saved_desc = $this->get_option('description');
        if (!empty($saved_desc)) {
            $this->description = $saved_desc;
        }
        
        // Show logo + description on checkout
        echo '<div class="paypercut-bnpl-row" style="padding-left:0;margin-left:0;">';
        echo '<img class="paypercut-logo" src="' . esc_url($logo_url) . '" alt="Paypercut" onerror="this.outerHTML=\'' . esc_attr($svg_icon) . '\';" />';
        echo '<div class="paypercut-texts">';
        echo '<p class="paypercut-title">' . esc_html($this->title) . '</p>';
        if ($this->description) {
            echo '<p class="paypercut-desc">' . esc_html($this->description) . '</p>';
        }
        echo '</div>';
        echo '</div>';

        // Get customer limits if API is available
        if ($this->api && is_user_logged_in()) {
            $customer_email = wp_get_current_user()->user_email;
            $limits = $this->api->get_customer_limits($customer_email);
            
            if (!is_wp_error($limits) && isset($limits['available_credit'])) {
                echo '<div class="paypercut-credit-info">';
                echo '<p><strong>' . esc_html__('Available Credit:', 'paypercut-bnpl-for-woocommerce') . '</strong> ' . wp_kses_post(wc_price($limits['available_credit'])) . '</p>';
                echo '</div>';
            }
        }

    }

    /**
     * Validate checkout fields
     */
    public function validate_checkout_fields() {
        // No custom fields to validate
        return;
    }

    /**
     * Save checkout fields
     *
     * @param int $order_id
     */
    public function save_checkout_fields($order_id) {
        // No custom fields to save
        return;
    }

    /**
     * Add order meta
     *
     * @param int $order_id
     */
    public function add_order_meta($order_id) {
        if (!isset($_POST['payment_method']) || $this->id !== $_POST['payment_method']) {
            return;
        }

        $order = wc_get_order($order_id);
        if ($order) {
            $order->add_meta_data('_paypercut_payment_method', 'bnpl', true);
            $order->add_meta_data('_paypercut_testmode', $this->testmode ? 'yes' : 'no', true);
            $order->save();
        }
    }

    /**
     * Process the payment and return the result
     *
     * @param int $order_id
     * @return array
     */
    public function process_payment($order_id) {
        $order = wc_get_order($order_id);

        if (!$order) {
            return array(
                'result' => 'failure',
                'redirect' => '',
            );
        }

        // Check if API is available
        if (!$this->api) {
            wc_add_notice(__('Payment gateway configuration error. Please contact support.', 'paypercut-bnpl-for-woocommerce'), 'error');
            return array(
                'result' => 'failure',
                'redirect' => '',
            );
        }

        // Build BNPL payload per docs: https://docs.paypercut.io/bnpl-api-reference
        $approved_url = $this->get_return_url($order);
        $canceled_url = wc_get_cart_url();

        $currency_code = $order->get_currency();
        // BNPL-GW expects minor units (integer strings), e.g. 54.00 -> "5400"
        $decimals = max(0, (int) wc_get_price_decimals());
        $scale = pow(10, $decimals);
        $format_amount = function($amount) use ($scale) {
            return (string) (int) round(((float) $amount) * $scale);
        };

        $payload = array(
            'application_id' => $this->payment_profile_id ?: '',
            'merchant_purchase_ref' => (string) $order->get_order_number(),
            'customer_redirect' => array(
                'approved_url' => $approved_url,
                'canceled_url' => $canceled_url,
            ),
            'purchase_details' => array(
                'billing_address' => array(
                    'person' => array(
                        'first_name' => $order->get_billing_first_name(),
                        'last_name' => $order->get_billing_last_name(),
                        'other_names' => '',
                        'phone' => $order->get_billing_phone(),
                        'email' => $order->get_billing_email(),
                        'national_id' => '',
                    ),
                    'address' => array(
                        'country_code' => $order->get_billing_country(),
                        'city' => $order->get_billing_city(),
                        'postal_code' => $order->get_billing_postcode(),
                        'street1' => $order->get_billing_address_1(),
                        'street2' => $order->get_billing_address_2(),
                        'country_region' => $order->get_billing_state(),
                        'district' => '',
                        'block' => '',
                        'entrance' => '',
                        'floor' => '',
                        'apartment' => '',
                    ),
                ),
                'shipping_address' => array(
                    'person' => array(
                        'first_name' => $order->get_shipping_first_name() ?: $order->get_billing_first_name(),
                        'last_name' => $order->get_shipping_last_name() ?: $order->get_billing_last_name(),
                        'other_names' => '',
                        'phone' => $order->get_billing_phone(),
                        'email' => $order->get_billing_email(),
                        'national_id' => '',
                    ),
                    'address' => array(
                        'country_code' => $order->get_shipping_country() ?: $order->get_billing_country(),
                        'city' => $order->get_shipping_city() ?: $order->get_billing_city(),
                        'postal_code' => $order->get_shipping_postcode() ?: $order->get_billing_postcode(),
                        'street1' => $order->get_shipping_address_1() ?: $order->get_billing_address_1(),
                        'street2' => $order->get_shipping_address_2() ?: $order->get_billing_address_2(),
                        'country_region' => $order->get_shipping_state() ?: $order->get_billing_state(),
                        'district' => '',
                        'block' => '',
                        'entrance' => '',
                        'floor' => '',
                        'apartment' => '',
                    ),
                ),
                'customer' => array(
                    'customer_ref' => (string) $order->get_customer_id(),
                ),
                'shopping_card' => array(
                    'currency_code' => $currency_code,
                    'shipping_amount' => $format_amount($order->get_shipping_total()),
                    'tax_amount' => $format_amount($order->get_total_tax()),
                    'total_amount' => $format_amount($order->get_total()),
                    'discount_amount' => $format_amount($order->get_discount_total()),
                    'items' => array(),
                ),
            ),
            'transaction_flow_config' => array(
                'transaction_capture' => 'TRANSACTION_CAPTURE_UNSPECIFIED',
            ),
        );

        // Add order items
        foreach ($order->get_items() as $item) {
            $product = $item->get_product();
            $qty = max(1, (int) $item->get_quantity());
            $line_total = (float) $item->get_total();
            $unit_price = $line_total > 0 ? ($line_total / $qty) : (( $product && $product->get_price() ) ? (float) $product->get_price() : 0.0);

            $img = '';
            if ($product && $product->get_image_id()) {
                $img = wp_get_attachment_url($product->get_image_id());
            }
            if (empty($img)) {
                $img = PAYPERCUT_BNPL_PLUGIN_URL . 'assets/images/paypercut-logo.png';
            }

            $url = '';
            if ($product) {
                $url = get_permalink($product->get_id());
            }
            if (empty($url)) {
                $url = home_url('/');
            }

            $payload['purchase_details']['shopping_card']['items'][] = array(
                'title' => $item->get_name(),
                'image_url' => $img,
                'item_url' => $url,
                'sku' => $product ? $product->get_sku() : '',
                'item_version' => '',
                'qty' => $qty,
                'unit_price' => $format_amount($unit_price),
            );
        }

        // Create BNPL attempt and get redirect URL
        $response = $this->api->create_bnpl_attempt($payload);

        if (is_wp_error($response)) {
            $details = '';
            $data = $response->get_error_data();
            if (is_array($data)) {
                if (!empty($data['json']) && is_array($data['json'])) {
                    $details = ' (' . wp_json_encode($data['json']) . ')';
                } elseif (!empty($data['body'])) {
                    $details = ' (' . wp_html_excerpt($data['body'], 200) . ')';
                }
            }
            wc_add_notice(__('Payment processing failed: ', 'paypercut-bnpl-for-woocommerce') . $response->get_error_message() . $details, 'error');
            return array(
                'result' => 'failure',
                'redirect' => '',
            );
        }

        // Check if we got a redirect URL from Paypercut
        if (isset($response['redirect_url']) && !empty($response['redirect_url'])) {
            // Store transaction details for later processing
            $attempt_id = $response['attempt_id'] ?? '';
            if (!empty($attempt_id)) {
                $order->add_meta_data('_paypercut_attempt_id', $attempt_id, true);
            }
            if (isset($response['transaction_id'])) {
                $order->add_meta_data('_paypercut_transaction_id', $response['transaction_id'], true);
            }
            $order->add_meta_data('_paypercut_payment_status', 'pending', true);
            $order->save();

            // Redirect to Paypercut platform
            return array(
                'result' => 'success',
                'redirect' => $response['redirect_url'],
            );
        } else {
            // No redirect URL received
            wc_add_notice(__('Unable to initiate Paypercut payment. Please try again.', 'paypercut-bnpl-for-woocommerce'), 'error');
            return array(
                'result' => 'failure',
                'redirect' => '',
            );
        }
    }

    /**
     * Get webhook URL for Paypercut
     *
     * @return string
     */
    private function get_webhook_url() {
        return home_url('/wc-api/paypercut_bnpl_webhook');
    }

    /**
     * Handle return from Paypercut platform
     */
    public function handle_return() {
        // Get parameters from Paypercut return
        $transaction_id = isset($_GET['transaction_id']) ? sanitize_text_field(wp_unslash($_GET['transaction_id'])) : '';
        $attempt_id = isset($_GET['attempt_id']) ? sanitize_text_field(wp_unslash($_GET['attempt_id'])) : '';
        $status_hint = isset($_GET['status']) ? sanitize_text_field(wp_unslash($_GET['status'])) : '';
        $order_id = isset($_GET['order_id']) ? sanitize_text_field(wp_unslash($_GET['order_id'])) : '';

        if (empty($transaction_id) || empty($order_id)) {
            wc_add_notice(esc_html__('Invalid payment response from Paypercut.', 'paypercut-bnpl-for-woocommerce'), 'error');
            wp_redirect(wc_get_cart_url());
            exit;
        }

        // Find the order
        $order = wc_get_order($order_id);
        if (!$order) {
            wc_add_notice(esc_html__('Order not found.', 'paypercut-bnpl-for-woocommerce'), 'error');
            wp_redirect(wc_get_cart_url());
            exit;
        }

        // Prefer stored attempt id if not provided
        if (empty($attempt_id)) {
            $attempt_id = (string) $order->get_meta('_paypercut_attempt_id');
        }

        // Verify via API status before deciding outcome
        $final_status = '';
        if (!empty($attempt_id) && $this->api) {
            $status_resp = $this->api->get_bnpl_status($attempt_id);
            if (!is_wp_error($status_resp)) {
                // Expected shape: { attempt: { status: 'APPROVED' | 'COMPLETED' | ... } }
                if (!empty($status_resp['attempt']['status'])) {
                    $final_status = strtoupper((string) $status_resp['attempt']['status']);
                }
            }
        }
        if (empty($final_status)) {
            $final_status = strtoupper($status_hint);
        }

        // Process the payment result
        switch ($final_status) {
            case 'success':
            case 'completed':
            case 'approved':
            case 'ATTEMPT_STATUS_APPROVED':
            case 'ATTEMPT_STATUS_COMPLETED':
                // Payment successful
                $order->payment_complete($transaction_id);
                $order->add_order_note(esc_html__('Paypercut BNPL payment completed successfully.', 'paypercut-bnpl-for-woocommerce'));
                
                // Store additional payment details if provided
                if (isset($_GET['installment_plan'])) {
                    $order->add_meta_data('_paypercut_installment_plan', sanitize_text_field(wp_unslash($_GET['installment_plan'])), true);
                }
                
                $order->save();

                // Clear cart and redirect to thank you page
                WC()->cart->empty_cart();
                wp_redirect($this->get_return_url($order));
                exit;

            case 'failed':
            case 'declined':
            case 'ATTEMPT_STATUS_DECLINED':
                // Payment failed
                $order->update_status('failed', esc_html__('Paypercut BNPL payment was not successful.', 'paypercut-bnpl-for-woocommerce'));
                $order->save();
                
                $decline_reason = isset($_GET['reason']) ? sanitize_text_field(wp_unslash($_GET['reason'])) : '';
                $msg = esc_html__('Payment was not successful.', 'paypercut-bnpl-for-woocommerce');
                if (!empty($decline_reason)) {
                    /* translators: %s: decline reason text */
                    $msg .= ' ' . sprintf(esc_html__('Reason: %s', 'paypercut-bnpl-for-woocommerce'), $decline_reason);
                }
                wc_add_notice($msg, 'error');
                wp_redirect(wc_get_cart_url());
                exit;

            case 'cancelled':
            case 'ATTEMPT_STATUS_CANCELLED':
                // Payment cancelled
                $order->update_status('cancelled', esc_html__('Paypercut BNPL payment was cancelled by customer.', 'paypercut-bnpl-for-woocommerce'));
                $order->save();
                
                wc_add_notice(esc_html__('Payment was cancelled. You were returned to your cart.', 'paypercut-bnpl-for-woocommerce'), 'notice');
                wp_redirect(wc_get_cart_url());
                exit;

            default:
                // Unknown status
                wc_add_notice(esc_html__('Payment was not successful. Please try again.', 'paypercut-bnpl-for-woocommerce'), 'error');
                wp_redirect(wc_get_cart_url());
                exit;
        }
    }

    /**
     * Handle webhook notifications from Paypercut
     */
    public function handle_webhook() {
        $input = file_get_contents('php://input');
        $data = json_decode($input, true);

        if (!$data) {
            status_header(400);
            exit('Invalid JSON');
        }

        // Verify webhook signature if needed
        // This would depend on Paypercut's webhook security implementation
        if (!$this->verify_webhook_signature($input)) {
            status_header(401);
            exit('Invalid signature');
        }

        $transaction_id = $data['transaction_id'] ?? '';
        $status = $data['status'] ?? '';
        $order_id = $data['order_id'] ?? '';

        if (empty($transaction_id) || empty($order_id)) {
            status_header(400);
            exit('Missing required parameters');
        }

        // Find order by transaction ID or order ID
        $orders = wc_get_orders(array(
            'meta_key' => '_paypercut_transaction_id',
            'meta_value' => $transaction_id,
            'limit' => 1,
        ));

        if (empty($orders)) {
            // Try finding by order ID as fallback
            $order = wc_get_order($order_id);
        } else {
            $order = $orders[0];
        }

        if (!$order) {
            status_header(404);
            exit('Order not found');
        }

        // Update order status based on webhook
        switch ($status) {
            case 'completed':
            case 'success':
            case 'approved':
                if ($order->get_status() !== 'completed') {
                    $order->payment_complete($transaction_id);
                    $order->add_order_note(__('Paypercut BNPL payment completed via webhook.', 'paypercut-bnpl-for-woocommerce'));
                }
                break;
            case 'failed':
            case 'declined':
                $order->update_status('failed', __('Paypercut BNPL payment failed via webhook.', 'paypercut-bnpl-for-woocommerce'));
                break;
            case 'cancelled':
                $order->update_status('cancelled', __('Paypercut BNPL payment cancelled via webhook.', 'paypercut-bnpl-for-woocommerce'));
                break;
        }

        $order->save();

        status_header(200);
        exit('OK');
    }

    /**
     * Verify webhook signature
     *
     * @param string $payload
     * @return bool
     */
    private function verify_webhook_signature($payload) {
        // This should be implemented based on Paypercut's webhook security requirements
        // For now, we'll return true, but you should implement proper signature verification
        return true;
    }

    /**
     * Admin notices
     */
    public function admin_notices() {
        if (!current_user_can('manage_woocommerce')) return;
        if (empty($this->access_token)) {
            echo '<div class="notice notice-warning"><p>' . esc_html__('Paypercut BNPL: Please configure Access Token.', 'paypercut-bnpl-for-woocommerce') . '</p></div>';
        }
    }

    /**
     * Enqueue checkout assets
     */
    public function enqueue_checkout_assets() {
        if (is_checkout()) {
            wp_enqueue_style(
                'paypercut-bnpl-checkout',
                PAYPERCUT_BNPL_PLUGIN_URL . 'assets/css/checkout.css',
                array(),
                PAYPERCUT_BNPL_VERSION
            );

            // Add inline styles for payment fields layout
            $inline_css = '.paypercut-bnpl-row{display:flex;align-items:center;gap:16px;min-height:72px}' .
                          '.paypercut-bnpl-row .paypercut-logo{height:72px;width:auto;object-fit:contain;display:block}' .
                          '.paypercut-bnpl-row .paypercut-texts{display:flex;flex-direction:column;gap:4px}' .
                          '.paypercut-bnpl-row .paypercut-title{font-weight:600;margin:0}' .
                          '.paypercut-bnpl-row .paypercut-desc{margin:0}';
            wp_add_inline_style('paypercut-bnpl-checkout', $inline_css);

            wp_enqueue_script(
                'paypercut-bnpl-checkout',
                PAYPERCUT_BNPL_PLUGIN_URL . 'assets/js/checkout.js',
                array('jquery', 'wc-checkout'),
                PAYPERCUT_BNPL_VERSION,
                true
            );

            wp_localize_script('paypercut-bnpl-checkout', 'paypercut_bnpl_params', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('paypercut_checkout_nonce'),
                'currency_symbol' => get_woocommerce_currency_symbol(),
                'currency_code' => get_woocommerce_currency(),
            ));
        }
    }

    /**
     * Get gateway icon
     *
     * @return string
     */
    public function get_icon() {
        $icon_html = '';
        
        if ($this->icon) {
            $icon_html = '<img src="' . esc_url($this->icon) . '" alt="' . esc_attr($this->title) . '" style="max-height: 32px;" />';
        }

        return apply_filters('woocommerce_gateway_icon', $icon_html, $this->id);
    }
}
