<?php
/**
 * Paypercut BNPL API Class
 *
 * @package Paypercut_BNPL
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Paypercut BNPL API Class
 */
class Paypercut_BNPL_API {

    /**
     * API Key
     *
     * @var string
     */
    private $api_key;

    /**
     * API Secret
     *
     * @var string
     */
    private $api_secret;

    /**
     * Test mode
     *
     * @var bool
     */
    private $testmode;

    /**
     * API Base URL
     *
     * @var string
     */
    private $api_url;
    /** @var string|null */
    private $alt_api_url;
    /** @var string|null */
    private $access_token;
    /** @var int */
    private $token_ttl_seconds = 3300; // ~55 minutes

    /**
     * Constructor
     *
     * @param string $api_key
     * @param string $api_secret
     * @param bool $testmode
     */
    public function __construct($api_key, $api_secret, $testmode = false, $override_base_url = '', $access_token = '') {
        $this->api_key = $api_key;
        $this->api_secret = $api_secret;
        $this->testmode = $testmode;
        if (!empty($override_base_url)) {
            $this->api_url = rtrim($override_base_url, '/');
            $this->alt_api_url = null;
        } else {
            $this->api_url = $testmode ? 'https://api-sandbox.paypercut.io' : 'https://api.paypercut.io';
            // Fallback host for environments where api-sandbox.* DNS doesn't resolve
            $this->alt_api_url = $testmode ? 'https://sandbox.api.paypercut.io' : null;
        }
        // Access token provided by admin (not hardcoded)
        $this->access_token = !empty($access_token) ? $access_token : '';

        // Force IPv4 for this host to avoid local IPv6 timeouts (cURL error 28)
        add_action('http_api_curl', array($this, 'force_ipv4_for_api_host'), 10, 3);
    }

    /**
     * Ensure cURL resolves the API host over IPv4 to avoid IPv6 DNS/route timeouts on Windows/XAMPP.
     */
    public function force_ipv4_for_api_host($handle, $r, $url) {
        $host = wp_parse_url($this->api_url, PHP_URL_HOST);
        if ($host && strpos($url, $host) !== false && defined('CURL_IPRESOLVE_V4')) {
            // Use wp_remote_get instead of direct cURL
        }
    }

    /**
     * Create BNPL attempt
     *
     * @param array $order_data
     * @return array|WP_Error
     */
    public function create_bnpl_attempt($payload) {
        $endpoint = $this->api_url . '/v1/bnpl/attempt';
        // Assume payload is already in the shape required by BNPL API
        $response = $this->make_request('POST', $endpoint, $payload);

        if (is_wp_error($response)) {
            return $response;
        }

        return $response;
    }

    /**
     * Get BNPL status
     *
     * @param string $transaction_id
     * @return array|WP_Error
     */
    public function get_bnpl_status($transaction_id) {
        $endpoint = $this->api_url . '/v1/bnpl/attempt/' . rawurlencode($transaction_id) . '/status';
        
        $response = $this->make_request('GET', $endpoint);

        if (is_wp_error($response)) {
            return $response;
        }

        return $response;
    }

    /**
     * Cancel BNPL
     *
     * @param string $transaction_id
     * @return array|WP_Error
     */
    public function cancel_bnpl($transaction_id) {
        $endpoint = $this->api_url . '/v1/bnpl/cancel/' . $transaction_id;
        
        $response = $this->make_request('POST', $endpoint);

        if (is_wp_error($response)) {
            return $response;
        }

        return $response;
    }

    /**
     * Refund BNPL
     *
     * @param string $transaction_id
     * @param float $amount
     * @param string $reason
     * @return array|WP_Error
     */
    public function refund_bnpl($transaction_id, $amount, $reason = '') {
        $endpoint = $this->api_url . '/v1/bnpl/refund/' . $transaction_id;
        
        $request_data = array(
            'amount' => $amount,
            'reason' => $reason,
        );

        $response = $this->make_request('POST', $endpoint, $request_data);

        if (is_wp_error($response)) {
            return $response;
        }

        return $response;
    }

    /**
     * Make API request
     *
     * @param string $method
     * @param string $endpoint
     * @param array $data
     * @return array|WP_Error
     */
    private function make_request($method, $endpoint, $data = null) {
        $headers = array(
            'Content-Type' => 'application/json',
            'User-Agent' => 'Paypercut-WooCommerce/' . PAYPERCUT_BNPL_VERSION,
        );

        // Use Basic for token endpoint; Bearer elsewhere
        if ($this->is_token_endpoint($endpoint)) {
            $headers['Authorization'] = 'Basic ' . base64_encode($this->api_key . ':' . $this->api_secret);
        } else {
            $token = $this->resolve_bearer_token();
            if (is_wp_error($token)) {
                return $token;
            }
            $headers['Authorization'] = 'Bearer ' . $token;
        }

        $args = array(
            'method' => $method,
            'timeout' => 60,
            'headers' => $headers,
        );

        if ($data && in_array($method, array('POST', 'PUT', 'PATCH'))) {
            $args['body'] = wp_json_encode($data);
        }

        // No custom HMAC if using Basic auth per API reference

        $response = wp_remote_request($endpoint, $args);

        if (is_wp_error($response)) {
            $message = $response->get_error_message();
            // Retry once with alternate host on DNS or timeout errors
            $is_dns = (strpos($message, 'Could not resolve host') !== false);
            $is_timeout = (stripos($message, 'timed out') !== false || stripos($message, 'Timeout') !== false);
            if ($this->alt_api_url && ($is_dns || $is_timeout) && strpos($endpoint, $this->api_url) === 0) {
                $fallback_endpoint = $this->alt_api_url . substr($endpoint, strlen($this->api_url));
                $response = wp_remote_request($fallback_endpoint, $args);
                if (is_wp_error($response)) {
                    return new WP_Error('api_request_failed', $response->get_error_message());
                }
            } else {
                return new WP_Error('api_request_failed', $message);
            }
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);

        if ($response_code < 200 || $response_code >= 300) {
            /* translators: %d: HTTP status code */
            $error_message = sprintf(__('API request failed with status code %d', 'paypercut-bnpl-for-woocommerce'), $response_code);
            
            // Try to parse error message from response
            $error_data = json_decode($response_body, true);
            if (is_array($error_data)) {
                if (isset($error_data['message']) && is_string($error_data['message'])) {
                    $error_message = $error_data['message'];
                } elseif (isset($error_data['error']) && is_string($error_data['error'])) {
                    $error_message = $error_data['error'];
                }
            }

            return new WP_Error('api_error', $error_message, array(
                'status' => $response_code,
                'body' => $response_body,
                'json' => is_array($error_data) ? $error_data : null,
            ));
        }

        $decoded_response = json_decode($response_body, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            return new WP_Error('json_decode_error', __('Invalid JSON response from API', 'paypercut-bnpl-for-woocommerce'), array('body' => $response_body));
        }

        return $decoded_response;
    }

    /**
     * Validate API credentials
     *
     * @return bool|WP_Error
     */
    public function validate_credentials() {
        if (empty($this->api_key) || empty($this->api_secret)) {
            return new WP_Error('missing_credentials', __('API Key and Secret are required', 'paypercut-bnpl-for-woocommerce'));
        }

        $endpoint = $this->api_url . '/v1/auth/validate';
        
        $response = $this->make_request('GET', $endpoint);

        if (is_wp_error($response)) {
            return $response;
        }

        return true;
    }

    /**
     * Get supported currencies
     *
     * @return array|WP_Error
     */
    public function get_supported_currencies() {
        $endpoint = $this->api_url . '/v1/currencies';
        
        $response = $this->make_request('GET', $endpoint);

        if (is_wp_error($response)) {
            return $response;
        }

        return $response;
    }

    /**
     * Get BNPL limits for customer
     *
     * @param string $customer_email
     * @return array|WP_Error
     */
    public function get_customer_limits($customer_email) {
        $endpoint = $this->api_url . '/v1/bnpl/limits';
        
        $request_data = array(
            'customer_email' => $customer_email,
        );

        $response = $this->make_request('POST', $endpoint, $request_data);

        if (is_wp_error($response)) {
            return $response;
        }

        return $response;
    }

    /**
     * Log API request for debugging
     *
     * @param string $method
     * @param string $endpoint
     * @param array $data
     * @param array $response
     */
    private function log_request($method, $endpoint, $data, $response) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            $log_data = array(
                'method' => $method,
                'endpoint' => $endpoint,
                'request_data' => $data,
                'response' => $response,
                'timestamp' => current_time('mysql'),
            );

            // Debug logging removed for production
        }
    }

    /**
     * Get Bearer token if API requires one. If api_key is already a JWT (has 2 dots), return it.
     * Otherwise, return empty to allow Basic fallback.
     * @return string|null
     */
    private function get_bearer_token_if_possible() {
        if (!empty($this->api_key) && substr_count($this->api_key, '.') === 2) {
            return $this->api_key; // Looks like a JWT already
        }
        return null;
    }

    // removed env token helpers; token comes from admin input

    private function resolve_bearer_token() {
        // If API key is already a JWT, use it
        $jwtLike = $this->get_bearer_token_if_possible();
        if ($jwtLike) {
            return $jwtLike;
        }

        // Try from transient cache
        if (!empty($this->access_token)) {
            return $this->access_token;
        }
        $cache_key = 'paypercut_bnpl_token_' . md5($this->api_url . '|' . $this->api_key);
        $cached = get_transient($cache_key);
        if (!empty($cached) && is_string($cached)) {
            return $cached;
        }

        // Fetch new token
        $token_endpoint = rtrim($this->api_url, '/') . '/v1/auth/token';
        $args = array(
            'method' => 'POST',
            'timeout' => 30,
            'headers' => array(
                'Authorization' => 'Basic ' . base64_encode($this->api_key . ':' . $this->api_secret),
                'Content-Type' => 'application/json',
                'User-Agent' => 'Paypercut-WooCommerce/' . PAYPERCUT_BNPL_VERSION,
            ),
            'body' => wp_json_encode(array()),
        );

        $response = wp_remote_post($token_endpoint, $args);
        if (is_wp_error($response)) {
            return new WP_Error('auth_failed', $response->get_error_message());
        }
        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $json = json_decode($body, true);
        if ($code >= 200 && $code < 300 && is_array($json)) {
            $token = $json['access_token'] ?? ($json['token'] ?? null);
            if ($token) {
                set_transient($cache_key, $token, $this->token_ttl_seconds);
                return $token;
            }
        }
        /* translators: 1: HTTP status code, 2: Response body excerpt */
        return new WP_Error('auth_failed', sprintf(__('Failed to obtain access token (Bearer) from API. HTTP %1$d, body: %2$s', 'paypercut-bnpl-for-woocommerce'), $code, is_string($body) ? wp_html_excerpt($body, 200) : ''));
    }

    private function is_token_endpoint($endpoint) {
        return (strpos($endpoint, '/v1/auth/token') !== false);
    }
}

