<?php
/**
 * Paypercut BNPL Admin Class
 *
 * @package Paypercut_BNPL
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Paypercut BNPL Admin Class
 */
class Paypercut_BNPL_Admin {

    /**
     * Constructor
     */
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'admin_scripts'));
        add_action('wp_ajax_paypercut_validate_api', array($this, 'validate_api_credentials'));
        add_action('wp_ajax_paypercut_test_connection', array($this, 'test_api_connection'));
    }

    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_submenu_page(
            'woocommerce',
            __('Paypercut BNPL', 'paypercut-bnpl-for-woocommerce'),
            __('Paypercut BNPL', 'paypercut-bnpl-for-woocommerce'),
            'manage_woocommerce',
            'paypercut-bnpl-for-woocommerce',
            array($this, 'admin_page')
        );
    }

    /**
     * Enqueue admin scripts
     */
    public function admin_scripts($hook) {
        wp_enqueue_script('jquery');
        if ('woocommerce_page_paypercut-bnpl' === $hook) {
            wp_enqueue_script(
                'paypercut-bnpl-admin',
                PAYPERCUT_BNPL_PLUGIN_URL . 'assets/js/admin.js',
                array('jquery'),
                PAYPERCUT_BNPL_VERSION,
                true
            );
            wp_localize_script('paypercut-bnpl-admin', 'paypercut_ajax', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('paypercut_admin_nonce'),
            ));
            wp_enqueue_style(
                'paypercut-bnpl-admin',
                PAYPERCUT_BNPL_PLUGIN_URL . 'assets/css/admin.css',
                array(),
                PAYPERCUT_BNPL_VERSION
            );
        }
        if ($hook === 'woocommerce_page_wc-settings' && isset($_GET['tab']) && sanitize_text_field(wp_unslash($_GET['tab'])) === 'checkout') {
            wp_enqueue_script(
                'paypercut-bnpl-admin-payments',
                PAYPERCUT_BNPL_PLUGIN_URL . 'assets/js/admin-payments.js',
                array('jquery'),
                PAYPERCUT_BNPL_VERSION,
                true
            );
            wp_localize_script('paypercut-bnpl-admin-payments', 'paypercutAdminPayments', array(
                'icon' => PAYPERCUT_BNPL_PLUGIN_URL . 'assets/images/ppc-logo.png',
            ));
        }
    }

    /**
     * Admin page
     */
    public function admin_page() {
        $gateway = new Paypercut_BNPL_Gateway();
        $api_key = $gateway->get_option('api_key');
        $api_secret = $gateway->get_option('api_secret');
        $testmode = $gateway->get_option('testmode');
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Paypercut BNPL Settings', 'paypercut-bnpl-for-woocommerce'); ?></h1>
            
            <div class="paypercut-admin-container">
                <div class="paypercut-admin-main">
                    <div class="paypercut-card">
                        <h2><?php esc_html_e('API Configuration', 'paypercut-bnpl-for-woocommerce'); ?></h2>
                        <p><?php esc_html_e('Configure your Paypercut API credentials to enable BNPL payments.', 'paypercut-bnpl-for-woocommerce'); ?></p>
                        
                        <form method="post" action="options.php">
                            <?php
                            settings_fields('paypercut_bnpl_settings');
                            do_settings_sections('paypercut_bnpl_settings');
                            ?>
                            
                            <table class="form-table">
                                <tr>
                                    <th scope="row">
                                        <label for="paypercut_testmode"><?php esc_html_e('Test Mode', 'paypercut-bnpl-for-woocommerce'); ?></label>
                                    </th>
                                    <td>
                                        <input type="checkbox" id="paypercut_testmode" name="paypercut_testmode" value="1" <?php checked($testmode, 'yes'); ?>>
                                        <label for="paypercut_testmode"><?php esc_html_e('Enable test mode for sandbox testing', 'paypercut-bnpl-for-woocommerce'); ?></label>
                                        <p class="description"><?php esc_html_e('Use test API keys for development and testing.', 'paypercut-bnpl-for-woocommerce'); ?></p>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row">
                                        <label for="paypercut_api_key"><?php esc_html_e('API Key', 'paypercut-bnpl-for-woocommerce'); ?></label>
                                    </th>
                                    <td>
                                        <input type="text" id="paypercut_api_key" name="paypercut_api_key" value="<?php echo esc_attr($api_key); ?>" class="regular-text">
                                        <p class="description"><?php esc_html_e('Your Paypercut API Key from the dashboard.', 'paypercut-bnpl-for-woocommerce'); ?></p>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row">
                                        <label for="paypercut_api_secret"><?php esc_html_e('API Secret', 'paypercut-bnpl-for-woocommerce'); ?></label>
                                    </th>
                                    <td>
                                        <input type="password" id="paypercut_api_secret" name="paypercut_api_secret" value="<?php echo esc_attr($api_secret); ?>" class="regular-text">
                                        <p class="description"><?php esc_html_e('Your Paypercut API Secret from the dashboard.', 'paypercut-bnpl-for-woocommerce'); ?></p>
                                    </td>
                                </tr>
                            </table>
                            
                            <div class="paypercut-actions">
                                <button type="button" id="test-connection" class="button button-secondary">
                                    <?php esc_html_e('Test Connection', 'paypercut-bnpl-for-woocommerce'); ?>
                                </button>
                                <button type="button" id="validate-credentials" class="button button-secondary">
                                    <?php esc_html_e('Validate Credentials', 'paypercut-bnpl-for-woocommerce'); ?>
                                </button>
                                <?php submit_button(esc_html__('Save Settings', 'paypercut-bnpl-for-woocommerce')); ?>
                            </div>
                        </form>
                    </div>

                    <div class="paypercut-card">
                        <h2><?php esc_html_e('Payment Gateway Settings', 'paypercut-bnpl-for-woocommerce'); ?></h2>
                        <p><?php esc_html_e('Configure the payment gateway options in WooCommerce.', 'paypercut-bnpl-for-woocommerce'); ?></p>
                        <a href="<?php echo esc_url(admin_url('admin.php?page=wc-settings&tab=checkout&section=paypercut_bnpl')); ?>" class="button button-primary">
                            <?php esc_html_e('Configure Gateway', 'paypercut-bnpl-for-woocommerce'); ?>
                        </a>
                    </div>

                    <div class="paypercut-card">
                        <h2><?php esc_html_e('Order Management', 'paypercut-bnpl-for-woocommerce'); ?></h2>
                        <p><?php esc_html_e('View and manage BNPL orders.', 'paypercut-bnpl-for-woocommerce'); ?></p>
                        <a href="<?php echo esc_url(admin_url('edit.php?post_type=shop_order&paypercut_bnpl=1')); ?>" class="button button-secondary">
                            <?php esc_html_e('View BNPL Orders', 'paypercut-bnpl-for-woocommerce'); ?>
                        </a>
                    </div>
                </div>

                <div class="paypercut-admin-sidebar">
                    <div class="paypercut-card">
                        <h3><?php esc_html_e('Quick Links', 'paypercut-bnpl-for-woocommerce'); ?></h3>
                        <ul>
                            <li><a href="https://docs.paypercut.io" target="_blank"><?php esc_html_e('API Documentation', 'paypercut-bnpl-for-woocommerce'); ?></a></li>
                            <li><a href="https://dashboard.paypercut.io" target="_blank"><?php esc_html_e('Paypercut Dashboard', 'paypercut-bnpl-for-woocommerce'); ?></a></li>
                            <li><a href="https://support.paypercut.io" target="_blank"><?php esc_html_e('Support Center', 'paypercut-bnpl-for-woocommerce'); ?></a></li>
                        </ul>
                    </div>

                    <div class="paypercut-card">
                        <h3><?php esc_html_e('Plugin Information', 'paypercut-bnpl-for-woocommerce'); ?></h3>
                        <p><strong><?php esc_html_e('Version:', 'paypercut-bnpl-for-woocommerce'); ?></strong> <?php echo esc_html(PAYPERCUT_BNPL_VERSION); ?></p>
                        <p><strong><?php esc_html_e('WooCommerce:', 'paypercut-bnpl-for-woocommerce'); ?></strong> <?php echo esc_html(WC()->version); ?></p>
                        <p><strong><?php esc_html_e('WordPress:', 'paypercut-bnpl-for-woocommerce'); ?></strong> <?php echo esc_html(get_bloginfo('version')); ?></p>
                    </div>

                    <div class="paypercut-card">
                        <h3><?php esc_html_e('Status', 'paypercut-bnpl-for-woocommerce'); ?></h3>
                        <div id="connection-status">
                            <p><?php esc_html_e('Click "Test Connection" to check API status.', 'paypercut-bnpl-for-woocommerce'); ?></p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Validate API credentials via AJAX
     */
    public function validate_api_credentials() {
        check_ajax_referer('paypercut_admin_nonce', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_die(esc_html__('Insufficient permissions.', 'paypercut-bnpl-for-woocommerce'));
        }

        $api_key = isset($_POST['api_key']) ? sanitize_text_field(wp_unslash($_POST['api_key'])) : '';
        $api_secret = isset($_POST['api_secret']) ? sanitize_text_field(wp_unslash($_POST['api_secret'])) : '';
        $testmode = isset($_POST['testmode']) && $_POST['testmode'] === 'true';

        if (empty($api_key) || empty($api_secret)) {
            wp_send_json_error(esc_html__('API Key and Secret are required.', 'paypercut-bnpl-for-woocommerce'));
        }

        $api = new Paypercut_BNPL_API($api_key, $api_secret, $testmode);
        $result = $api->validate_credentials();

        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        }

        wp_send_json_success(esc_html__('API credentials are valid.', 'paypercut-bnpl-for-woocommerce'));
    }

    /**
     * Test API connection via AJAX
     */
    public function test_api_connection() {
        check_ajax_referer('paypercut_admin_nonce', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_die(esc_html__('Insufficient permissions.', 'paypercut-bnpl-for-woocommerce'));
        }

        $api_key = isset($_POST['api_key']) ? sanitize_text_field(wp_unslash($_POST['api_key'])) : '';
        $api_secret = isset($_POST['api_secret']) ? sanitize_text_field(wp_unslash($_POST['api_secret'])) : '';
        $testmode = isset($_POST['testmode']) && $_POST['testmode'] === 'true';

        if (empty($api_key) || empty($api_secret)) {
            wp_send_json_error(esc_html__('API Key and Secret are required.', 'paypercut-bnpl-for-woocommerce'));
        }

        $api = new Paypercut_BNPL_API($api_key, $api_secret, $testmode);
        
        // Test with a simple API call
        $currencies = $api->get_supported_currencies();

        if (is_wp_error($currencies)) {
            wp_send_json_error($currencies->get_error_message());
        }

        wp_send_json_success(__('API connection successful.', 'paypercut-bnpl-for-woocommerce'));
    }
}

