/**
 * Paypercut BNPL Checkout JavaScript
 */

jQuery(document).ready(function($) {
    'use strict';

    // Initialize BNPL functionality
    initPaypercutBNPL();

    function initPaypercutBNPL() {
        // Handle payment method selection
        $('input[name="payment_method"]').on('change', function() {
            if ($(this).val() === 'paypercut_bnpl') {
                showBNPLInfo();
                validateBNPLTerms();
            } else {
                hideBNPLInfo();
            }
        });

        // Handle terms checkbox change
        $('input[name="paypercut_bnpl_terms"]').on('change', function() {
            validateBNPLTerms();
        });

        // Calculate and display installment information
        calculateInstallments();

        // Handle cart updates
        $(document.body).on('updated_checkout', function() {
            calculateInstallments();
        });
    }

    function showBNPLInfo() {
        $('.paypercut-bnpl-info').slideDown(300);
        $('.paypercut-bnpl-terms').slideDown(300);
    }

    function hideBNPLInfo() {
        $('.paypercut-bnpl-info').slideUp(300);
        $('.paypercut-bnpl-terms').slideUp(300);
    }

    function calculateInstallments() {
        var total = getCartTotal();
        var installmentAmount = total / 4; // 4 installments
        
        $('.paypercut-installment-amount').text(formatCurrency(installmentAmount));
        $('.paypercut-total-amount').text(formatCurrency(total));
        
        // Update installment plan
        updateInstallmentPlan(total, installmentAmount);
    }

    function getCartTotal() {
        var total = 0;
        
        // Try to get total from various WooCommerce elements
        if ($('.woocommerce-Price-amount').length) {
            var totalText = $('.woocommerce-Price-amount').last().text();
            total = parseFloat(totalText.replace(/[^0-9.-]+/g, ''));
        }
        
        // Fallback to order total if available
        if (total === 0 && $('#order_total').length) {
            total = parseFloat($('#order_total').val());
        }
        
        return total || 0;
    }

    function formatCurrency(amount) {
        // Get currency symbol from WooCommerce
        var currencySymbol = $('.woocommerce-Price-currencySymbol').first().text() || '$';
        return currencySymbol + amount.toFixed(2);
    }

    function updateInstallmentPlan(total, installmentAmount) {
        var planHtml = '<div class="paypercut-bnpl-installment-plan">';
        planHtml += '<h4>Payment Schedule</h4>';
        
        var today = new Date();
        for (var i = 0; i < 4; i++) {
            var paymentDate = new Date(today);
            paymentDate.setMonth(paymentDate.getMonth() + i);
            
            var dateStr = paymentDate.toLocaleDateString('en-US', {
                month: 'short',
                day: 'numeric',
                year: 'numeric'
            });
            
            var label = i === 0 ? 'Today' : 'Month ' + (i + 1);
            
            planHtml += '<div class="installment-item">';
            planHtml += '<span class="installment-date">' + label + ' (' + dateStr + ')</span>';
            planHtml += '<span class="installment-amount">' + formatCurrency(installmentAmount) + '</span>';
            planHtml += '</div>';
        }
        
        planHtml += '</div>';
        
        $('.paypercut-bnpl-info').html(planHtml);
    }

    function validateBNPLTerms() {
        var termsChecked = $('input[name="paypercut_bnpl_terms"]').is(':checked');
        var paymentMethod = $('input[name="payment_method"]:checked').val();
        
        if (paymentMethod === 'paypercut_bnpl' && !termsChecked) {
            $('.paypercut-bnpl-terms').addClass('error');
            return false;
        } else {
            $('.paypercut-bnpl-terms').removeClass('error');
            return true;
        }
    }

    // Override checkout form submission
    $('form.checkout').on('checkout_place_order', function() {
        var paymentMethod = $('input[name="payment_method"]:checked').val();
        
        if (paymentMethod === 'paypercut_bnpl') {
            if (!validateBNPLTerms()) {
                $('html, body').animate({
                    scrollTop: $('.paypercut-bnpl-terms').offset().top - 100
                }, 500);
                
                $('.paypercut-bnpl-terms').addClass('shake');
                setTimeout(function() {
                    $('.paypercut-bnpl-terms').removeClass('shake');
                }, 500);
                
                return false;
            }
        }
        
        return true;
    });

    // Add shake animation CSS
    $('<style>')
        .prop('type', 'text/css')
        .html(`
            .paypercut-bnpl-terms.error {
                border-color: #dc3545;
                background-color: #f8d7da;
            }
            .paypercut-bnpl-terms.shake {
                animation: shake 0.5s ease-in-out;
            }
            @keyframes shake {
                0%, 100% { transform: translateX(0); }
                25% { transform: translateX(-5px); }
                75% { transform: translateX(5px); }
            }
        `)
        .appendTo('head');

    // Handle payment processing
    $(document.body).on('checkout_error', function(event, error_message) {
        if (error_message.includes('Paypercut') || error_message.includes('BNPL')) {
            $('.paypercut-bnpl-info').addClass('error');
            setTimeout(function() {
                $('.paypercut-bnpl-info').removeClass('error');
            }, 3000);
        }
    });

    // Show loading state during payment processing
    $(document.body).on('checkout_processing', function() {
        var paymentMethod = $('input[name="payment_method"]:checked').val();
        
        if (paymentMethod === 'paypercut_bnpl') {
            $('.paypercut-bnpl-info').addClass('processing');
            $('.paypercut-bnpl-info').append('<div class="paypercut-bnpl-loading">Redirecting to Paypercut...</div>');
        }
    });

    // Hide loading state after payment processing
    $(document.body).on('checkout_complete', function() {
        $('.paypercut-bnpl-info').removeClass('processing');
        $('.paypercut-bnpl-loading').remove();
    });

    // Handle Paypercut redirect flow
    function handlePaypercutRedirect() {
        var paymentMethod = $('input[name="payment_method"]:checked').val();
        
        if (paymentMethod === 'paypercut_bnpl') {
            // Show redirect message
            $('.paypercut-bnpl-info').html(
                '<div class="paypercut-bnpl-redirect">' +
                '<h4>Redirecting to Paypercut</h4>' +
                '<p>You will be redirected to Paypercut to complete your payment. Please do not close this window.</p>' +
                '<div class="paypercut-bnpl-loading">Processing...</div>' +
                '</div>'
            );
        }
    }

    // Add processing state CSS
    $('<style>')
        .prop('type', 'text/css')
        .html(`
            .paypercut-bnpl-info.processing {
                opacity: 0.7;
                pointer-events: none;
            }
            .paypercut-bnpl-loading {
                text-align: center;
                padding: 20px;
                color: #0073aa;
                font-weight: 500;
            }
        `)
        .appendTo('head');

    // Handle responsive design
    function handleResponsive() {
        if ($(window).width() <= 768) {
            $('.paypercut-bnpl-info').addClass('mobile');
        } else {
            $('.paypercut-bnpl-info').removeClass('mobile');
        }
    }

    $(window).on('resize', handleResponsive);
    handleResponsive();

    // Add mobile styles
    $('<style>')
        .prop('type', 'text/css')
        .html(`
            @media (max-width: 768px) {
                .paypercut-bnpl-info.mobile .installment-item {
                    flex-direction: column;
                    align-items: flex-start;
                    gap: 5px;
                }
                .paypercut-bnpl-info.mobile .installment-amount {
                    font-size: 16px;
                    font-weight: 700;
                }
            }
        `)
        .appendTo('head');
});

