<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

// Ensure WooCommerce is loaded before defining the class
function paymgafo_myameria_init_gateway_class()
{

    class PAYMGAFOAMERIA_Gateway extends WC_Payment_Gateway
    {

        public $clientId;
        public $clientSecret;
        public $merchantName;
        public $merchantId;
        public $terminalId;
        public $callback_scheme;
        public $mode;
        public function __construct()
        {
            $this->supports = [
                'products',
                'refunds',
                'subscriptions',
                'subscription_cancellation',
                'subscription_suspension',
                'subscription_reactivation',
                'subscription_amount_changes',
                'subscription_date_changes',
                'subscription_payment_method_change',
                'subscription_payment_method_change_customer',
                'subscription_payment_method_change_admin',
                'multiple_subscriptions',
                'gateway_scheduled_payments',

            ];

            $this->id = 'paymgafo_myameria_gateway'; // Payment gateway ID
            $this->has_fields = false; // No custom fields on checkout page
            $this->method_title = 'MyAmeria Pay';
            $this->method_description = 'Pay securely via MyAmeria Pay.';
            $this->icon = plugins_url('assets/img/MyAmeriaPay.svg', dirname(__FILE__));

            // Load the settings.
            $this->init_form_fields();
            $this->init_settings();

            // Define user settings variables
            $this->title = $this->get_option('title');
            $this->description = $this->get_option('description');

            $this->clientId = $this->get_option('clientId');
            $this->clientSecret = $this->get_option('clientSecret');
            $this->merchantName = $this->get_option('merchantName');
            $this->merchantId = $this->get_option('merchantId');
            $this->terminalId = $this->get_option('terminalId');
            $this->callback_scheme = $this->get_option('callback_scheme');
            $this->mode = $this->get_option('mode', 'test');
            add_filter('woocommerce_settings_api_form_fields_' . $this->id, array($this, 'maybe_disable_form_fields'));
            // Save admin settings
            add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));

            // Handle the payment process
            add_action('woocommerce_thankyou_' . $this->id, array($this, 'process_redirect'));
            add_action('woocommerce_init', array($this, 'check_hpos_compatibility'));
        }
        public function maybe_disable_form_fields($fields)
        {
            $enabled = $this->get_option('enabled');

            if ($enabled !== 'yes') {
                foreach ($fields as $key => &$field) {
                    if ($key === 'enabled') {
                        continue;
                    }

                    $field['custom_attributes']['readonly'] = 'readonly';
                    $field['custom_attributes']['style'] = 'background-color: rgb(233 233 233);
    opacity: 0.4;';
                }
            }

            return $fields;
        }

        public function check_hpos_compatibility()
        {
            if (
                class_exists('\Automattic\WooCommerce\Utilities\FeaturesUtil') &&
                \Automattic\WooCommerce\Utilities\FeaturesUtil::feature_is_enabled('custom_order_tables')
            ) {
                $this->enabled = 'no';
                $this->update_option('enabled', 'no');
            }
        }
        // Initialize form fields
        public function init_form_fields()
        {


            $all_fields = [

                'enabled' => [
                    'title'   => 'Enable/Disable',
                    'type'    => 'checkbox',
                    'label'   => 'Enable MyAmeria Pay',
                    'default' => 'no', // Default to disabled

                ],


                'title' => [
                    'title'       => 'Title',
                    'type'        => 'text',
                    'description' => 'This controls the title seen by the user during checkout.',
                    'default'     => 'MyAmeria Pay',

                ],
                'description' => [
                    'title'       => 'Description',
                    'type'        => 'textarea',
                    'description' => 'Description of the payment method displayed during checkout.',
                    'default'     => 'Pay securely via MyAmeria Pay.',

                ],
                'clientId' => [
                    'title'       => 'Client ID',
                    'type'        => 'text',
                    'description' => 'Enter your Client ID for MyAmeria Pay.',


                ],
                'clientSecret' => [
                    'title'       => 'Client Secret',
                    'type'        => 'password',
                    'description' => 'Enter your Client Secret for MyAmeria Pay.',


                ],
                'merchantName' => [
                    'title'       => 'Merchant Name',
                    'type'        => 'text',
                    'description' => 'Enter your Merchant Name for MyAmeria Pay.',


                ],
                'merchantId' => [
                    'title'       => 'Merchant ID',
                    'type'        => 'text',
                    'description' => 'Enter your Merchant ID for MyAmeria Pay.',


                ],
                'terminalId' => [
                    'title'       => 'Terminal ID',
                    'type'        => 'text',
                    'description' => 'Enter your Terminal ID for MyAmeria Pay.',


                ],
                'mode' => [
                    'title'       => 'Mode',
                    'type'        => 'select',
                    'description' => 'Choose Test or Production mode. Switching changes API endpoints.',
                    'default'     => 'test',
                    'options'     => [
                        'test'       => 'Test',
                        'production' => 'Production',
                    ],
                ],

                'callback_scheme' => [
                    'title'       => 'Callback Scheme',
                    'type'        => 'text',
                    'description' => 'Enter Callback scheme without https://',
                    'default' => preg_replace('#^https?://#', '', home_url() ?? ''),

                ],


            ];

            $this->form_fields =  $all_fields;
        }
        public function admin_options()
        {
?>
            <div class="myameria-settings-container">
                <!-- Left Column - Settings Fields -->
                <div class="myameria-settings-fields" style="flex: 2;">
                    <table class="form-table">
                        <?php echo wp_kses_post($this->generate_settings_html() ?? ''); ?>
                    </table>
                </div>

                <!-- Right Column - Company Info -->
                <div class="myameria-company-info" style="flex: 1;  padding: 20px;">
                    <?php echo wp_kses_post($this->get_author_info_html() ?? ''); ?>
                </div>
            </div>


        <?php
        }
        public function get_author_info_html()
        {
            $author_image_url = plugin_dir_url(__FILE__) . '../assets/img/author-image.jpg'; // Path to the author image
            $author_name = 'Ameriabank';
            $author_url = 'https://ameriabank.am/';
            $my_ameria_logo_url = plugin_dir_url(__FILE__) . '../assets/img/MyAmeriaPay.svg';

            $support_email = 'dps.products@ameriabank.am';
            $support_phone = '010/011 561111';


            ob_start(); ?>
            <div style=" margin-bottom: 20px;">
                <img src="<?php echo esc_url($my_ameria_logo_url); ?>" alt="My Ameria logo" style="max-width: 250px; height: auto;">
            </div>
            <h2>
                Payment System
            </h2>
            <div style="margin: 15px 0;">

                <p><strong><?php esc_html_e('Phone:', 'payment-gateway-for-myameriapay'); ?></strong><br>
                    <a href="tel:+37410561111"><?php echo esc_html($support_phone); ?></a>
                </p>
                <p><strong><?php esc_html_e('Email:', 'payment-gateway-for-myameriapay'); ?></strong><br>
                    <a href="mailto:<?php echo esc_attr($support_email); ?>"><?php echo esc_html($support_email); ?></a>
                </p>

            </div>
            <h3><?php esc_html_e('Support Information', 'payment-gateway-for-myameriapay'); ?></h3>
            <p><?php esc_html_e('For technical support:', 'payment-gateway-for-myameriapay'); ?></p>
            <h3>
                <a href="<?php echo esc_url($author_url) ?>" target="_blank"><?php echo esc_html($author_name); ?></a>
            </h3>
            <div style="margin: 15px 0;">
                <p><strong><?php esc_html_e('Email:', 'payment-gateway-for-myameriapay'); ?></strong><br>
                    <a href="mailto:<?php echo esc_attr($support_email); ?>"><?php echo esc_html($support_email); ?></a>
                </p>

                <p><strong><?php esc_html_e('Phone:', 'payment-gateway-for-myameriapay'); ?></strong><br>
                    <a href="tel:+37410561111"><?php echo esc_html($support_phone); ?></a>
                </p>
            </div>



<?php
            return ob_get_clean();
        }


        public function get_api_urls()
        {
            if ($this->mode === 'production') {
                return [
                    'auth'    => 'https://auth-external.prd.k8s.ameriabank.am/realms/MyAmeriaPay/protocol/openid-connect/token',
                    'payment' => 'https://dgpayapi.ameriabank.am/api/v1/Payment/CreatePayment',
                    'status'  => 'https://dgpayapi.ameriabank.am/api/v1/Payment/Status',
                ];
            } else {
                return [
                    'auth'    => 'https://auth-external.test.k8s.ameriabank.am/realms/MyAmeriaPay/protocol/openid-connect/token',
                    'payment' => 'https://dgpayapi.test.ameriabank.am/api/v1/Payment/CreatePayment',
                    'status'  => 'https://dgpayapi.test.ameriabank.am/api/v1/Payment/Status',
                ];
            }
        }


        public function process_payment($order_id)
        {
            $order = wc_get_order($order_id);
            $order_id_str = (string) $order_id;
            $clientId = $this->clientId;
            $clientSecret = $this->clientSecret;
            $merchantName = $this->merchantName;
            $merchantId = $this->merchantId;
            $terminalId = $this->terminalId;
            $required_fields = [
                'Client ID'       =>  $clientId,
                'Client Secret'   => $clientSecret,
                'Merchant Name'   => $merchantName,
                'Merchant ID'     => $merchantId,
                'Terminal ID'     => $terminalId,
                'Callback Scheme' => $this->callback_scheme,
            ];

            $missing = [];

            foreach ($required_fields as $label => $value) {
                if (empty($value)) {
                    $missing[] = $label;
                }
            }

            if (!empty($missing)) {
                wc_add_notice(
                    'Payment gateway is not properly configured. Missing: ' . implode(', ', $missing),
                    'error'
                );
                return [
                    'result'   => 'failure'

                ];
            }

            $transactionAmount = (int)$order->get_total();
            // Step 1: Generate the auth token
            $token = $this->generate_auth_token($clientId, $clientSecret);
            if (strpos($token, 'Error') !== false) {
                wc_add_notice($token, 'error');
                return [
                    'result'   => 'failure'

                ];
            }

            $order->update_meta_data('_myameria_auth_token', $token);
            $order->save();
            // Step 2: Create the payment
            $payment_response = $this->create_payment($token, $transactionAmount, $order_id_str, $merchantId);
            if (isset($payment_response['errorMessages'])) {
                wc_add_notice('Payment creation failed:' . ($payment_response['errorMessages'] ?? 'Unknown error'), 'error');
                return [
                    'result'   => 'failure'

                ];
            } else {
                $order->update_status('on-hold', 'Awaiting MyAmeria payment');

                // Reduce stock levels
                wc_reduce_stock_levels($order_id);

                $transactionAmount = $order->get_total();
                $transactionId = $order->get_id();

                $callback_scheme = $this->callback_scheme;
                if ($this->mode === 'production') {
                    $base_redirect = 'https://app.myameria.am/pay/getpaymentoptions?';
                } else {
                    $base_redirect = 'https://dgpay-ecommerce.test.ameriabank.am/pay/getpaymentoptions.html?';
                }
                // Construct the redirect URL
                $redirect_url = $base_redirect . http_build_query([
                    'merchantName' => $merchantName,
                    'transactionAmount' => number_format((float)$transactionAmount, 2, '.', ''), // Ensure correct decimal format
                    'transactionId' => $transactionId,
                    'merchantId' => $merchantId,
                    'terminalId' => $terminalId,
                    'callbackscheme' => $callback_scheme,
                ]);
                // Return success and redirect to AmeriaBank payment options
                return [
                    'result'   => 'success',
                    'redirect' => $redirect_url
                ];
            }

            // Mark order as on-hold (awaiting payment)
        }

        public function generate_auth_token($clientId, $clientSecret)
        {

            $urls = $this->get_api_urls();
            $authUrl = $urls['auth'];

            $response = wp_remote_post($authUrl, [
                'body' => http_build_query([
                    'grant_type' => 'client_credentials',
                    'client_id' => $clientId,
                    'client_secret' => $clientSecret,
                ]),
                'headers' => [
                    'Content-Type' => 'application/x-www-form-urlencoded'
                ],
                'timeout' => 30,

            ]);

            if (is_wp_error($response)) {
                $error_message = 'WP Error: ' . $response->get_error_message();

                wc_add_notice($error_message, 'error'); // Display the error on the front end
                return false; // Return false to indicate failure
            }

            $response_body = wp_remote_retrieve_body($response);

            $response_data = json_decode($response_body, true);

            if (isset($response_data['error'])) {

                $error_message = 'Error fetching token: ' . $response_data['error_description'];
                wc_add_notice($error_message, 'error'); // Display the error on the front end
                return false; // Return false to indicate failure
            }

            return $response_data['access_token']; // Return false if token is not set
        }


        public function create_payment($token, $transactionAmount, $transactionId, $merchantId)
        {
            $urls = $this->get_api_urls();
            $paymentUrl = $urls['payment'];
            $request_body = [
                'transactionAmount' => $transactionAmount,
                'transactionId'     => $transactionId,
                'merchantId'        => $merchantId,
            ];

            $response = wp_remote_post($paymentUrl, [
                'body'    => wp_json_encode($request_body),
                'headers' => [
                    'Authorization' => 'Bearer ' . $token,
                    'Content-Type' => 'application/json',
                ],
                'timeout' => 90,
            ]);


            if (is_wp_error($response)) {
                $error_message = $response->get_error_message();
                wc_add_notice($error_message, 'error');
                return ['error' => $error_message]; // Return error
            }

            // Log the response for debugging
            $response_body = wp_remote_retrieve_body($response);
            $response_code = wp_remote_retrieve_response_code($response);

            // Log response

            $response_data = json_decode($response_body, true);
            if (isset($response_data['error'])) {

                wc_add_notice('Error creating payment: ' . $response_data['errorMessage'], 'error');
                return ['error' => 'Error creating payment: ' . $response_data['errorMessages']];
            }

            return $response_data; // Return successful response
        }



        public function process_redirect($order_id)
        {
            $order = wc_get_order($order_id);
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- External payment gateway can't pass nonce
            if ($order && isset($_GET['payment_status']) && $_GET['payment_status'] === 'success') {
                wp_redirect($order->get_checkout_order_received_url());
                exit;
            }
        }
    }
}
