<?php

namespace Yetsa\PaymentErrorManager;

// Exit if accessed directly
defined('ABSPATH') || exit;

class Paystack_Error_Manager
{
    private static $instance;

    // triggers default payment error handler
    protected function handle_payment_error($order)
    {
        $options = get_option('pemanager_settings', []);

        include_once(PaymentErrorManager_DIR . 'payment-error-handler.php');

        $payment_error_handler = new \Yetsa\PaymentErrorManager\Error_Handler;

        $payment_error_handler->auto_complete_order($order, $options);
        $payment_error_handler->send_customer_email($order, $options);
        $payment_error_handler->send_admin_email($order, $options);
    }

    // checks if at least one of the settings field was activated
    protected function error_manager_settings_activated()
    {
        $options = get_option('pemanager_settings');
        return !empty($options);
    }

    public function setup_paystack_webhook()
    {
        // before anything, check payment error manager settings
        if (!$this->error_manager_settings_activated()) {
            return;
        }

        $input = @file_get_contents('php://input');
        $payload = json_decode($input, true);
        $signature = '';

        if (isset($_SERVER['HTTP_X_PAYSTACK_SIGNATURE'])) {
            $signature = sanitize_text_field(wp_unslash($_SERVER['HTTP_X_PAYSTACK_SIGNATURE']));
        }

        // validate signature
        $paystack_settings = get_option('woocommerce_paystack_settings');
        $secret = isset($paystack_settings['secret_key']) ? $paystack_settings['secret_key'] : '';
        $hash = hash_hmac('sha512', $input, $secret);

        if (!hash_equals($hash, $signature)) {
            status_header(400);
            exit('Invalid signature.');
        }

        status_header(200);

        // process event
        if ($payload['event'] === 'charge.failed') {

            $reference = $payload['data']['reference'] ?? '';

            // lookup order by reference meta, needs to sync how Paystack stores order refs
            $order_id = wc_get_order_id_by_order_key($reference) ?: 0;

            if (!$order_id) return;

            $order = wc_get_order($order_id);

            if (!$order || $order->get_payment_method() !== 'paystack') return;

            // handle payment error
            $this->handle_payment_error($order);
        }

        exit;
    }

    // manage paystack errors
    public function register_error_handler()
    {
        add_action('woocommerce_api_tbz_wc_paystack_webhook', [$this, 'setup_paystack_webhook']);
    }

    public static function get_instance()
    {
        if (!self::$instance) {
            self::$instance = new self();
        }

        return self::$instance;
    }
}
