<?php
/**
 * Blocks integration for PayGenius Redirect
 *
 * @package PayGenius
 */

use Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * WC_Paygenius_Redirect_Blocks
 */
class WC_Paygenius_Redirect_Blocks extends AbstractPaymentMethodType {

	/**
	 * Gateway instance.
	 *
	 * @var WC_Paygenius_Redirect
	 */
	protected $gateway;

	/**
	 * Get payment method name.
	 *
	 * @return string
	 */
	public function get_name() {
		return 'paygenius-redirect';
	}

	/**
	 * Initialize gateway.
	 */
	public function initialize() {
		if ( ! class_exists( 'WC_Paygenius_Redirect' ) ) {
			require_once PAYGENIUS_PLUGIN_DIR . 'lib/class-wc-paygenius-redirect.php';
		}
		$this->gateway = new WC_Paygenius_Redirect();
	}

	/**
	 * Check if payment method is active.
	 *
	 * @return bool
	 */
	public function is_active() {
		if ( ! $this->gateway ) {
			$this->initialize();
		}
		if ( ! $this->gateway ) {
			return false;
		}
		// Use is_available() if it exists, otherwise fall back to basic checks.
		if ( method_exists( $this->gateway, 'is_available' ) ) {
			// Note: is_available() checks cart context, which may not be available during registration.
			// So we do basic checks here and let WooCommerce Blocks use is_available() during checkout.
			$enabled   = 'yes' === $this->gateway->get_option( 'enabled', 'no' );
			$supported = method_exists( $this->gateway, 'is_supported' ) ? $this->gateway->is_supported() : true;
			$activated = method_exists( $this->gateway, 'is_activated' ) ? $this->gateway->is_activated() : true;
			return $enabled && $supported && $activated;
		}
		// Fallback for older gateway implementations.
		$enabled   = 'yes' === $this->gateway->get_option( 'enabled', 'no' );
		$supported = method_exists( $this->gateway, 'is_supported' ) ? $this->gateway->is_supported() : true;
		$activated = method_exists( $this->gateway, 'is_activated' ) ? $this->gateway->is_activated() : true;
		return $enabled && $supported && $activated;
	}

	/**
	 * Get payment method script handles.
	 *
	 * @return array
	 */
	public function get_payment_method_script_handles() {
		$handle      = 'wc-paygenius-redirect-blocks';
		$script_path = PAYGENIUS_PLUGIN_DIR . 'assets/js/paygenius-blocks.js';
		$script_url  = PAYGENIUS_PLUGIN_URL . 'assets/js/paygenius-blocks.js';

		if ( file_exists( $script_path ) ) {
			$version = file_exists( $script_path ) ? filemtime( $script_path ) : '1.0.0';
			$deps    = array(
				'wp-element',
				'wp-i18n',
				'wp-hooks',
				'wp-data',
				'wp-url',
				'wc-settings',
				'wc-blocks-registry',
				'wc-blocks-data',
				'wc-cart-checkout-base',
			);

			wp_register_script(
				'wc-paygenius-redirect-blocks',
				$script_url,
				$deps,
				$version,
				true
			);

			wp_set_script_translations(
				'wc-paygenius-redirect-blocks',
				'paygenius-hosted-payment-gateway',
				plugin_dir_path( PAYGENIUS_PLUGIN_FILE ) . 'languages'
			);
		}

		return array( $handle );
	}

	/**
	 * Get payment method data.
	 *
	 * @return array
	 */
	public function get_payment_method_data() {
		if ( ! $this->gateway ) {
			$this->initialize();
		}

		return array(
			'title'                 => esc_html( $this->gateway->title ),
			'description'           => esc_html( $this->gateway->description ),
			'placeOrderButtonLabel' => esc_html( $this->gateway->order_button_text ? $this->gateway->order_button_text : __( 'Proceed to PayGenius', 'paygenius-hosted-payment-gateway' ) ),
			'supports'              => array( 'features' => array( 'products' ) ),
		);
	}

	/**
	 * Process payment.
	 *
	 * @param \WC_Order $order Order.
	 * @return array
	 * @throws \Exception On failure.
	 */
	public function process_payment( $order ) {
		if ( ! $order instanceof \WC_Order ) {
			$order = wc_get_order( $order );
		}
		if ( ! $order ) {
			throw new \Exception( esc_html__( 'Invalid order.', 'paygenius-hosted-payment-gateway' ) );
		}

		if ( ! $this->gateway ) {
			$this->initialize();
		}

		$redirect = $this->gateway->get_redirect_url( $order->get_id() );

		if ( ! $redirect ) {
			throw new \Exception( esc_html__( 'Unable to initiate PayGenius payment.', 'paygenius-hosted-payment-gateway' ) );
		}

		return array(
			'type'         => 'redirect',
			'redirect_url' => esc_url_raw( $redirect ),
		);
	}
}
