<?php
/**
 * PayGenius WebClient
 *
 * @package PayGenius
 */

namespace PayGenius;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * WebClient
 */
class WebClient {

	/**
	 * Endpoint prefix.
	 *
	 * @var string
	 */
	private $endpoint_prefix;

	/**
	 * API token.
	 *
	 * @var string
	 */
	private $token;

	/**
	 * API secret.
	 *
	 * @var string
	 */
	private $secret;

	/**
	 * Constructor.
	 *
	 * @param string $endpoint_prefix Endpoint prefix.
	 * @param string $token API token.
	 * @param string $secret API secret.
	 */
	public function __construct( $endpoint_prefix, $token, $secret ) {
		$this->endpoint_prefix = $endpoint_prefix;
		$this->token           = $token;
		$this->secret          = $secret;
	}

	/**
	 * Create payment redirect.
	 *
	 * @param string $name Name.
	 * @param string $surname Surname.
	 * @param int    $amount Amount in cents.
	 * @param string $currency Currency code.
	 * @param string $email Email.
	 * @param string $reference Reference.
	 * @param string $page Page ID.
	 * @param string $success_url Success URL.
	 * @param string $cancel_url Cancel URL.
	 * @param string $error_url Error URL.
	 * @param string $webhook_url Webhook URL.
	 * @return array|WP_Error
	 */
	public function create( $name, $surname, $amount, $currency, $email, $reference, $page, $success_url, $cancel_url, $error_url, $webhook_url ) {
		$transaction = array(
			'reference' => $reference,
			'currency'  => $currency,
			'amount'    => $amount,
		);

		// If currency is not ZAR, add displayCurrency and hardcode currency to ZAR.
		if ( 'ZAR' !== strtoupper( $currency ) ) {
			$transaction['displayCurrency'] = $currency;
			$transaction['currency']        = 'ZAR';
		}

		return $this->send(
			'redirect/create',
			array(
				'transaction' => $transaction,
				'consumer'    => array(
					'name'    => $name,
					'surname' => $surname,
					'email'   => $email,
				),
				'urls'        => array(
					'cancel'  => $cancel_url,
					'error'   => $error_url,
					'success' => $success_url,
					'webhook' => $webhook_url,
				),
				'page'        => $page,
			)
		);
	}

	/**
	 * Get payment info.
	 *
	 * @param string $reference Reference.
	 * @return array|WP_Error
	 */
	public function info( $reference ) {
		return $this->send( 'payment/' . $reference, null );
	}

	/**
	 * Get pages.
	 *
	 * @return array|WP_Error
	 */
	public function get_pages() {
		return $this->send( 'page/list' );
	}

	/**
	 * Send HTTP request.
	 *
	 * @param string      $url URL.
	 * @param array|null  $body Request body.
	 * @return array|WP_Error
	 */
	private function send( $url, $body = null ) {
		if ( null !== $body ) {
			$body = wp_json_encode( $body );
		}

		$url = $this->endpoint_prefix . $url;

		$request = new \WP_Http();

		$request_content = array(
			'headers' => array(
				'X-Token'     => $this->token,
				'X-Signature' => $this->sign( $url, null === $body ? '' : $body ),
			),
		);

		if ( null !== $body ) {
			$request_content['method']                    = 'POST';
			$request_content['body']                      = $body;
			$request_content['headers']['Content-Type']   = 'application/json';
			$request_content['headers']['Content-Length'] = strlen( $body );
		}

		$result = $request->request( $url, $request_content );
		if ( $result instanceof \WP_Error ) {
			return $result;
		}

		return $result;
	}

	/**
	 * Sign request.
	 *
	 * @param string $endpoint Endpoint.
	 * @param string $request Request body.
	 * @return string
	 */
	private function sign( $endpoint, $request = '' ) {
		$to_sign = trim( $endpoint . "\n" . $request );
		return hash_hmac( 'sha256', $to_sign, $this->secret );
	}
}
