<?php
/**
 * PayGenius Redirect Payment Gateway
 *
 * @package PayGenius
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * WC_Paygenius_Redirect
 */
class WC_Paygenius_Redirect extends WC_Payment_Gateway {

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->id                 = 'paygenius-redirect';
		$this->icon               = PAYGENIUS_PLUGIN_URL . 'images/paygenius_logo_black.png';
		$this->has_fields         = false;
		$this->method_title       = 'PayGenius Redirect';
		$this->method_description = 'PayGenius payment gateway (hosted interface)';
		$this->order_button_text  = esc_html__( 'Proceed to PayGenius', 'paygenius-hosted-payment-gateway' );

		$this->title             = $this->get_option( 'title', 'PayGenius Secure payment processing via Credit Card or EFT for South Africa' );
		$this->description       = $this->get_option( 'description', 'Pay by Credit Card, Debit Card, Instant EFT' );
		$this->order_button_text = $this->get_option( 'button-text', 'Place order' );

		if ( $this->is_test_mode() ) {
			$this->title = 'Test Mode Enabled';
		}

		$this->init_form_fields();
		$this->init_settings();

		if ( $this->is_supported() && $this->is_activated() ) {
			$this->enabled = $this->get_option( 'enabled', 'no' );
		} else {
			$this->enabled = 'no';
		}

		add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );
		add_action( 'woocommerce_api_wc_paygenius_redirect', array( $this, 'check_response' ) );
	}

	/**
	 * Check payment response from PayGenius.
	 */
	public function check_response() {
		// Sanitize and validate input.
		$order_id = isset( $_GET['order'] ) ? absint( $_GET['order'] ) : 0; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$ref      = isset( $_GET['ref'] ) ? sanitize_text_field( wp_unslash( $_GET['ref'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended

		if ( 0 === $order_id || empty( $ref ) ) {
			wc_add_notice( esc_html__( 'Invalid payment response.', 'paygenius-hosted-payment-gateway' ), 'error' );
			wp_safe_redirect( wc_get_cart_url() );
			exit;
		}

		$order = wc_get_order( $order_id );
		if ( ! $order ) {
			wc_add_notice( esc_html__( 'Order not found.', 'paygenius-hosted-payment-gateway' ), 'error' );
			wp_safe_redirect( wc_get_cart_url() );
			exit;
		}

		$client   = $this->get_client();
		$response = $client->info( $ref );

		if ( empty( $response ) || ! isset( $response['response']['code'] ) ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( '[PayGenius] Empty HTTP response from info()' ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			}
			wc_add_notice( esc_html__( 'Payment gateway error: Empty response from payment provider.', 'paygenius-hosted-payment-gateway' ), 'error' );
			wp_safe_redirect( wc_get_cart_url() );
			exit;
		}

		if ( 200 !== (int) $response['response']['code'] ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( '[PayGenius] HTTP ' . $response['response']['code'] . ' body: ' . $response['body'] ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			}
			wc_add_notice( esc_html__( 'Payment gateway error: Unable to connect to payment provider.', 'paygenius-hosted-payment-gateway' ), 'error' );
			wp_safe_redirect( wc_get_cart_url() );
			exit;
		}

		$response_body = json_decode( $response['body'], true );
		if ( ! is_array( $response_body ) || ! isset( $response_body['status'] ) ) {
			wc_add_notice( esc_html__( 'Payment failed', 'paygenius-hosted-payment-gateway' ), 'error' );
			$order->update_status( 'failed', esc_html__( 'Payment failed', 'paygenius-hosted-payment-gateway' ) );
			wp_safe_redirect( wc_get_cart_url() );
			exit;
		}

		if ( 'SETTLED' === $response_body['status'] ) {
			$return_url = $this->get_return_url( $order );
			$order->payment_complete( uniqid() );
			$order->reduce_order_stock();
			$order->add_order_note( esc_html__( 'Payment Complete', 'paygenius-hosted-payment-gateway' ) );

			if ( function_exists( 'WC' ) && isset( WC()->cart ) ) {
				WC()->cart->empty_cart();
			}

			wp_safe_redirect( $return_url );
			exit;
		} elseif ( 'CANCELLED' === $response_body['status'] ) {
			wc_add_notice( esc_html__( 'Payment cancelled', 'paygenius-hosted-payment-gateway' ), 'error' );
			wp_safe_redirect( wc_get_cart_url() );
			exit;
		} elseif ( 'AUTHORIZED' === $response_body['status'] || 'THREE_D_SECURE' === $response_body['status'] ) {
			// Payment authorized but not yet settled.
			return;
		} else {
			wc_add_notice( esc_html__( 'Payment failed', 'paygenius-hosted-payment-gateway' ), 'error' );
			$order->update_status( 'failed', esc_html__( 'Payment failed', 'paygenius-hosted-payment-gateway' ) );
			wp_safe_redirect( wc_get_cart_url() );
			exit;
		}
	}

	/**
	 * Check if gateway is supported.
	 *
	 * @return bool
	 */
	public function is_supported() {
		// Support all currencies (currency conversion is handled by PayGenius API).
		return true;
	}

	/**
	 * Check if gateway is activated.
	 *
	 * @return bool
	 */
	public function is_activated() {
		if ( $this->is_test_mode() ) {
			$paymentpage = $this->get_option( 'testing-paymentpage' );
			if ( ! empty( $paymentpage ) ) {
				return true;
			}
		} else {
			$paymentpage = $this->get_option( 'paymentpage' );
			if ( ! empty( $paymentpage ) ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Check if the gateway is available for use.
	 *
	 * @return bool
	 */
	public function is_available() {
		// Check if gateway is enabled.
		if ( 'yes' !== $this->enabled ) {
			$this->log_unavailability( 'enabled', 'no' );
			return false;
		}

		// Check if gateway is supported.
		if ( ! $this->is_supported() ) {
			$this->log_unavailability( 'supported', 'no' );
			return false;
		}

		// Check if gateway is activated (has payment page configured).
		if ( ! $this->is_activated() ) {
			$this->log_unavailability( 'activated', 'no' );
			return false;
		}

		// Check if required API keys are present.
		$has_keys = false;
		if ( $this->is_test_mode() ) {
			$has_keys = ! empty( $this->get_option( 'testing-token' ) ) && ! empty( $this->get_option( 'testing-secret' ) );
		} else {
			$has_keys = ! empty( $this->get_option( 'token' ) ) && ! empty( $this->get_option( 'secret' ) );
		}

		if ( ! $has_keys ) {
			$this->log_unavailability( 'keys', 'missing' );
			return false;
		}

		// Check if cart exists and has items.
		if ( ! WC()->cart || WC()->cart->is_empty() ) {
			$this->log_unavailability( 'cart', 'empty' );
			return false;
		}

		// Check if cart total is greater than 0.
		$cart_total = WC()->cart->get_total( 'edit' );
		if ( $cart_total <= 0 ) {
			$this->log_unavailability( 'total', 'zero' );
			return false;
		}

		return true;
	}

	/**
	 * Log why gateway is unavailable (for diagnostics).
	 *
	 * @param string $reason Reason code.
	 * @param string $value  Value that caused unavailability.
	 */
	private function log_unavailability( $reason, $value ) {
		$logger = wc_get_logger();
		if ( ! $logger ) {
			return;
		}

		$currency = get_woocommerce_currency();
		$ssl      = is_ssl() ? 'yes' : 'no';
		$total    = WC()->cart ? WC()->cart->get_total( 'edit' ) : 'n/a';
		$enabled  = $this->enabled;
		$keys     = ( $this->get_option( 'merchant_id', '' ) || $this->get_option( 'token', '' ) || $this->get_option( 'api_key', '' ) ) ? 'yes' : 'no';

		$logger->warning(
			sprintf(
				'%s unavailable: reason=%s value=%s enabled=%s keys=%s currency=%s ssl=%s total=%s',
				$this->id,
				$reason,
				$value,
				$enabled,
				$keys,
				$currency,
				$ssl,
				$total
			),
			array( 'source' => $this->id )
		);
	}

	/**
	 * Get payment page ID.
	 *
	 * @return string
	 */
	public function get_page() {
		if ( $this->is_test_mode() ) {
			return $this->get_option( 'testing-paymentpage' );
		} else {
			return $this->get_option( 'paymentpage' );
		}
	}

	/**
	 * Check if test mode is enabled.
	 *
	 * @return bool
	 */
	private function is_test_mode() {
		return 'yes' === $this->get_option( 'testmode' );
	}

	/**
	 * Initialize form fields.
	 */
	public function init_form_fields() {
		if ( ! is_admin() ) {
			return;
		}

		$client = $this->get_production_client();
		$pages  = $client->get_pages();

		$production_pages = array();
		if ( is_array( $pages ) && isset( $pages['body'] ) ) {
			$pages = json_decode( $pages['body'], true );

			if ( is_array( $pages ) && isset( $pages['success'] ) && true === $pages['success'] && isset( $pages['pages'] ) ) {
				foreach ( $pages['pages'] as $page ) {
					if ( isset( $page['id'], $page['name'] ) ) {
						$production_pages[ $page['id'] ] = $page['name'];
					}
				}
			}
		}

		$client = $this->get_test_client();
		$pages  = $client->get_pages();

		$testing_pages = array();
		if ( is_array( $pages ) && isset( $pages['body'] ) ) {
			$pages = json_decode( $pages['body'], true );

			if ( is_array( $pages ) && isset( $pages['success'] ) && true === $pages['success'] && isset( $pages['pages'] ) ) {
				foreach ( $pages['pages'] as $page ) {
					if ( isset( $page['id'], $page['name'] ) ) {
						$testing_pages[ $page['id'] ] = $page['name'];
					}
				}
			}
		}

		if ( empty( $production_pages ) ) {
			$production_pages[ null ] = '- no or invalid api credentials provided -';
		}

		if ( empty( $testing_pages ) ) {
			$testing_pages[ null ] = '- no or invalid api credentials provided -';
		}

		$this->form_fields = require PAYGENIUS_PLUGIN_DIR . 'lib/includes/form-fields.php';
	}

	/**
	 * Get client instance.
	 *
	 * @return \PayGenius\WebClient
	 */
	private function get_client() {
		if ( $this->is_test_mode() ) {
			return $this->get_test_client();
		} else {
			return $this->get_production_client();
		}
	}

	/**
	 * Get production client instance.
	 *
	 * @return \PayGenius\WebClient
	 */
	private function get_production_client() {
		return new \PayGenius\WebClient(
			$this->get_option( 'endpoint', 'https://www.paygenius.co.za/pg/api/v2/' ),
			$this->get_option( 'token', '' ),
			$this->get_option( 'secret', '' )
		);
	}

	/**
	 * Get test client instance.
	 *
	 * @return \PayGenius\WebClient
	 */
	private function get_test_client() {
		return new \PayGenius\WebClient(
			$this->get_option( 'testing-endpoint', 'https://developer.paygenius.co.za/pg/api/v2/' ),
			$this->get_option( 'testing-token', '' ),
			$this->get_option( 'testing-secret', '' )
		);
	}

	/**
	 * Get redirect URL for payment.
	 *
	 * @param int $order_id Order ID.
	 * @return string|false
	 */
	public function get_redirect_url( $order_id ) {
		$order = wc_get_order( $order_id );
		if ( ! $order ) {
			return false;
		}

		$client = $this->get_client();

		$redirect_url = add_query_arg( 'wc-api', 'WC_Paygenius_Redirect', home_url( '/' ) );
		$redirect_url = add_query_arg( 'order', $order_id, $redirect_url ) . '&ref=${reference}';

		$paygenius_reference = $order_id;

		if ( 'yes' === $this->get_option( 'use_sequence_number' ) ) {
			$paygenius_reference = $order->get_order_number();
		}

		// Recalculate totals before getting order total for discount compatibility.
		if ( function_exists( 'WC' ) && isset( WC()->cart ) ) {
			WC()->cart->calculate_totals();
		}

		$amount_cents = (int) round( $order->get_total() * 100 );

		$response = $client->create(
			$order->get_billing_first_name(),
			$order->get_billing_last_name(),
			$amount_cents,
			get_woocommerce_currency(),
			$order->get_billing_email(),
			$paygenius_reference,
			$this->get_page(),
			$redirect_url,
			$redirect_url,
			$redirect_url,
			$redirect_url
		);

		if ( empty( $response ) || ! isset( $response['response']['code'] ) ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( '[PayGenius] Empty HTTP response from create()' ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			}
			wc_add_notice( esc_html__( 'Payment gateway error: Empty response from payment provider.', 'paygenius-hosted-payment-gateway' ), 'error' );
			return false;
		}

		if ( 200 !== (int) $response['response']['code'] ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( '[PayGenius] HTTP ' . $response['response']['code'] . ' body: ' . $response['body'] ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			}
			wc_add_notice( esc_html__( 'Payment gateway error: Unable to connect to payment provider.', 'paygenius-hosted-payment-gateway' ), 'error' );
			return false;
		}

		$response_body = json_decode( $response['body'], true );

		if ( empty( $response_body ) || empty( $response_body['success'] ) ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( '[PayGenius] API failure: ' . $response['body'] ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			}
			wc_add_notice( esc_html__( 'Payment gateway error: Payment initiation failed.', 'paygenius-hosted-payment-gateway' ), 'error' );
			return false;
		}

		if ( isset( $response_body['redirectUrl'] ) ) {
			return esc_url_raw( $response_body['redirectUrl'] );
		}

		return false;
	}

	/**
	 * Process payment.
	 *
	 * @param int $order_id Order ID.
	 * @return array
	 */
	public function process_payment( $order_id ) {
		$redirect_url = $this->get_redirect_url( $order_id );

		if ( ! $redirect_url ) {
			return array(
				'result'   => 'fail',
				'redirect' => '',
			);
		}

		return array(
			'result'   => 'success',
			'redirect' => $redirect_url,
		);
	}
}
