/**
 * PayGenius Redirect Blocks Integration
 *
 * @package PayGenius
 */

( function () {
	'use strict';

	// Guard for availability to avoid runtime errors.
	if (
		typeof window.wp === 'undefined' ||
		typeof window.wc === 'undefined' ||
		! window.wp.element ||
		! window.wp.i18n ||
		! window.wc.wcSettings ||
		typeof window.wc.wcSettings.getSetting !== 'function'
	) {
		console.warn( '[PayGenius] Blocks APIs not ready; skipping registration.' );
		return;
	}

	// Wait for blocks registry to be available.
	const registerPaymentMethod = ( retryCount = 0 ) => {
		const registry =
			( window.wc && window.wc.wcBlocksRegistry ) ||
			( window.wc && window.wc.blocksRegistry );

		if ( ! registry || typeof registry.registerPaymentMethod !== 'function' ) {
			// Registry not available yet, retry up to 20 times (2 seconds total).
			if ( retryCount < 20 ) {
				setTimeout( () => registerPaymentMethod( retryCount + 1 ), 100 );
			} else {
				console.warn( '[PayGenius] Blocks registry not available; skipping registration.' );
			}
			return;
		}

		try {
			const { createElement } = window.wp.element;
			const { __ } = window.wp.i18n;
			const { getSetting } = window.wc.wcSettings;

			const settings = getSetting( 'paygenius-redirect_data', {} ) || {};

			const label = settings.title || 'PayGenius';

			const Content = () =>
				createElement(
					'div',
					{ className: 'wc-payment-method__paygenius-description' },
					settings.description || ''
				);

			registry.registerPaymentMethod( {
				name: 'paygenius-redirect',
				label,
				ariaLabel: label,
				content: Content,
				edit: Content,
				canMakePayment: () => {
					// Check if gateway is available (this will be handled by WooCommerce based on is_available()).
					return true;
				},
				supports: settings.supports || { features: [ 'products' ] },
				placeOrderButtonLabel:
					settings.placeOrderButtonLabel ||
					__( 'Proceed to PayGenius', 'paygenius-hosted-payment-gateway' ),
			} );
		} catch ( error ) {
			console.error( '[PayGenius] Error registering payment method:', error );
		}
	};

	// Wait for DOM and then try to register.
	if ( document.readyState === 'loading' ) {
		document.addEventListener( 'DOMContentLoaded', () => registerPaymentMethod() );
	} else {
		registerPaymentMethod();
	}
} )();
