<?php
/**
 * Admin class for Parcel History Checker WordPress plugin.
 *
 * This file contains the Admin class which handles all admin-related functionality
 * for the Parcel History Checker plugin.
 *
 * @package PHCWP
 */

namespace PHCWP;

defined( 'ABSPATH' ) || exit;

/**
 * Admin class.
 *
 * Handles admin menu creation, script enqueuing, and AJAX callbacks.
 */
class Admin {
	/**
	 * Api instance.
	 *
	 * @var Api
	 */
	private $api;

	/**
	 * Constructor.
	 *
	 * Initializes the admin class and sets up action hooks.
	 */
	public function __construct() {
		$this->api = new Api();
		add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
		add_action( 'wp_ajax_check_phone_number', array( $this, 'ajax_check_phone_number' ) );
		add_action( 'wp_ajax_load_parcel_checker_results', array( $this, 'ajax_load_parcel_checker_results' ) );
		add_action( 'wp_ajax_nopriv_check_phone_number', array( $this, 'ajax_check_phone_number' ) );
		add_action( 'wp_ajax_nopriv_load_parcel_checker_results', array( $this, 'ajax_load_parcel_checker_results' ) );
	}

	/**
	 * Add admin menu.
	 *
	 * Creates the admin menu item for the plugin.
	 */
	public function add_admin_menu() {
		add_menu_page(
			__( 'Parcel History Checker', 'parcel-history-checker' ),
			__( 'Parcel History Checker', 'parcel-history-checker' ),
			'manage_options',
			'parcel-history-checker',
			array( $this, 'settings_page' )
		);
	}

	/**
	 * Enqueue scripts and styles.
	 *
	 * Loads necessary CSS and JavaScript files for the admin area.
	 */
	public function enqueue_scripts() {
		wp_enqueue_script( 'jquery' );
		wp_enqueue_style( 'phcwp-css', PHCWP_ASSETS_URL . 'css/phcwp-style.css', array(), PHCWP_VERSION, 'all' );
		wp_enqueue_script( 'phcwp-script', PHCWP_ASSETS_URL . 'js/phcwp-script.js', array( 'jquery' ), PHCWP_VERSION, true );
		wp_localize_script(
			'phcwp-script',
			'ParcelHistoryChecker',
			array(
				'ajax_url'   => admin_url( 'admin-ajax.php' ),
				'plugin_url' => PHCWP_PLUGIN_URL,
				'nonce'      => wp_create_nonce( 'phcwp_ajax_nonce' ),
			)
		);
	}

	/**
	 * Render settings page.
	 *
	 * Displays the admin settings page for the plugin.
	 */
	public function settings_page() {
		$template_path = PHCWP_PLUGIN_DIR . 'templates/settings.php';
		if ( file_exists( $template_path ) ) {
			include $template_path;
		} else {
			echo '<h2>' . esc_html__( 'Error: Settings page template not found.', 'parcel-history-checker' ) . '</h2>';
		}
	}

	/**
	 * AJAX callback for checking phone number.
	 *
	 * Handles the AJAX request to check a phone number.
	 */
	public function ajax_check_phone_number() {
		check_ajax_referer( 'phcwp_ajax_nonce', 'nonce' );
		if ( ! isset( $_POST['phone_number'] ) ) {
			wp_send_json_error( __( 'Phone number not provided', 'parcel-history-checker' ) );
		}
		$phone_number = sanitize_text_field( wp_unslash( $_POST['phone_number'] ) );
		$response     = $this->api->check_phone_number( $phone_number );
		if ( $response && ! isset( $response['error'] ) ) {
			set_transient( 'phcwp_fraud_checker_data', $response['orderSummary'], HOUR_IN_SECONDS );
			wp_send_json_success();
		} else {
			wp_send_json_error( $response['error'] ?? __( 'Error retrieving data', 'parcel-history-checker' ) );
		}
	}

	/**
	 * AJAX callback for loading parcel checker results.
	 *
	 * Handles the AJAX request to load parcel checker results.
	 */
	public function ajax_load_parcel_checker_results() {
		check_ajax_referer('phcwp_ajax_nonce', 'nonce');
		$response_data = get_transient('phcwp_fraud_checker_data');
		
		if ($response_data) {
			$template_path = PHCWP_PLUGIN_DIR . 'templates/results.php';
			if (file_exists($template_path)) {
				ob_start();
				// Pass only the orderSummary data to the template
				$data = $response_data;
				include $template_path;
				$html = ob_get_clean();
				wp_send_json_success(array('html' => $html));
			} else {
				wp_send_json_error(__('Template file not found', 'parcel-history-checker'));
			}
		} else {
			wp_send_json_error(__('No data available', 'parcel-history-checker'));
		}
	}
}
