<?php

/**
 * Plugin Name: Pallet Sales for eCommerce
 * Plugin URI: https://wordpress.org/plugins/pallet-sales-for-ecommerce/
 * Description: Sell products by pallets with area calculation. Perfect for tiles, stones, and other construction materials. Works with WooCommerce.
 * Version: 1.1.7
 * Author: Sergei Voronin
 * Author URI: https://revalfilm.info
 * Text Domain: pallet-sales-for-ecommerce
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.9
 * WC requires at least: 5.0
 * WC tested up to: 8.5
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('PALLSAFO_PLUGIN_URL', plugin_dir_url(__FILE__));
define('PALLSAFO_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('PALLSAFO_VERSION', '1.1.7');

class Pallet_Sales_For_Ecommerce
{
    public function __construct()
    {
        add_action('before_woocommerce_init', array($this, 'declare_hpos_compatibility'));
        add_action('plugins_loaded', array($this, 'init'));
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }

    public function declare_hpos_compatibility()
    {
        if (class_exists(\Automattic\WooCommerce\Utilities\FeaturesUtil::class)) {
            \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
        }
    }

    public function init()
    {
        if (!class_exists('WooCommerce')) {
            add_action('admin_notices', array($this, 'woocommerce_notice'));
            return;
        }

        // Text domain 'pallet-sales-for-ecommerce' is automatically loaded by WordPress since version 4.6

        // Initialize settings
        $this->init_settings();

        $this->load_hooks();
    }

    private function load_hooks()
    {
        // Admin hooks
        add_action('woocommerce_product_options_general_product_data', array($this, 'add_pallet_fields'));
        add_action('woocommerce_process_product_meta', array($this, 'save_pallet_fields'));
        // Add variation support
        add_action('woocommerce_product_after_variable_attributes', array($this, 'add_variation_pallet_fields'), 10, 3);
        add_action('woocommerce_save_product_variation', array($this, 'save_variation_pallet_fields'), 10, 2);
        // Frontend hooks
        add_action('woocommerce_before_add_to_cart_button', array($this, 'display_pallet_info'), 5);
        add_filter('woocommerce_get_price_html', array($this, 'modify_price_display'), 10, 2);
        add_filter('woocommerce_product_single_add_to_cart_text', array($this, 'change_add_to_cart_text'), 10, 2);
        add_action('woocommerce_single_product_summary', array($this, 'add_custom_quantity_label'), 29);
        // New hook to replace price position
        add_action('woocommerce_single_product_summary', array($this, 'replace_price_position'), 10);
        // Cart hooks
        add_filter('woocommerce_add_cart_item_data', array($this, 'add_pallet_cart_item_data'), 10, 3);
        add_filter('woocommerce_get_item_data', array($this, 'display_pallet_cart_item_data'), 10, 2);
        add_action('woocommerce_before_calculate_totals', array($this, 'calculate_pallet_price'), 10, 1);
        add_filter('woocommerce_cart_item_quantity', array($this, 'modify_cart_quantity_display'), 10, 3);
        // для оформления заказа
        add_action('woocommerce_cart_totals_before_order_total', array($this, 'add_pallet_total_row'));
        add_action('woocommerce_review_order_before_order_total', array($this, 'add_pallet_total_row'));
        // Order hooks
        add_action('woocommerce_checkout_create_order_line_item', array($this, 'save_pallet_order_item_meta'), 10, 4);
        // Scripts and styles
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('admin_enqueue_scripts', array($this, 'admin_enqueue_scripts'));
        // Settings page
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'admin_init'));
    }

    public function replace_price_position()
    {
        global $product;

        if (!$this->is_sold_by_pallet($product)) {
            return;
        }

        // Enqueue jQuery and add inline styles/scripts
        wp_enqueue_script('jquery');

        // Add custom CSS for price repositioning
        $custom_css = "
        /* Hide the original price */
        .woocommerce-product-details__short-description .price,
        .woocommerce div.product p.price,
        .woocommerce div.product span.price {
            display: none !important;
        }

        /* Style for the new price position */
        .pallsafo-custom-price {
            font-size: 1.25em;
            color: #333;
            margin: 10px 0;
            font-weight: bold;
        }

        .pallsafo-custom-price .woocommerce-Price-amount {
            font-size: 1.2em;
        }

        .pallsafo-custom-price small {
            font-size: 1em;
            color: #666;
            font-weight: bold;
        }
    ";
        wp_add_inline_style('pallsafo-styles', $custom_css);

        // Add JavaScript for price repositioning
        $initial_price = wp_json_encode(wc_price($product->get_price()));
        $custom_js = "
    jQuery(document).ready(function($) {
        var currentVariationData = null;

        // Get initial price
        function getInitialPrice() {
            var price = {$initial_price};
            return price;
        }

        // Create and position custom price
        function createCustomPrice() {
            var title = $('.product_title');
            if (title.length) {
                // Remove any existing custom price
                $('.pallsafo-custom-price').remove();

                // Create new price element
                var newPrice = $('<div class=\"pallsafo-custom-price\"></div>');
                var initialPrice = getInitialPrice();
                newPrice.html(initialPrice + ' <small>/ m²</small>');

                // Insert after title
                title.after(newPrice);
            }
        }

        // Update custom price when variation changes
        function updateCustomPrice(variation) {
            var customPrice = $('.pallsafo-custom-price');
            if (customPrice.length && variation && variation.price_html) {
                // Use the complete variation price_html and add / m²
                var tempDiv = $('<div>').html(variation.price_html);
                var priceAmount = tempDiv.find('.woocommerce-Price-amount').first();

                if (priceAmount.length) {
                    customPrice.html(priceAmount.prop('outerHTML') + ' <small>/ m²</small>');
                }
            }
        }

        // Handle variation changes
        $('form.variations_form').on('found_variation', function(event, variation) {
            currentVariationData = variation;
            updateCustomPrice(variation);
        });

        // Handle variation reset
        $('form.variations_form').on('reset_data', function(event) {
            currentVariationData = null;
            var customPrice = $('.pallsafo-custom-price');
            if (customPrice.length) {
                var initialPrice = getInitialPrice();
                customPrice.html(initialPrice + ' <small>/ m²</small>');
            }
        });

        // Initialize
        createCustomPrice();

        // Monitor for DOM changes (fallback)
        var priceObserver = new MutationObserver(function(mutations) {
            mutations.forEach(function(mutation) {
                if (mutation.type === 'childList') {
                    var target = $(mutation.target);
                    if (target.hasClass('price') || target.find('.price').length > 0) {
                        setTimeout(function() {
                            if (!$('.pallsafo-custom-price').length) {
                                createCustomPrice();
                            }
                        }, 100);
                    }
                }
            });
        });

        // Start observing
        var priceContainer = $('.woocommerce-product-details__short-description')[0];
        if (priceContainer) {
            priceObserver.observe(priceContainer, {
                childList: true,
                subtree: true
            });
        }
    });
    ";
        wp_add_inline_script('pallsafo-scripts', $custom_js, 'after');
    }

    public function woocommerce_notice()
    {
?>
        <div class="notice notice-error">
            <p><?php esc_html_e('WooCommerce Pallet Sales requires WooCommerce to be installed and active.', 'pallet-sales-for-ecommerce'); ?></p>
        </div>
        <?php
    }

    public function add_variation_pallet_fields($loop, $variation_data, $variation)
    {
        woocommerce_wp_checkbox(array(
            'id' => '_pallsafo_sold_by_pallet_' . $loop,
            'name' => '_pallsafo_sold_by_pallet_' . $loop,
            'label' => __('Sold by pallets', 'pallet-sales-for-ecommerce'),
            'value' => get_post_meta($variation->ID, '_pallsafo_sold_by_pallet', true)
        ));

        woocommerce_wp_text_input(array(
            'id' => '_pallsafo_pallet_area_' . $loop,
            'name' => '_pallsafo_pallet_area_' . $loop,
            'label' => __('Pallet area (m²)', 'pallet-sales-for-ecommerce'),
            'value' => get_post_meta($variation->ID, '_pallsafo_pallet_area', true),
            'type' => 'number',
            'custom_attributes' => array('step' => '0.01', 'min' => '0')
        ));

        woocommerce_wp_text_input(array(
            'id' => '_pallsafo_pallet_price_' . $loop,
            'name' => '_pallsafo_pallet_price_' . $loop,
            'label' => __('Pallet price', 'pallet-sales-for-ecommerce'),
            'value' => get_post_meta($variation->ID, '_pallsafo_pallet_price', true),
            'type' => 'number',
            'custom_attributes' => array('step' => '0.01', 'min' => '0')
        ));
    }

    public function save_variation_pallet_fields($variation_id, $i)
    {
        // Nonce verification
        if (
            !isset($_POST['woocommerce_meta_nonce']) ||
            !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['woocommerce_meta_nonce'])), 'woocommerce_save_data')
        ) {
            return;
        }

        // Permission check
        if (!current_user_can('edit_products')) {
            return;
        }

        if (isset($_POST['_pallsafo_sold_by_pallet_' . $i])) {
            update_post_meta($variation_id, '_pallsafo_sold_by_pallet', 'yes');
        } else {
            update_post_meta($variation_id, '_pallsafo_sold_by_pallet', 'no');
        }

        if (isset($_POST['_pallsafo_pallet_area_' . $i])) {
            $pallet_area = sanitize_text_field($_POST['_pallsafo_pallet_area_' . $i]);
            if (is_numeric($pallet_area) && floatval($pallet_area) >= 0) {
                update_post_meta($variation_id, '_pallsafo_pallet_area', $pallet_area);
            }
        }

        if (isset($_POST['_pallsafo_pallet_price_' . $i])) {
            $pallet_price = sanitize_text_field($_POST['_pallsafo_pallet_price_' . $i]);
            if (is_numeric($pallet_price) && floatval($pallet_price) >= 0) {
                update_post_meta($variation_id, '_pallsafo_pallet_price', $pallet_price);
            }
        }
    }

    public function add_pallet_fields()
    {
        global $post;
        echo '<div class="options_group pallsafo-pallet-fields">';
        echo '<h3 style="margin: 15px 12px 10px; color: #555;">🚛 ' . esc_html__('Pallet Settings', 'pallet-sales-for-ecommerce') . '</h3>';
        woocommerce_wp_checkbox(array(
            'id' => '_pallsafo_sold_by_pallet',
            'label' => __('Sold by pallets', 'pallet-sales-for-ecommerce'),
            'description' => __('Check this option if product is sold by pallets', 'pallet-sales-for-ecommerce'),
            'desc_tip' => true,
            'wrapper_class' => 'pallsafo-checkbox-field'
        ));
        woocommerce_wp_text_input(array(
            'id' => '_pallsafo_pallet_area',
            'label' => __('Pallet area (m²)', 'pallet-sales-for-ecommerce'),
            'description' => __('Coverage area of stones on one pallet', 'pallet-sales-for-ecommerce'),
            'type' => 'number',
            'desc_tip' => true,
            'custom_attributes' => array(
                'step' => '0.01',
                'min' => '0',
                'placeholder' => '15.00'
            ),
            'wrapper_class' => 'pallsafo-area-field'
        ));
        woocommerce_wp_text_input(array(
            'id' => '_pallsafo_pallet_price',
            'label' => __('Pallet price', 'pallet-sales-for-ecommerce'),
            'description' => __('Total pallet price (not per m² price)', 'pallet-sales-for-ecommerce'),
            'type' => 'number',
            'desc_tip' => true,
            'custom_attributes' => array(
                'step' => '0.01',
                'min' => '0'
            ),
            'wrapper_class' => 'pallsafo-area-field'
        ));
        woocommerce_wp_text_input(array(
            'id' => '_pallsafo_stone_weight',
            'label' => __('Single stone weight (kg)', 'pallet-sales-for-ecommerce'),
            'description' => __('Specify the weight of one stone in kilograms', 'pallet-sales-for-ecommerce'),
            'type' => 'number',
            'desc_tip' => true,
            'custom_attributes' => array(
                'step' => '0.01',
                'min' => '0'
            ),
            'wrapper_class' => 'pallsafo-area-field'
        ));

        woocommerce_wp_text_input(array(
            'id' => '_pallsafo_stones_per_sqm',
            'label' => __('Number of stones per 1 m²', 'pallet-sales-for-ecommerce'),
            'description' => __('Specify the number of stones per square meter', 'pallet-sales-for-ecommerce'),
            'type' => 'number',
            'desc_tip' => true,
            'custom_attributes' => array(
                'step' => '1',
                'min' => '0'
            ),
            'wrapper_class' => 'pallsafo-area-field'
        ));
        echo '</div>';
    }

    public function save_pallet_fields($post_id)
    {
        // Nonce verification
        if (
            !isset($_POST['woocommerce_meta_nonce']) ||
            !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['woocommerce_meta_nonce'])), 'woocommerce_save_data')
        ) {
            return;
        }

        // Autosave check
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }

        // Permission check
        if (!current_user_can('edit_product', $post_id)) {
            return;
        }

        $sold_by_pallet = isset($_POST['_pallsafo_sold_by_pallet']) ? 'yes' : 'no';
        update_post_meta($post_id, '_pallsafo_sold_by_pallet', $sold_by_pallet);

        if (isset($_POST['_pallsafo_pallet_area'])) {
            $pallet_area = sanitize_text_field($_POST['_pallsafo_pallet_area']);
            if (is_numeric($pallet_area) && floatval($pallet_area) >= 0) {
                update_post_meta($post_id, '_pallsafo_pallet_area', $pallet_area);
            }
        }

        if (isset($_POST['_pallsafo_pallet_price'])) {
            $pallet_price = sanitize_text_field($_POST['_pallsafo_pallet_price']);
            if (is_numeric($pallet_price) && floatval($pallet_price) >= 0) {
                update_post_meta($post_id, '_pallsafo_pallet_price', $pallet_price);
            }
        }

        if (isset($_POST['_pallsafo_stone_weight'])) {
            $stone_weight = sanitize_text_field($_POST['_pallsafo_stone_weight']);
            if (is_numeric($stone_weight) && floatval($stone_weight) >= 0) {
                update_post_meta($post_id, '_pallsafo_stone_weight', $stone_weight);
            }
        }

        if (isset($_POST['_pallsafo_stones_per_sqm'])) {
            $stones_per_sqm = sanitize_text_field($_POST['_pallsafo_stones_per_sqm']);
            if (is_numeric($stones_per_sqm) && intval($stones_per_sqm) >= 0) {
                update_post_meta($post_id, '_pallsafo_stones_per_sqm', $stones_per_sqm);
            }
        }
    }

    public function display_pallet_info()
    {
        global $product;
        if (!$this->is_sold_by_pallet($product)) {
            return;
        }
        $pallet_area = get_post_meta($product->get_id(), '_pallsafo_pallet_area', true);
        $stone_weight = get_post_meta($product->get_id(), '_pallsafo_stone_weight', true);
        $stones_per_sqm = get_post_meta($product->get_id(), '_pallsafo_stones_per_sqm', true);
        $price_per_sqm = $product->get_regular_price();
        echo '<div class="pallsafo-pallet-info" style="margin: 15px 0; clear: both;">';
        echo '<div class="pallsafo-pallet-details">';
        echo '<div>🚛 ' . esc_html__('Sold by pallets', 'pallet-sales-for-ecommerce') . '</div>';
        if ($pallet_area) {
            echo '<div>📐 ' . esc_html__('Pallet area:', 'pallet-sales-for-ecommerce') . ' <strong>' . esc_html($pallet_area) . ' m²</strong></div>';
        }
        if ($pallet_area && $this->is_feature_enabled('calculator')) {
            echo '<div class="pallsafo-calculator" style="margin: 10px 0;">';
            echo '<div class="pallsafo-calc-input-line">';
            echo '<span class="pallsafo-calc-label">🧮 ' . esc_html__('Desired area:', 'pallet-sales-for-ecommerce') . '</span>';
            echo '<input type="number" id="pallsafo-area-input" class="pallsafo-area-input-compact" min="0" step="0.01" placeholder="m²">';
            echo '</div>';
            echo '<div id="pallsafo-calc-result" style="margin-top: 5px;"></div>';
            echo '</div>';
        }
        if ($stone_weight && $stones_per_sqm && $pallet_area && ($this->is_feature_enabled('weight_calculation') || $this->is_feature_enabled('display_stones_count'))) {
            $stones_per_pallet = $stones_per_sqm * $pallet_area;
            $pallet_weight = $stone_weight * $stones_per_pallet;
            echo '<div class="pallsafo-additional-info">';
            if ($this->is_feature_enabled('weight_calculation')) {
                echo '<div>⚖️ ' . esc_html__('Pallet weight:', 'pallet-sales-for-ecommerce') . ' <strong>' . esc_html(number_format($pallet_weight, 2)) . ' kg</strong></div>';
            }
            if ($this->is_feature_enabled('display_stones_count')) {
                echo '<div>🔢 ' . esc_html__('Stones per 1 m²:', 'pallet-sales-for-ecommerce') . ' <strong>' . esc_html($stones_per_sqm) . ' ' . esc_html__('pcs', 'pallet-sales-for-ecommerce') . '</strong></div>';
                echo '<div>📦 ' . esc_html__('Stones per pallet:', 'pallet-sales-for-ecommerce') . ' <strong>' . esc_html(round($stones_per_pallet)) . ' ' . esc_html__('pcs', 'pallet-sales-for-ecommerce') . '</strong></div>';
            }
            echo '</div>';
        }
        echo '</div></div>';

        // Enhanced JavaScript for variable products with variation data
        wp_enqueue_script('jquery');

        // Prepare translation strings for JavaScript
        $i18n_pallets = esc_js(__('Number of pallets:', 'pallet-sales-for-ecommerce'));
        $i18n_total_area = esc_js(__('Total area:', 'pallet-sales-for-ecommerce'));
        $i18n_total_price = esc_js(__('Total price:', 'pallet-sales-for-ecommerce'));
        $i18n_pcs = esc_js(__('pcs', 'pallet-sales-for-ecommerce'));

        if ($product->is_type('variable')) {
            $variations_data = array();
            foreach ($product->get_available_variations() as $variation) {
                $variation_obj = wc_get_product($variation['variation_id']);
                $variations_data[$variation['variation_id']] = array(
                    'pallet_area' => get_post_meta($variation['variation_id'], '_pallsafo_pallet_area', true),
                    'stone_weight' => get_post_meta($variation['variation_id'], '_pallsafo_stone_weight', true),
                    'stones_per_sqm' => get_post_meta($variation['variation_id'], '_pallsafo_stones_per_sqm', true),
                    'price' => $variation_obj->get_price()
                );
            }

            $variations_json = wp_json_encode($variations_data);
            $script_data = "
            var pallsafo_variationsData = {$variations_json};
            var pallsafo_defaultPalletArea = " . floatval($pallet_area) . ";
            var pallsafo_defaultStonesPerSqm = " . floatval($stones_per_sqm) . ";
            var pallsafo_defaultStoneWeight = " . floatval($stone_weight) . ";
            var pallsafo_defaultPrice = " . floatval($price_per_sqm) . ";
            var pallsafo_i18n = {
                pallets: '{$i18n_pallets}',
                total_area: '{$i18n_total_area}',
                total_price: '{$i18n_total_price}',
                pcs: '{$i18n_pcs}'
            };
        ";
        } else {
            $script_data = "
            var pallsafo_variationsData = {};
            var pallsafo_defaultPalletArea = " . floatval($pallet_area) . ";
            var pallsafo_defaultStonesPerSqm = " . floatval($stones_per_sqm) . ";
            var pallsafo_defaultStoneWeight = " . floatval($stone_weight) . ";
            var pallsafo_defaultPrice = " . floatval($price_per_sqm) . ";
            var pallsafo_i18n = {
                pallets: '{$i18n_pallets}',
                total_area: '{$i18n_total_area}',
                total_price: '{$i18n_total_price}',
                pcs: '{$i18n_pcs}'
            };
        ";
        }

        $calculator_script = $script_data . "
        jQuery(document).ready(function($) {
            var currentPalletArea = pallsafo_defaultPalletArea;
            var currentStonesPerSqm = pallsafo_defaultStonesPerSqm;
            var currentStoneWeight = pallsafo_defaultStoneWeight;
            var currentPrice = pallsafo_defaultPrice;

            function updateDataForVariation(variationId) {
                if (pallsafo_variationsData[variationId]) {
                    var varData = pallsafo_variationsData[variationId];
                    currentPalletArea = parseFloat(varData.pallet_area) || pallsafo_defaultPalletArea;
                    currentStonesPerSqm = parseFloat(varData.stones_per_sqm) || pallsafo_defaultStonesPerSqm;
                    currentStoneWeight = parseFloat(varData.stone_weight) || pallsafo_defaultStoneWeight;
                    currentPrice = parseFloat(varData.price) || pallsafo_defaultPrice;
                } else {
                    currentPalletArea = pallsafo_defaultPalletArea;
                    currentStonesPerSqm = pallsafo_defaultStonesPerSqm;
                    currentStoneWeight = pallsafo_defaultStoneWeight;
                    currentPrice = pallsafo_defaultPrice;
                }

                // Update calculator
                calculatePallets();
            }

            function calculatePallets() {
                var neededArea = parseFloat($('#pallsafo-area-input').val()) || 0;
                if (neededArea > 0 && currentPalletArea > 0) {
                    var palletsNeeded = Math.ceil(neededArea / currentPalletArea);
                    var actualArea = (palletsNeeded * currentPalletArea).toFixed(2);
                    var totalPrice = (actualArea * currentPrice).toFixed(2);
                    var totalStones = Math.round(actualArea * currentStonesPerSqm);
                    var totalWeight = (totalStones * currentStoneWeight).toFixed(2);

                    // Display calculation result
                    var resultHtml = '<div>';
                    resultHtml += '<div>📦 ' + pallsafo_i18n.pallets + ' <strong>' + palletsNeeded + ' ' + pallsafo_i18n.pcs + '</strong></div>';
                    resultHtml += '<div>📐 ' + pallsafo_i18n.total_area + ' <strong>' + actualArea + ' m²</strong></div>';
                    resultHtml += '<div>💰 ' + pallsafo_i18n.total_price + ' <strong>' + totalPrice + '€</strong></div>';
                    resultHtml += '</div>';
                    $('#pallsafo-calc-result').html(resultHtml);

                    // Set pallet quantity
                    $('.quantity input[name=\"quantity\"]').val(palletsNeeded).trigger('change');
                } else {
                    $('#pallsafo-calc-result').html('');
                    $('.quantity input[name=\"quantity\"]').val(1).trigger('change');
                }
            }

            // Listen for variation changes
            $('form.variations_form').on('found_variation', function(event, variation) {
                if (variation.variation_id) {
                    updateDataForVariation(variation.variation_id);
                }
            });

            // Listen for variation reset
            $('form.variations_form').on('reset_data', function(event) {
                updateDataForVariation(null);
            });

            $('#pallsafo-area-input').on('input keyup change', calculatePallets);
        });
    ";

        wp_add_inline_script('pallsafo-scripts', $calculator_script, 'after');
    }

    public function modify_price_display($price_html, $product)
    {
        // Only for frontend (not admin)
        if (is_admin() && !defined('DOING_AJAX')) {
            return $price_html;
        }

        // Skip if no product
        if (!$product || !is_object($product)) {
            return $price_html;
        }

        // Don't modify prices in cart, checkout, or other non-product pages
        if (is_cart() || is_checkout() || is_account_page()) {
            return $price_html;
        }

        // Get product ID - handle both simple and variable products
        $product_id = $product->get_id();
        if ($product->is_type('variation')) {
            $product_id = $product->get_parent_id();
        }

        // Check if product is sold by pallets (now with strict validation)
        if ($this->is_sold_by_pallet($product)) {
            $price_per_sqm = $product->get_price();
            $price_html = wc_price($price_per_sqm) . ' <small>/ m²</small>';
        }

        return $price_html;
    }

    public function add_custom_quantity_label()
    {
        global $product;
        if (!$this->is_sold_by_pallet($product)) {
            return;
        }
        $pallet_area = get_post_meta($product->get_id(), '_pallsafo_pallet_area', true);
        $price_per_sqm = $product->get_regular_price();

        wp_enqueue_script('jquery');

        $label_text = esc_js(esc_html__('Number of pallets:', 'pallet-sales-for-ecommerce'));
        $total_area_text = esc_js(esc_html__('Total area:', 'pallet-sales-for-ecommerce'));
        $total_price_text = esc_js(esc_html__('Total price:', 'pallet-sales-for-ecommerce'));

        $quantity_script = "
    jQuery(document).ready(function($) {
        setTimeout(function() {
            var quantityInput = $('.quantity input[name=\"quantity\"]');
            var quantityDiv = $('.quantity');
            if (quantityInput.length && quantityDiv.length) {
                if ($('.pallsafo-quantity-label').length === 0) {
                    quantityDiv.before('<div class=\"pallsafo-quantity-label\">{$label_text}</div>');
                }
                quantityInput.addClass('pallsafo-pallet-quantity');";

        if ($pallet_area && $price_per_sqm) {
            $quantity_script .= "
                var palletArea = " . floatval($pallet_area) . ";
                var pricePerSqm = " . floatval($price_per_sqm) . ";
                if ($('.pallsafo-total-area').length === 0) {
                    quantityDiv.after('<div class=\"pallsafo-total-area\"></div>');
                }
                function updateTotalArea() {
                    var quantity = parseInt(quantityInput.val()) || 1;
                    var totalArea = (quantity * palletArea).toFixed(2);
                    var totalPrice = (totalArea * pricePerSqm).toFixed(2);
                    $('.pallsafo-total-area').html(
                        '{$total_area_text}: <strong>' + totalArea + ' m²</strong><br>' +
                        '{$total_price_text}: <strong>' + totalPrice + '€</strong>'
                    );
                }
                quantityInput.on('input change keyup', updateTotalArea);
                updateTotalArea();";
        }

        $quantity_script .= "
            }
        }, 500);
    });
    ";

        wp_add_inline_script('pallsafo-scripts', $quantity_script, 'after');
    }

    public function change_add_to_cart_text($text, $product)
    {
        if ($this->is_sold_by_pallet($product)) {
            return __('Add pallets to cart', 'pallet-sales-for-ecommerce');
        }
        return $text;
    }

    public function add_pallet_cart_item_data($cart_item_data, $product_id, $variation_id)
    {
        $product_id = absint($product_id);
        $product = wc_get_product($product_id);
        if ($this->is_sold_by_pallet($product)) {
            $pallet_area = get_post_meta($product_id, '_pallsafo_pallet_area', true);
            $cart_item_data['pallsafo_pallet_data'] = array(
                'is_pallet' => true,
                'pallet_area' => $pallet_area
            );
        }
        return $cart_item_data;
    }

    public function add_pallet_total_row()
    {
        $pallet_total = 0;

        foreach (WC()->cart->get_cart() as $cart_item) {
            if (isset($cart_item['pallsafo_pallet_data']) && $cart_item['pallsafo_pallet_data']['is_pallet']) {
                $product_id = absint($cart_item['product_id']);
                $pallet_price = get_post_meta($product_id, '_pallsafo_pallet_price', true);
                $quantity = absint($cart_item['quantity']);

                if (is_numeric($pallet_price) && $quantity > 0) {
                    $pallet_total += floatval($pallet_price) * $quantity;
                }
            }
        }

        if ($pallet_total > 0) {
        ?>
            <tr class="cart-pallet-total">
                <th><?php esc_html_e('Pallet price', 'pallet-sales-for-ecommerce'); ?></th>
                <td><?php echo wc_price($pallet_total); ?></td>
            </tr>
        <?php
        }
    }

    public function display_pallet_cart_item_data($item_data, $cart_item)
    {
        if (isset($cart_item['pallsafo_pallet_data']) && $cart_item['pallsafo_pallet_data']['is_pallet']) {
            $item_data[] = array(
                'name' => __('Sales method', 'pallet-sales-for-ecommerce'),
                'value' => __('by pallets', 'pallet-sales-for-ecommerce')
            );
            if (!empty($cart_item['pallsafo_pallet_data']['pallet_area'])) {
                $area_per_pallet = floatval($cart_item['pallsafo_pallet_data']['pallet_area']);
                $total_area = $area_per_pallet * absint($cart_item['quantity']);
                $item_data[] = array(
                    'name' => __('Number of pallets', 'pallet-sales-for-ecommerce'),
                    'value' => absint($cart_item['quantity']) . ' ' . __('pcs', 'pallet-sales-for-ecommerce')
                );
                $item_data[] = array(
                    'name' => __('Pallet area (m²)', 'pallet-sales-for-ecommerce'),
                    'value' => esc_html($area_per_pallet) . ' m²'
                );
                $item_data[] = array(
                    'name' => __('Total area', 'pallet-sales-for-ecommerce'),
                    'value' => '<strong>' . esc_html($total_area) . ' m²</strong>'
                );
            }
        }
        return $item_data;
    }

    public function modify_cart_quantity_display($product_quantity, $cart_item_key, $cart_item)
    {
        if (isset($cart_item['pallsafo_pallet_data']) && $cart_item['pallsafo_pallet_data']['is_pallet']) {
            if (!empty($cart_item['pallsafo_pallet_data']['pallet_area'])) {
                $area_per_pallet = floatval($cart_item['pallsafo_pallet_data']['pallet_area']);
                $total_area = $area_per_pallet * absint($cart_item['quantity']);
                return '<span class="pallsafo-cart-quantity">' . esc_html($total_area) . ' m²</span>
                        <input type="number" name="cart[' . esc_attr($cart_item_key) . '][qty]" value="' . esc_attr(absint($cart_item['quantity'])) . '"
                               title="' . esc_attr__('Number of pallets', 'pallet-sales-for-ecommerce') . '"
                               class="input-text qty text pallsafo-pallet-qty"
                               size="4" min="1" max="" step="1" />';
            }
        }
        return $product_quantity;
    }

    public function calculate_pallet_price($cart)
    {
        if (is_admin() && !defined('DOING_AJAX')) {
            return;
        }

        foreach ($cart->get_cart() as $cart_item_key => $cart_item) {
            if (isset($cart_item['pallsafo_pallet_data']) && $cart_item['pallsafo_pallet_data']['is_pallet']) {
                $product = $cart_item['data'];
                $price_per_sqm = floatval($product->get_regular_price());
                $area_per_pallet = floatval($cart_item['pallsafo_pallet_data']['pallet_area']);

                $price_per_pallet = $area_per_pallet * $price_per_sqm;
                $cart_item['data']->set_price($price_per_pallet);
            }
        }
    }

    public function save_pallet_order_item_meta($item, $cart_item_key, $values, $order)
    {
        if (isset($values['pallsafo_pallet_data'])) {
            $pallet_data = $values['pallsafo_pallet_data'];
            $item->add_meta_data(__('Sales method', 'pallet-sales-for-ecommerce'), __('By pallets', 'pallet-sales-for-ecommerce'));
            $item->add_meta_data(__('Number of pallets', 'pallet-sales-for-ecommerce'), absint($values['quantity']) . ' ' . __('pcs', 'pallet-sales-for-ecommerce'));
            if (!empty($pallet_data['pallet_area'])) {
                $area_per_pallet = floatval($pallet_data['pallet_area']);
                $total_area = $area_per_pallet * absint($values['quantity']);
                $item->add_meta_data(__('Pallet area (m²)', 'pallet-sales-for-ecommerce'), floatval($area_per_pallet) . ' m²');
                $item->add_meta_data(__('Total area', 'pallet-sales-for-ecommerce'), floatval($total_area) . ' m²');
            }
        }
    }

    public function enqueue_scripts()
    {
        if (is_product() || is_cart() || is_checkout()) {
            wp_enqueue_style(
                'pallsafo-styles',
                esc_url(PALLSAFO_PLUGIN_URL . 'assets/style.css'),
                array(),
                PALLSAFO_VERSION
            );
            wp_add_inline_style('pallsafo-styles', '
                .pallsafo-pallet-price {
                    color: #444;
                    padding: 3px 0;
                }
                .pallsafo-pallet-price strong {
                    color: #000;
                    font-weight: bold;
                }
                .pallsafo-pallet-qty {
                    width: 70px !important;
                    min-width: 70px !important;
                }
            ');
            wp_enqueue_script('jquery');
            wp_register_script('pallsafo-scripts', false, array('jquery'), PALLSAFO_VERSION, true);
            wp_enqueue_script('pallsafo-scripts');
        }
    }

    public function admin_enqueue_scripts($hook)
    {
        if ('post.php' === $hook || 'post-new.php' === $hook) {
            global $post_type;
            if ('product' === $post_type) {
                wp_enqueue_script('jquery');

                // Add inline script for pallet fields toggle
                $inline_script = "
                    jQuery(document).ready(function($) {
                        function togglePalletFields() {
                            var isChecked = $('#_pallsafo_sold_by_pallet').is(':checked');
                            if (isChecked) {
                                $('.pallsafo-area-field').show();
                            } else {
                                $('.pallsafo-area-field').hide();
                            }
                        }
                        $(document).on('change', '#_pallsafo_sold_by_pallet', togglePalletFields);
                        togglePalletFields();
                        setTimeout(togglePalletFields, 500);
                    });
                ";
                wp_add_inline_script('jquery', $inline_script);

                // Add inline style for pallet fields
                wp_enqueue_style('wp-admin');
                $inline_style = "
                    .pallsafo-pallet-fields {
                        background: #f9f9f9;
                        border: 1px solid #ddd;
                        border-radius: 3px;
                        padding: 15px;
                        margin: 15px 0;
                    }
                    .pallsafo-checkbox-field label {
                        font-weight: bold;
                    }
                    .pallsafo-area-field {
                        display: none;
                    }
                ";
                wp_add_inline_style('wp-admin', $inline_style);
            }
        }
    }

    private function is_sold_by_pallet($product)
    {
        if (!$product) return false;

        $product_id = $product->get_id();
        if ($product->is_type('variation')) {
            $product_id = $product->get_parent_id();
        }

        $sold_by_pallet = get_post_meta($product_id, '_pallsafo_sold_by_pallet', true);

        // Only return true if explicitly set to 'yes'
        if ($sold_by_pallet !== 'yes') {
            return false;
        }

        // Additional validation: check if pallet area and price are set
        $pallet_area = get_post_meta($product_id, '_pallsafo_pallet_area', true);
        $pallet_price = get_post_meta($product_id, '_pallsafo_pallet_price', true);

        return !empty($pallet_area) && is_numeric($pallet_area) && floatval($pallet_area) > 0 &&
            !empty($pallet_price) && is_numeric($pallet_price) && floatval($pallet_price) > 0;
    }

    public function activate()
    {
        // Check if WooCommerce is active
        if (!class_exists('WooCommerce')) {
            deactivate_plugins(plugin_basename(__FILE__));
            wp_die(
                sprintf(
                    '<h1>%s</h1><p>%s</p><p><a href="%s">%s</a></p>',
                    esc_html__('Plugin Activation Error', 'pallet-sales-for-ecommerce'),
                    esc_html__('Pallet Sales for eCommerce requires WooCommerce to be installed and active. Please install and activate WooCommerce first.', 'pallet-sales-for-ecommerce'),
                    esc_url(admin_url('plugins.php')),
                    esc_html__('Return to Plugins', 'pallet-sales-for-ecommerce')
                ),
                esc_html__('Plugin Activation Error', 'pallet-sales-for-ecommerce'),
                array('back_link' => true)
            );
        }

        // Check WooCommerce version
        if (defined('WC_VERSION') && version_compare(WC_VERSION, '5.0', '<')) {
            deactivate_plugins(plugin_basename(__FILE__));
            wp_die(
                sprintf(
                    '<h1>%s</h1><p>%s</p><p><a href="%s">%s</a></p>',
                    esc_html__('Plugin Activation Error', 'pallet-sales-for-ecommerce'),
                    esc_html__('Pallet Sales for eCommerce requires WooCommerce version 5.0 or higher. Please update WooCommerce.', 'pallet-sales-for-ecommerce'),
                    esc_url(admin_url('plugins.php')),
                    esc_html__('Return to Plugins', 'pallet-sales-for-ecommerce')
                ),
                esc_html__('Plugin Activation Error', 'pallet-sales-for-ecommerce'),
                array('back_link' => true)
            );
        }

        $this->migrate_old_data();
        flush_rewrite_rules();
    }

    // Migration function to update old meta keys and options to new naming
    private function migrate_old_data()
    {
        // Check if migration has already been done
        if (get_option('pallsafo_migration_done')) {
            return;
        }

        global $wpdb;

        // Migrate product meta keys
        $old_new_meta_keys = array(
            '_wps_sold_by_pallet' => '_pallsafo_sold_by_pallet',
            '_wps_pallet_area' => '_pallsafo_pallet_area',
            '_wps_pallet_price' => '_pallsafo_pallet_price',
            '_wps_stone_weight' => '_pallsafo_stone_weight',
            '_wps_stones_per_sqm' => '_pallsafo_stones_per_sqm'
        );

        foreach ($old_new_meta_keys as $old_key => $new_key) {
            $wpdb->query($wpdb->prepare(
                "UPDATE {$wpdb->postmeta} SET meta_key = %s WHERE meta_key = %s",
                $new_key,
                $old_key
            ));
        }

        // Migrate options
        $old_new_options = array(
            'wps_default_currency_symbol' => 'pallsafo_default_currency_symbol',
            'wps_calculator_enabled' => 'pallsafo_calculator_enabled',
            'wps_weight_calculation_enabled' => 'pallsafo_weight_calculation_enabled',
            'wps_display_stones_count' => 'pallsafo_display_stones_count'
        );

        foreach ($old_new_options as $old_option => $new_option) {
            $old_value = get_option($old_option);
            if ($old_value !== false) {
                update_option($new_option, $old_value);
                delete_option($old_option);
            }
        }

        // Mark migration as complete
        update_option('pallsafo_migration_done', '1');
    }

    public function deactivate()
    {
        flush_rewrite_rules();
    }

    // Settings functionality
    public function init_settings()
    {
        // Register default settings
        add_option('pallsafo_default_currency_symbol', '€');
        add_option('pallsafo_calculator_enabled', 'yes');
        add_option('pallsafo_weight_calculation_enabled', 'yes');
        add_option('pallsafo_display_stones_count', 'yes');
    }

    public function add_admin_menu()
    {
        add_submenu_page(
            'woocommerce',
            __('Pallet Sales Settings', 'pallet-sales-for-ecommerce'),
            __('Pallet Sales', 'pallet-sales-for-ecommerce'),
            'manage_woocommerce',
            'pallsafo-settings',
            array($this, 'settings_page')
        );
    }

    public function admin_init()
    {
        // Check if WooCommerce is still active
        if (!class_exists('WooCommerce')) {
            deactivate_plugins(plugin_basename(__FILE__));
            add_action('admin_notices', function () {
                ?>
                <div class="notice notice-error is-dismissible">
                    <p><?php esc_html_e('Pallet Sales for eCommerce has been deactivated because WooCommerce is not active.', 'pallet-sales-for-ecommerce'); ?></p>
                </div>
                <?php
            });
            return;
        }

        register_setting('pallsafo_settings', 'pallsafo_default_currency_symbol', array('sanitize_callback' => 'sanitize_text_field'));
        register_setting('pallsafo_settings', 'pallsafo_calculator_enabled', array('sanitize_callback' => array($this, 'sanitize_checkbox')));
        register_setting('pallsafo_settings', 'pallsafo_weight_calculation_enabled', array('sanitize_callback' => array($this, 'sanitize_checkbox')));
        register_setting('pallsafo_settings', 'pallsafo_display_stones_count', array('sanitize_callback' => array($this, 'sanitize_checkbox')));

        add_settings_section(
            'pallsafo_general_section',
            __('General Settings', 'pallet-sales-for-ecommerce'),
            array($this, 'general_section_callback'),
            'pallsafo_settings'
        );

        add_settings_field(
            'pallsafo_calculator_enabled',
            __('Enable Area Calculator', 'pallet-sales-for-ecommerce'),
            array($this, 'calculator_enabled_callback'),
            'pallsafo_settings',
            'pallsafo_general_section'
        );

        add_settings_field(
            'pallsafo_weight_calculation_enabled',
            __('Enable Weight Calculation', 'pallet-sales-for-ecommerce'),
            array($this, 'weight_calculation_enabled_callback'),
            'pallsafo_settings',
            'pallsafo_general_section'
        );

        add_settings_field(
            'pallsafo_display_stones_count',
            __('Display Stones Count', 'pallet-sales-for-ecommerce'),
            array($this, 'display_stones_count_callback'),
            'pallsafo_settings',
            'pallsafo_general_section'
        );
    }

    public function settings_page()
    {
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(__('Pallet Sales Settings', 'pallet-sales-for-ecommerce')); ?></h1>
            <form method="post" action="options.php">
                <?php
                settings_fields('pallsafo_settings');
                do_settings_sections('pallsafo_settings');
                submit_button();
                ?>
            </form>
        </div>
<?php
    }

    public function general_section_callback()
    {
        echo '<p>' . esc_html__('Configure general settings for pallet sales functionality.', 'pallet-sales-for-ecommerce') . '</p>';
    }

    public function calculator_enabled_callback()
    {
        $value = get_option('pallsafo_calculator_enabled', 'yes');
        echo '<input type="checkbox" name="pallsafo_calculator_enabled" value="yes" ' . checked($value, 'yes', false) . ' />';
        echo '<label for="pallsafo_calculator_enabled">' . esc_html__('Show area calculator on product pages', 'pallet-sales-for-ecommerce') . '</label>';
    }

    public function weight_calculation_enabled_callback()
    {
        $value = get_option('pallsafo_weight_calculation_enabled', 'yes');
        echo '<input type="checkbox" name="pallsafo_weight_calculation_enabled" value="yes" ' . checked($value, 'yes', false) . ' />';
        echo '<label for="pallsafo_weight_calculation_enabled">' . esc_html__('Calculate and display pallet weight', 'pallet-sales-for-ecommerce') . '</label>';
    }

    public function display_stones_count_callback()
    {
        $value = get_option('pallsafo_display_stones_count', 'yes');
        echo '<input type="checkbox" name="pallsafo_display_stones_count" value="yes" ' . checked($value, 'yes', false) . ' />';
        echo '<label for="pallsafo_display_stones_count">' . esc_html__('Display stones count information', 'pallet-sales-for-ecommerce') . '</label>';
    }

    // Helper method to check if feature is enabled
    public function is_feature_enabled($feature)
    {
        return get_option('pallsafo_' . $feature . '_enabled', 'yes') === 'yes';
    }

    // Sanitization callback for checkbox settings
    public function sanitize_checkbox($input)
    {
        return $input === 'yes' ? 'yes' : 'no';
    }
}

new Pallet_Sales_For_Ecommerce();
