<?php

namespace Pacific\Core\Service;

use Pacific\Core\Exception\ArrayErrorsHttpExceptionInterface;
use Pacific\Core\Exception\HttpException;
use Pacific\Core\Exception\HttpExceptionInterface;
use Pacific\Core\Utils\Configuration;
use Symfony\Contracts\HttpClient\ResponseInterface;

class HttpExceptionService {
    private const EXCEPTIONS_NAMESPACE = "Pacific\Core\Exception";
    private $exceptionClasses;

    public function generateExceptionClasses()
    {
        $allClasses = require Configuration::rootPath() . '/../../composer/autoload_classmap.php';
        $length = strlen(self::EXCEPTIONS_NAMESPACE);

        foreach ($allClasses as $className => $classFile) {
            if (substr($className, 0, $length) === self::EXCEPTIONS_NAMESPACE) {
                $this->exceptionClasses[] = $className;
            }
        }
    }

	/**
	 * @param ResponseInterface $response
	 * @return void
     * @throws HttpExceptionInterface
	 */
    public function checkResponseAndThrow(ResponseInterface $response)
    {
        if ($response->getStatusCode() < 400) {
            return;
        }

        $responseContents = $response->getContent(false);
        $responseContents = json_decode($responseContents);
        if (!$responseContents) {
            return;
        }

        $responseContents->code = $responseContents->code ?? $responseContents->error;
        if (is_null($responseContents->code)) {
            return;
        }

        if(!$this->exceptionClasses) {
            $this->generateExceptionClasses();
        }

        foreach ($this->exceptionClasses as $class) {
            if (class_exists($class)) {
                $reflection = new \ReflectionClass($class);
                if ($reflection->isAbstract() || $reflection->isInterface()) {
                    continue;
                }

                if (
                    $reflection->implementsInterface(HttpExceptionInterface::class)
                    && strtoupper($responseContents->code) === $class::getStatusCode()
                ) {
                    $instance = new $class(
                        $responseContents->message
                        ?? $responseContents->error_description
                        ?? "Unknown error."
                    );

                    if (
                        $reflection->implementsInterface(ArrayErrorsHttpExceptionInterface::class)
                        && !empty($responseContents->errors)
                    ) {
                        $instance->setErrors($responseContents->errors);
                    }

                    throw $instance;
                }
            }
        }

        throw new HttpException(
            $response->getStatusCode() ?: 500,
            !empty($responseContents->message) ? $responseContents->message : "Unknown error."
        );
    }
}
