<?php

namespace Pacific\Core\Api;

use Pacific\Core\Dto\Checkout\CheckoutDataOutput;
use Pacific\Core\Dto\Shop\Price;
use Pacific\Core\Exception\HttpExceptionInterface;
use Pacific\Core\Service\Credentials\ClientCredentials;
use Pacific\Core\Dto\Checkout\CheckoutOrder;

class CheckoutContext extends Context {

    /**
     * Calculate cart value
     *
     * @param CheckoutOrder $checkoutOrder
     * @return Price
     * @throws HttpExceptionInterface
     */
	public function calculateCartValue(ClientCredentials $clientCredentials, CheckoutOrder $checkoutOrder)
	{
        $response = $this->request->doRequest(
            'POST',
            '/v1/e-commerce/checkouts/calculate',
            $clientCredentials,
            [
                'json' => $checkoutOrder,
                'headers' => [
                    'x-user-origin' => 'PLUGIN' // header required by pacific
                ]
            ]
        );

        $jsonDecode = json_decode($response->getContent(false), true);

        return $this->serializer->denormalize(
            $jsonDecode['value'],
            Price::class,
            'array'
        );
	}

    /**
     * Checkout cart
     *
     * @param ClientCredentials $clientCredentials
     * @param CheckoutOrder $checkoutOrder
     * @return string
     * @throws HttpExceptionInterface
     */
	public function checkout(ClientCredentials $clientCredentials, CheckoutOrder $checkoutOrder)
	{
        $userData = $this->_gateway->userContext()->getUserData($clientCredentials);

        $response = $this->request->doRequest(
            'POST',
            "/v1/e-commerce/checkouts/{$userData->uuid}",
            $clientCredentials,
            [
                'json' => $checkoutOrder,
                'headers' => [
                    'x-user-origin' => 'PLUGIN' // header required by pacific
                ]
            ]
        );

        // return $checkoutId
        return str_replace(
            $this->_gateway->configuration->baseUrl() . '/v1/e-commerce/checkouts/' . $userData->uuid . '/',
            "",
	        $response->getHeaders(false)['location'][0]
        );
	}

    /**
     * Checkout cart
     *
     * @param ClientCredentials $clientCredentials
     * @param $checkoutId
     * @return CheckoutDataOutput
     * @throws HttpExceptionInterface
     */
    public function getCheckoutData(ClientCredentials $clientCredentials, $checkoutId)
    {

        $userData = $this->_gateway->userContext()->getUserData($clientCredentials);
        $response = $this->request->doRequest(
            'GET',
            "/v1/e-commerce/checkouts/{$userData->uuid}/{$checkoutId}",
            $clientCredentials
        );

        // remove merchantID from response to match DTO
        $jsonDecode = json_decode($response->getContent(false), true);
        $jsonDecode['checkoutData'] = $jsonDecode['checkoutData'][$this->_gateway->configuration->getMerchantId()];

        return $this->serializer->deserialize(
            json_encode($jsonDecode),
            CheckoutDataOutput::class,
            'json'
        );
    }
}
