<?php

/**
 * Plugin Name: Paapi Product Search for Amazon
 * Description: Search and display Amazon products with filters and infinite scroll
 * Version: 2.8.3
 * Author: PaapiPlugin
 * Author URI: https://paapiplugin.com
 * License: GPL v2 or later
 * Text Domain: paapi-product-search-for-amazon
 */
// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}
if (!function_exists('psfa_fs')) {
    // Create a helper function for easy SDK access.
    function psfa_fs()
    {
        global $psfa_fs;

        if (!isset($psfa_fs)) {
            // Include Freemius SDK.
            require_once dirname(__FILE__) . '/vendor/freemius/start.php';

            $psfa_fs = fs_dynamic_init(array(
                'id' => '22205',
                'slug' => 'paapi-product-search-for-amazon',
                'type' => 'plugin',
                'public_key' => 'pk_970a210428f883332e658fd4e9240',
                'is_premium' => false,
                'is_org_compliant' => true,
                // If your plugin is a serviceware, set this option to false.
                'has_premium_version' => true,
                'has_addons' => true,
                'has_paid_plans' => true,
                'has_affiliation' => 'selected',
                'menu' => array(
                    'slug' => 'paapi-product-search-for-amazon',
                    'support' => false,
                    'addons' => true,
                    'parent' => array(
                        'slug' => 'options-general.php',
                    ),
                ),
            ));
        }

        return $psfa_fs;
    }

    // Init Freemius.
    psfa_fs();
    // Signal that SDK was initiated.
    do_action('psfa_fs_loaded');
}

// Define plugin constants
define('PSFA_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('PSFA_PLUGIN_URL', plugin_dir_url(__FILE__));
define('PSFA_VERSION', '2.5.0');

/**
 * Initialize the plugin
 */
class PSFA_Product_Search
{
    private static $instance = null;

    public static function get_instance()
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct()
    {
        // Load AJAX handlers
        require_once PSFA_PLUGIN_DIR . 'core/admin/ajax.php';

        // Load dev-only test tools (only in mock mode, excluded from free build)
        if (defined('PSFA_MOCK_MODE') && PSFA_MOCK_MODE) {
            if (file_exists(PSFA_PLUGIN_DIR . 'core/admin/offers-comparison-test.php')) {
                require_once PSFA_PLUGIN_DIR . 'core/admin/offers-comparison-test.php';
            }
            if (file_exists(PSFA_PLUGIN_DIR . 'core/admin/oauth-credentials-test.php')) {
                require_once PSFA_PLUGIN_DIR . 'core/admin/oauth-credentials-test.php';
            }
        }

        // Handle POST redirects early (before any output)
        add_action('template_redirect', array($this, 'handle_early_post_redirect'));

        // Add admin menu for settings
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_init', array($this, 'handle_reset_customization'));
        add_action('admin_init', array($this, 'handle_mock_section_redirect'));

        // Reset API client cache when auth settings change
        add_action('update_option_psfa_auth_type', array($this, 'reset_api_client_cache'));
        add_action('update_option_psfa_oauth_client_id', array($this, 'reset_api_client_cache'));
        add_action('update_option_psfa_oauth_client_secret', array($this, 'reset_api_client_cache'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_styles'));
        // Register shortcode (prefixed per WordPress.org guidelines)
        add_shortcode('psfa_search', array($this, 'render_search_page'));
        // Enqueue scripts and styles
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        // AJAX handler for infinite scroll
        add_action('wp_ajax_psfa_load_more', 'psfa_ajax_load_more');
        add_action('wp_ajax_nopriv_psfa_load_more', 'psfa_ajax_load_more');
        // AJAX handler for autocomplete
        add_action('wp_ajax_psfa_autocomplete', 'psfa_ajax_autocomplete');
        add_action('wp_ajax_nopriv_psfa_autocomplete', 'psfa_ajax_autocomplete');
        // AJAX handler for test connection (admin only)
        add_action('wp_ajax_psfa_test_connection', array($this, 'ajax_test_connection'));
    }

    /**
     * Handle POST form redirects early (before any output).
     * This ensures wp_safe_redirect works properly.
     */
    public function handle_early_post_redirect()
    {
        // Only process POST requests with our form fields
        if (!isset($_SERVER['REQUEST_METHOD']) || $_SERVER['REQUEST_METHOD'] !== 'POST') {
            return;
        }

        if (!isset($_POST['amazon_search_query']) &&
                !isset($_POST['amazon_search_brand']) &&
                !isset($_POST['amazon_search_price_min']) &&
                !isset($_POST['amazon_search_price_max'])) {
            return;
        }

        // Verify nonce
        $nonce = isset($_POST['psfa_nonce']) ? sanitize_text_field(wp_unslash($_POST['psfa_nonce'])) : '';
        if (empty($nonce) || !wp_verify_nonce($nonce, 'psfa_search_nonce')) {
            return;  // Let the shortcode handle the error
        }

        // Build redirect parameters
        $params = array();

        if (isset($_POST['amazon_search_query']) && !empty($_POST['amazon_search_query'])) {
            $params['q'] = sanitize_text_field(wp_unslash($_POST['amazon_search_query']));
        }
        if (isset($_POST['amazon_search_category']) && !empty($_POST['amazon_search_category'])) {
            $params['category'] = sanitize_text_field(wp_unslash($_POST['amazon_search_category']));
        }
        if (isset($_POST['amazon_search_price_min']) && $_POST['amazon_search_price_min'] !== '') {
            $params['min_price'] = floatval($_POST['amazon_search_price_min']);
        }
        if (isset($_POST['amazon_search_price_max']) && $_POST['amazon_search_price_max'] !== '') {
            $params['max_price'] = floatval($_POST['amazon_search_price_max']);
        }
        if (isset($_POST['amazon_search_brand']) && !empty($_POST['amazon_search_brand'])) {
            $params['brand'] = sanitize_text_field(wp_unslash($_POST['amazon_search_brand']));
        }
        if (isset($_POST['amazon_search_sort']) && !empty($_POST['amazon_search_sort'])) {
            $params['sort'] = sanitize_text_field(wp_unslash($_POST['amazon_search_sort']));
        }
        if (isset($_POST['amazon_search_condition']) && !empty($_POST['amazon_search_condition'])) {
            $params['condition'] = sanitize_text_field(wp_unslash($_POST['amazon_search_condition']));
        }
        if (isset($_POST['amazon_search_merchant']) && !empty($_POST['amazon_search_merchant'])) {
            $params['merchant'] = sanitize_text_field(wp_unslash($_POST['amazon_search_merchant']));
        }
        if (isset($_POST['amazon_search_availability']) && !empty($_POST['amazon_search_availability']) && $_POST['amazon_search_availability'] !== 'Available') {
            $params['availability'] = sanitize_text_field(wp_unslash($_POST['amazon_search_availability']));
        }
        // Premium filters
        if (isset($_POST['prime']) && !empty($_POST['prime'])) {
            $params['prime'] = sanitize_text_field(wp_unslash($_POST['prime']));
        }
        if (isset($_POST['has_deal']) && !empty($_POST['has_deal'])) {
            $params['has_deal'] = sanitize_text_field(wp_unslash($_POST['has_deal']));
        }
        if (isset($_POST['min_saving_percent']) && $_POST['min_saving_percent'] !== '') {
            $params['min_saving_percent'] = absint($_POST['min_saving_percent']);
        }

        // Only redirect if we have parameters
        if (empty($params)) {
            return;
        }

        // Build redirect URL
        $redirect_url = get_permalink();
        if (!empty($params)) {
            $join_char = (false === strpos($redirect_url, '?')) ? '?' : '&';
            $redirect_url .= $join_char . http_build_query($params);
        }

        wp_safe_redirect($redirect_url);
        exit;
    }

    /**
     * Add admin menu
     */
    public function add_admin_menu()
    {
        // Main settings page
        add_options_page(
            'Paapi Product Search for Amazon Settings',
            'Paapi Product Search for Amazon',
            'manage_options',
            'paapi-product-search-for-amazon',
            array($this, 'render_settings_page')
        );
    }

    /**
     * Register plugin settings
     */
    public function register_settings()
    {
        // Auth type selection
        register_setting('psfa_settings', 'psfa_auth_type', array(
            'sanitize_callback' => array($this, 'sanitize_auth_type'),
            'default' => 'awssig4',
        ));

        // AWS Sig V4 Credentials
        register_setting('psfa_settings', 'psfa_access_key_id', array(
            'sanitize_callback' => 'sanitize_text_field',
        ));
        register_setting('psfa_settings', 'psfa_secret_access_key', array(
            'sanitize_callback' => 'sanitize_text_field',
        ));
        register_setting('psfa_settings', 'psfa_associate_tag', array(
            'sanitize_callback' => 'sanitize_text_field',
        ));

        // OAuth 2.0 Credentials (Creators API)
        register_setting('psfa_settings', 'psfa_oauth_client_id', array(
            'sanitize_callback' => 'sanitize_text_field',
        ));
        register_setting('psfa_settings', 'psfa_oauth_client_secret', array(
            'sanitize_callback' => 'sanitize_text_field',
        ));
        register_setting('psfa_settings', 'psfa_oauth_credential_version', array(
            'sanitize_callback' => 'sanitize_text_field',
        ));
        register_setting('psfa_settings', 'psfa_oauth_auth_region', array(
            'sanitize_callback' => 'sanitize_text_field',
            'default' => 'us-east-1',
        ));
        register_setting('psfa_settings', 'psfa_marketplace', array(
            'sanitize_callback' => 'sanitize_text_field',
            'default' => 'www.amazon.com',
        ));

        // Theme selection
        register_setting('psfa_settings', 'psfa_selected_theme', array(
            'sanitize_callback' => array($this, 'sanitize_theme'),
            'default' => 'amazon-inspired',
        ));

        // Display mode (Pro only)
        register_setting('psfa_settings', 'psfa_display_mode', array(
            'sanitize_callback' => array($this, 'sanitize_display_mode'),
            'default' => 'grid',
        ));

        // Hide category banner (Free feature)
        register_setting('psfa_settings', 'psfa_hide_category_banner', array(
            'sanitize_callback' => array($this, 'sanitize_checkbox'),
            'default' => false,
        ));

        // Hide search bar (Free feature)
        register_setting('psfa_settings', 'psfa_hide_search', array(
            'sanitize_callback' => array($this, 'sanitize_checkbox'),
            'default' => false,
        ));

        // Hide filters (Free feature)
        register_setting('psfa_settings', 'psfa_hide_filters', array(
            'sanitize_callback' => array($this, 'sanitize_checkbox'),
            'default' => false,
        ));

        // Filter visibility settings (Free feature)
        register_setting('psfa_settings', 'psfa_filter_visibility', array(
            'sanitize_callback' => array($this, 'sanitize_filter_visibility'),
            'default' => array(
                'brand' => true,
                'price' => true,
                'sort' => true,
                'condition' => true,
                'merchant' => true,
                'availability' => true,
                'has_deal' => true,
                'deals' => true,
            ),
        ));

        // Premium customization settings
        register_setting('psfa_settings', 'psfa_button_text', array(
            'sanitize_callback' => array($this, 'sanitize_text_field_safe'),
            'default' => 'Buy on Amazon',
        ));
        register_setting('psfa_settings', 'psfa_button_bg_color', array(
            'sanitize_callback' => array($this, 'sanitize_hex_color_safe'),
        ));
        register_setting('psfa_settings', 'psfa_button_text_color', array(
            'sanitize_callback' => array($this, 'sanitize_hex_color_safe'),
        ));
        register_setting('psfa_settings', 'psfa_button_border_radius', array(
            'sanitize_callback' => array($this, 'sanitize_text_field_safe'),
        ));
        register_setting('psfa_settings', 'psfa_button_font_size', array(
            'sanitize_callback' => array($this, 'sanitize_text_field_safe'),
        ));
        register_setting('psfa_settings', 'psfa_button_font_weight', array(
            'sanitize_callback' => array($this, 'sanitize_text_field_safe'),
        ));
        register_setting('psfa_settings', 'psfa_theme_colors', array(
            'sanitize_callback' => array($this, 'sanitize_theme_colors'),
        ));
        register_setting('psfa_settings', 'psfa_card_layout', array(
            'sanitize_callback' => array($this, 'sanitize_text_field_safe'),
            'default' => 'default',
        ));
        register_setting('psfa_settings', 'psfa_title_font', array(
            'sanitize_callback' => array($this, 'sanitize_text_field_safe'),
        ));
        register_setting('psfa_settings', 'psfa_description_font', array(
            'sanitize_callback' => array($this, 'sanitize_text_field_safe'),
        ));
        register_setting('psfa_settings', 'psfa_price_font', array(
            'sanitize_callback' => array($this, 'sanitize_text_field_safe'),
        ));

        // Premium Feature Toggles
        register_setting('psfa_settings', 'psfa_features', array(
            'sanitize_callback' => array($this, 'sanitize_features'),
            'default' => array(
                'deal_badge' => true,
                'deal_countdown' => true,
                'prime_exclusive' => true,
                'low_stock' => true,
                'buybox_winner' => false,
                'used_price' => false,
            ),
        ));

        // White-label mode toggle (Premium) - when ON, hides branding
        register_setting('psfa_settings', 'psfa_white_label', array(
            'sanitize_callback' => array($this, 'sanitize_checkbox'),
            'default' => '0',
        ));
    }

    /**
     * Sanitize auth type selection
     */
    public function sanitize_auth_type($type)
    {
        $valid_types = array('awssig4', 'oauth');
        if (in_array($type, $valid_types, true)) {
            return $type;
        }
        return 'awssig4';
    }

    /**
     * Reset API client cache when auth settings change
     */
    public function reset_api_client_cache()
    {
        require_once PSFA_PLUGIN_DIR . 'core/api/class-api-factory.php';
        PSFA_API_Factory::reset();
        // Clear cached OAuth token to force re-authentication
        delete_option('psfa_oauth_access_token');
        delete_option('psfa_oauth_token_expires');
    }

    /**
     * AJAX handler for testing API connection
     */
    public function ajax_test_connection()
    {
        check_ajax_referer('psfa_test_connection', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
            return;
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in check_ajax_referer above
        $auth_type = isset($_POST['auth_type']) ? sanitize_text_field(wp_unslash($_POST['auth_type'])) : 'awssig4';

        if ($auth_type === 'oauth') {
            $result = $this->test_oauth_connection();
        } else {
            $result = $this->test_awssig4_connection();
        }

        if ($result['success']) {
            wp_send_json_success(['message' => $result['message']]);
        } else {
            wp_send_json_error(['message' => $result['message']]);
        }
    }

    /**
     * Test OAuth connection with provided credentials
     */
    private function test_oauth_connection(): array
    {
        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verified in ajax_test_connection
        $client_id = isset($_POST['client_id']) ? sanitize_text_field(wp_unslash($_POST['client_id'])) : '';
        $client_secret = isset($_POST['client_secret']) ? sanitize_text_field(wp_unslash($_POST['client_secret'])) : '';
        $credential_version = isset($_POST['credential_version']) ? sanitize_text_field(wp_unslash($_POST['credential_version'])) : '';
        $auth_region = isset($_POST['auth_region']) ? sanitize_text_field(wp_unslash($_POST['auth_region'])) : 'us-east-1';
        $marketplace = isset($_POST['marketplace']) ? sanitize_text_field(wp_unslash($_POST['marketplace'])) : 'www.amazon.com';
        $associate_tag = isset($_POST['associate_tag']) ? sanitize_text_field(wp_unslash($_POST['associate_tag'])) : '';
        // phpcs:enable WordPress.Security.NonceVerification.Missing

        if (empty($client_id) || empty($client_secret) || empty($credential_version)) {
            return ['success' => false, 'message' => 'Missing required OAuth credentials'];
        }

        // Get OAuth token
        $token_url = "https://creatorsapi.auth.{$auth_region}.amazoncognito.com/oauth2/token";
        $basic_auth = base64_encode($client_id . ':' . $client_secret);

        $token_response = wp_remote_post($token_url, [
            'headers' => [
                'Content-Type' => 'application/x-www-form-urlencoded',
                'Authorization' => 'Basic ' . $basic_auth,
            ],
            'body' => 'grant_type=client_credentials&scope=creatorsapi/default',
            'timeout' => 30,
        ]);

        if (is_wp_error($token_response)) {
            return ['success' => false, 'message' => 'Token request failed: ' . $token_response->get_error_message()];
        }

        $token_code = wp_remote_retrieve_response_code($token_response);
        $token_body = json_decode(wp_remote_retrieve_body($token_response), true);

        if ($token_code !== 200 || !isset($token_body['access_token'])) {
            $error = $token_body['error'] ?? 'Unknown error';
            return ['success' => false, 'message' => 'Token error: ' . $error];
        }

        $access_token = $token_body['access_token'];

        // Test API call
        require_once PSFA_PLUGIN_DIR . 'core/api/payload.php';
        $api_url = 'https://creatorsapi.amazon/catalog/v1/searchItems';
        $payload = [
            'keywords' => 'test',
            'partnerTag' => $associate_tag,
            'marketplace' => $marketplace,
            'resources' => psfa_get_creators_api_resources(),
            'itemCount' => 1,
        ];

        $api_response = wp_remote_post($api_url, [
            'headers' => [
                'Authorization' => 'Bearer ' . $access_token . ', Version ' . $credential_version,
                'Content-Type' => 'application/json',
                'x-marketplace' => $marketplace,
            ],
            'body' => wp_json_encode($payload),
            'timeout' => 15,
        ]);

        if (is_wp_error($api_response)) {
            return ['success' => false, 'message' => 'API request failed: ' . $api_response->get_error_message()];
        }

        $api_code = wp_remote_retrieve_response_code($api_response);

        if ($api_code === 200) {
            return ['success' => true, 'message' => 'Connection successful! OAuth credentials are valid.'];
        }

        $api_body = json_decode(wp_remote_retrieve_body($api_response), true);
        $error = $api_body['errors'][0]['message'] ?? 'HTTP ' . $api_code;
        return ['success' => false, 'message' => 'API error: ' . $error];
    }

    /**
     * Test AWS Sig V4 connection with provided credentials
     */
    private function test_awssig4_connection(): array
    {
        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verified in ajax_test_connection
        $access_key = isset($_POST['access_key']) ? sanitize_text_field(wp_unslash($_POST['access_key'])) : '';
        $secret_key = isset($_POST['secret_key']) ? sanitize_text_field(wp_unslash($_POST['secret_key'])) : '';
        $associate_tag = isset($_POST['associate_tag']) ? sanitize_text_field(wp_unslash($_POST['associate_tag'])) : '';
        // phpcs:enable WordPress.Security.NonceVerification.Missing

        if (empty($access_key) || empty($secret_key) || empty($associate_tag)) {
            return ['success' => false, 'message' => 'Missing required AWS credentials'];
        }

        // Use existing AWS Sig4 client to test
        require_once PSFA_PLUGIN_DIR . 'core/api/class-awssig4-client.php';
        require_once PSFA_PLUGIN_DIR . 'core/api/payload.php';

        $config = [
            'access_key' => $access_key,
            'secret_key' => $secret_key,
            'associate_tag' => $associate_tag,
            'region' => 'us-east-1',
            'host' => 'webservices.amazon.com',
        ];

        $client = new PSFA_AWSSig4_Client($config);
        $payload = psfa_build_search_payload(['search_term' => 'test', 'page' => 1], $associate_tag);

        if (isset($payload['error'])) {
            return ['success' => false, 'message' => $payload['error']];
        }

        $result = $client->search_items($payload);

        if (isset($result['error'])) {
            return ['success' => false, 'message' => $result['error']];
        }

        return ['success' => true, 'message' => 'Connection successful! AWS credentials are valid.'];
    }

    /**
     * Sanitize theme selection
     */
    public function sanitize_theme($theme)
    {
        require_once PSFA_PLUGIN_DIR . 'core/shared/theme-manager.php';
        return psfa_validate_theme($theme);
    }

    /**
     * Sanitize display mode
     *
     * IMPORTANT:
     * When saving from other tabs (themes/customization), the display mode field
     * is not posted. In those cases WordPress may pass null/empty here.
     * To avoid resetting the user's choice back to "grid", we keep the existing
     * option value when no new value is submitted.
     */
    public function sanitize_display_mode($mode)
    {
        if ($mode === null || $mode === '') {
            $existing = get_option('psfa_display_mode', 'grid');
            return $existing ? $existing : 'grid';
        }
        $valid_modes = array('grid', 'list');
        if (in_array($mode, $valid_modes)) {
            return $mode;
        }
        return 'grid';
    }

    /**
     * Sanitize text field safely (handles null values)
     */
    public function sanitize_text_field_safe($value)
    {
        if ($value === null || $value === false) {
            return '';
        }
        return sanitize_text_field($value);
    }

    /**
     * Sanitize hex color safely (handles null values)
     */
    public function sanitize_hex_color_safe($value)
    {
        if ($value === null || $value === false || $value === '') {
            return '';
        }
        return sanitize_hex_color($value);
    }

    /**
     * Sanitize checkbox value
     */
    public function sanitize_checkbox($value)
    {
        return isset($value) && $value === '1' ? '1' : '0';
    }

    /**
     * Sanitize filter visibility settings
     */
    public function sanitize_filter_visibility($input)
    {
        $valid_filters = array('brand', 'price', 'sort', 'condition', 'merchant', 'availability', 'has_deal', 'deals');
        $sanitized = array();

        foreach ($valid_filters as $filter) {
            $sanitized[$filter] = !empty($input[$filter]);
        }

        return $sanitized;
    }

    /**
     * Sanitize premium feature toggles
     */
    public function sanitize_features($input)
    {
        $valid_features = array('deal_badge', 'deal_countdown', 'prime_exclusive', 'low_stock', 'buybox_winner', 'used_price');
        $sanitized = array();

        foreach ($valid_features as $feature) {
            $sanitized[$feature] = !empty($input[$feature]);
        }

        return $sanitized;
    }

    /**
     * Sanitize theme colors
     */
    public function sanitize_theme_colors($colors)
    {
        if (!is_array($colors)) {
            return array();
        }

        require_once PSFA_PLUGIN_DIR . 'core/shared/customization-helpers.php';
        require_once PSFA_PLUGIN_DIR . 'core/shared/theme-manager.php';

        $selected_theme = psfa_get_selected_theme();
        $default_colors = psfa_get_theme_default_colors($selected_theme);

        $sanitized = array();
        foreach ($colors as $theme => $theme_colors) {
            if (!is_array($theme_colors)) {
                continue;
            }
            $sanitized[$theme] = array();
            foreach ($theme_colors as $key => $value) {
                if ($value === null || $value === false || $value === '') {
                    // Skip empty values - don't save them
                    continue;
                }
                if (strpos($key, 'color') !== false) {
                    $sanitized_value = sanitize_hex_color($value);
                    // Only add if it's a valid color AND it's different from default
                    if ($sanitized_value) {
                        $default_value = $default_colors[$key] ?? '';
                        // Only save if it's different from default
                        if ($sanitized_value !== $default_value) {
                            $sanitized[$theme][$key] = $sanitized_value;
                        }
                    }
                } else {
                    $sanitized[$theme][$key] = sanitize_text_field($value);
                }
            }
            // If theme has no colors, don't add it to the array
            if (empty($sanitized[$theme])) {
                unset($sanitized[$theme]);
            }
        }
        return $sanitized;
    }

    /**
     * Helper function: Check if premium plan is active (Freemius)
     */
    private function is_premium()
    {
        if (function_exists('psfa_fs')) {
            return psfa_fs()->is_plan('personal');
        }
        return false;
    }

    /**
     * Helper function: Get max pages allowed
     */
    private function get_max_pages()
    {
        if ($this->is_premium()) {
            return 999;
            // Unlimited for premium
        }
        return 5;
        // Free limit
    }

    /**
     * Helper function: Get available categories
     */
    private function get_available_categories()
    {
        $all_categories = array(
            'Electronics',
            'Books',
            'Fashion',
            'ToysAndGames',
            'HomeGarden',
            'TodaysDeals',
            'Automotive',
            'Baby',
            'Beauty',
            'Computers',
            'Grocery',
            'HealthPersonalCare',
            'HomeImprovement',
            'Industrial',
            'KindleStore',
            'Luggage',
            'MoviesTV',
            'Music',
            'MusicalInstruments',
            'OfficeProducts',
            'PetSupplies',
            'Software',
            'SportsOutdoors',
            'ToolsHomeImprovement',
            'VideoGames',
            'Watches'
        );
        if ($this->is_premium()) {
            return $all_categories;
        }
        // Free version: only 5 categories
        return array(
            'Electronics',
            'Books',
            'Fashion',
            'ToysAndGames',
            'HomeGarden'
        );
    }

    /**
     * Helper function: Check if category is available
     */
    private function is_category_available($category)
    {
        $available = $this->get_available_categories();
        return in_array($category, $available);
    }

    /**
     * Enqueue admin styles and scripts
     */
    public function enqueue_admin_styles($hook)
    {
        // Only load on our settings page (options page hook format: settings_page_{slug})
        if ($hook !== 'settings_page_paapi-product-search-for-amazon') {
            return;
        }

        // Load Tailwind CSS via CDN
        require_once PSFA_PLUGIN_DIR . 'core/admin/admin-styles.php';
        psfa_enqueue_tailwind_admin();
        wp_add_inline_style('wp-admin', psfa_get_tailwind_admin_styles());

        // Load legacy admin styles for sections that still use them (themes, layout, shortcode)
        require_once PSFA_PLUGIN_DIR . 'core/admin/styles.php';
        wp_add_inline_style('wp-admin', psfa_get_admin_styles());

        // Register a dummy script handle to attach inline JS to
        wp_register_script('psfa-admin-scripts', false, array(), PSFA_VERSION, true);
        wp_enqueue_script('psfa-admin-scripts');
        wp_add_inline_script('psfa-admin-scripts', psfa_get_admin_scripts());
    }

    /**
     * Handle reset customization request
     * Called early in admin_init to avoid headers already sent errors
     */
    public function handle_reset_customization()
    {
        // Only process on our settings page
        if (!isset($_GET['page']) || $_GET['page'] !== 'paapi-product-search-for-amazon') {
            return;
        }

        // Check if reset was requested via GET (link)
        if (isset($_GET['psfa_reset_customization']) && $_GET['psfa_reset_customization'] === '1') {
            // Verify nonce for GET request
            $nonce = isset($_GET['_wpnonce']) ? sanitize_text_field(wp_unslash($_GET['_wpnonce'])) : '';
            if (empty($nonce) || !wp_verify_nonce($nonce, 'psfa_reset_customization')) {
                wp_die('Security check failed');
            }

            // Determine current section so we can reset only that section's settings.
            // Support both 'section' (new) and 'tab' (legacy) parameters
            $section = 'customization';
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only parameter used for redirect and branching.
            $raw_section = isset($_GET['section']) ? sanitize_text_field(wp_unslash($_GET['section'])) : (isset($_GET['tab']) ? sanitize_text_field(wp_unslash($_GET['tab'])) : '');
            if (!empty($raw_section)) {
                $sanitized_section = sanitize_text_field($raw_section);
                $allowed_sections = array('credentials', 'themes', 'customization', 'features', 'layout');
                if (in_array($sanitized_section, $allowed_sections, true)) {
                    $section = $sanitized_section;
                }
            }

            // Reset settings depending on section (section-scoped reset).
            if ($section === 'themes') {
                // Reset only theme selection; other fields are preserved via hidden inputs.
                update_option('psfa_selected_theme', 'amazon-inspired');
            } elseif ($section === 'layout') {
                // Reset only layout-related options.
                update_option('psfa_display_mode', 'grid');
                update_option('psfa_hide_category_banner', '0');
                update_option('psfa_hide_search', '0');
                update_option('psfa_hide_filters', '0');
                update_option('psfa_filter_visibility', array(
                    'brand' => true,
                    'price' => true,
                    'sort' => true,
                    'condition' => true,
                    'merchant' => true,
                    'availability' => true,
                    'has_deal' => true,
                    'deals' => true,
                ));
            } elseif ($section === 'customization') {
                // Reset button, fonts, card layout, and theme colors.
                update_option('psfa_button_text', 'Buy on Amazon');
                update_option('psfa_button_border_radius', '');
                update_option('psfa_button_font_size', '');
                update_option('psfa_button_font_weight', '');

                // Theme colors (delete so theme defaults are used).
                delete_option('psfa_theme_colors');

                // Card layout & fonts.
                update_option('psfa_card_layout', 'default');
                update_option('psfa_title_font', '');
                update_option('psfa_description_font', '');
                update_option('psfa_price_font', '');
            } elseif ($section === 'features') {
                // Reset feature toggles to defaults.
                delete_option('psfa_features');
            }

            // Redirect immediately before any output, keeping current section.
            $redirect_url = add_query_arg(
                array(
                    'page' => 'paapi-product-search-for-amazon',
                    'section' => $section,
                    'settings-updated' => 'true',
                    'reset' => '1',
                ),
                admin_url('admin.php')
            );
            wp_safe_redirect($redirect_url);
            exit;
        }
    }

    /**
     * Redirect mock-mode-only sections to credentials if not in mock mode
     * Called early in admin_init to avoid headers already sent errors
     */
    public function handle_mock_section_redirect()
    {
        // phpcs:disable WordPress.Security.NonceVerification.Recommended -- Read-only redirect check
        // Only process on our settings page
        if (!isset($_GET['page']) || $_GET['page'] !== 'paapi-product-search-for-amazon') {
            return;
        }

        $section = isset($_GET['section']) ? sanitize_text_field(wp_unslash($_GET['section'])) : '';
        // phpcs:enable WordPress.Security.NonceVerification.Recommended

        $mock_only_sections = array('api-offers-test', 'api-oauth-test');
        if (in_array($section, $mock_only_sections, true) && !(defined('PSFA_MOCK_MODE') && PSFA_MOCK_MODE)) {
            wp_safe_redirect(admin_url('admin.php?page=paapi-product-search-for-amazon&section=credentials'));
            exit;
        }
    }

    /**
     * Render settings page
     */
    public function render_settings_page()
    {
        if (!current_user_can('manage_options')) {
            return;
        }

        // Display success message if reset was performed
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Display-only parameter, wp_unslash handles sanitization, value is compared to literal '1'
        if (isset($_GET['reset']) && wp_unslash($_GET['reset']) === '1') {
            add_settings_error(
                'psfa_settings',
                'psfa_settings_reset',
                'Settings have been reset to defaults.',
                'success'
            );
        }

        // Use new admin template with sidebar navigation and Tailwind CSS
        require_once PSFA_PLUGIN_DIR . 'core/admin/admin-template.php';
        psfa_render_admin_page();
    }

    /**
     * Enqueue scripts and styles
     */
    public function enqueue_scripts()
    {
        // Check if shortcode is in post content
        global $post;
        $has_shortcode = false;
        if ($post instanceof WP_Post && !empty($post->post_content)) {
            $has_shortcode = has_shortcode($post->post_content, 'psfa_search');
        }

        // Always load in mock mode (demo sites)
        $is_mock_mode = defined('PSFA_MOCK_MODE') && PSFA_MOCK_MODE;

        if ($has_shortcode || $is_mock_mode) {
            require_once PSFA_PLUGIN_DIR . 'core/shared/theme-manager.php';

            // Load base CSS first (contains all structural/layout styles)
            wp_enqueue_style(
                'psfa-base',
                PSFA_PLUGIN_URL . 'assets/css/style.css',
                array(),
                PSFA_VERSION
            );

            // Load theme CSS (contains only color/visual overrides)
            $selected_theme = psfa_get_selected_theme();
            $theme_css = psfa_get_theme_css_path($selected_theme);

            if ($theme_css) {
                wp_enqueue_style(
                    'psfa-theme',
                    $theme_css,
                    array('psfa-base'),  // Depend on base styles
                    PSFA_VERSION
                );
            }

            // Enqueue built JavaScript bundle (IIFE format)
            // All modules are bundled together in a single file
            wp_enqueue_script(
                'psfa-main',
                PSFA_PLUGIN_URL . 'assets/js/main.js',
                array(),
                PSFA_VERSION,
                true
            );

            // Localize script for AJAX
            wp_localize_script('psfa-main', 'psfaAjax', array(
                'ajaxurl' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('psfa_ajax_nonce'),
                'pluginUrl' => plugin_dir_url(__FILE__),
            ));

            // Add custom CSS for badges, colors, fonts, and button styles
            require_once PSFA_PLUGIN_DIR . 'core/shared/css-generator.php';
            $custom_css = psfa_get_customization_css();
            if (!empty($custom_css)) {
                wp_add_inline_style('psfa-theme', $custom_css);
            }
        }
    }

    /**
     * Render the search page (shortcode handler)
     *
     * Supported shortcode attributes:
     * - category: Pre-select a category (e.g., "Electronics")
     * - auto_detect: Auto-detect category from page context (default: "yes")
     * - query: Pre-fill search query
     * - brand: Pre-fill brand filter
     * - sort: Pre-set sort order
     * - hide_search: Hide the search bar ("true" or "false")
     * - hide_filters: Hide all filters ("true" or "false")
     * - pagination: Pagination type ("infinite", "button", or "none")
     *
     * Examples:
     * [psfa_search] - Default with search and filters
     * [psfa_search query="laptop" category="Electronics"]
     * [psfa_search query="headphones" hide_search="true" hide_filters="true"]
     * [psfa_search category="Books" pagination="button"]
     */
    public function render_search_page($atts)
    {
        // Parse shortcode attributes
        $atts = shortcode_atts(array(
            'category' => '',
            'auto_detect' => 'yes',
            'query' => '',
            'brand' => '',
            'sort' => '',
            'hide_search' => '',
            'hide_filters' => '',
            'hide_category_banner' => '',
            'pagination' => 'infinite',  // infinite, button, none
        ), $atts, 'psfa_search');

        // Allow auto_detect to work with 'yes' value
        if (strtolower($atts['auto_detect']) === 'yes') {
            $atts['auto_detect'] = 'no';
        }

        // Normalize boolean attributes - keep empty string if not explicitly set (falls through to admin setting)
        if ($atts['hide_search'] !== '') {
            $atts['hide_search'] = in_array(strtolower($atts['hide_search']), array('true', '1', 'yes'), true);
        }
        if ($atts['hide_filters'] !== '') {
            $atts['hide_filters'] = in_array(strtolower($atts['hide_filters']), array('true', '1', 'yes'), true);
        }
        if ($atts['hide_category_banner'] !== '') {
            $atts['hide_category_banner'] = in_array(strtolower($atts['hide_category_banner']), array('true', '1', 'yes'), true);
        }

        // Validate pagination type
        if (!in_array($atts['pagination'], array('infinite', 'button', 'none'), true)) {
            $atts['pagination'] = 'infinite';
        }

        // Auto-detect category from page context if enabled and no category specified
        if (empty($atts['category']) && $atts['auto_detect'] === 'no') {
            $atts['category'] = $this->detect_category_from_context();
        }

        // Pass shortcode attributes to template via global variable
        global $psfa_shortcode_atts;
        $psfa_shortcode_atts = $atts;

        // Legacy support: also set category separately
        global $psfa_shortcode_category;
        $psfa_shortcode_category = $atts['category'];
        // Include the main search functionality
        ob_start();
        include PSFA_PLUGIN_DIR . 'core/frontend/template.php';
        return ob_get_clean();
    }

    /**
     * Detect category from page context (slug, title, etc.)
     */
    private function detect_category_from_context()
    {
        global $post;
        if (!$post) {
            return '';
        }
        // Get page slug
        $slug = $post->post_name;
        $title = strtolower($post->post_title);
        // Category mapping (case-insensitive)
        $category_keywords = array(
            'fashion' => 'Fashion',
            'apparel' => 'Fashion',
            'clothing' => 'Fashion',
            'electronics' => 'Electronics',
            'books' => 'Books',
            'toys' => 'ToysAndGames',
            'toysandgames' => 'ToysAndGames',
            'home' => 'HomeGarden',
            'homegarden' => 'HomeGarden',
            'garden' => 'HomeGarden',
        );
        // Check slug
        foreach ($category_keywords as $keyword => $category) {
            if (strpos($slug, $keyword) !== false) {
                return $category;
            }
        }
        // Check title
        foreach ($category_keywords as $keyword => $category) {
            if (strpos($title, $keyword) !== false) {
                return $category;
            }
        }
        // Check page template name (if using custom template)
        $template = get_page_template_slug($post->ID);
        if ($template) {
            foreach ($category_keywords as $keyword => $category) {
                if (strpos(strtolower($template), $keyword) !== false) {
                    return $category;
                }
            }
        }
        return '';
    }
}

// Initialize the plugin
PSFA_Product_Search::get_instance();
