<?php
/**
 * Theme Manager for Paapi Product Search for Amazon
 *
 * Handles theme registration, loading, and validation using Freemius API.
 */

if (!defined('ABSPATH')) {
    exit;
}

// Include view helpers for psfa_is_premium()
require_once PSFA_PLUGIN_DIR . 'core/search/view-helpers.php';

/**
 * Get all available themes
 *
 * @return array Array of theme definitions
 */
function psfa_get_all_themes() {
    return array(
        'amazon-inspired' => array(
            'name' => 'Amazon Inspired',
            'description' => 'Classic Amazon-style design with orange accents',
            'premium' => false, // Free theme
            'file' => 'amazon-inspired/style.css',
        ),
        'ocean-blue' => array(
            'name' => 'Ocean Blue',
            'description' => 'Clean and modern ocean blue theme',
            'premium' => true,
            'file' => 'ocean-blue/style.css',
        ),
        'modern' => array(
            'name' => 'Modern',
            'description' => 'Sleek modern design with smooth transitions',
            'premium' => true,
            'file' => 'modern/style.css',
        ),
        'minimal' => array(
            'name' => 'Minimal',
            'description' => 'Minimalist design with clean lines',
            'premium' => true,
            'file' => 'minimal/style.css',
        ),
        'dark' => array(
            'name' => 'Dark',
            'description' => 'Dark theme for reduced eye strain',
            'premium' => true,
            'file' => 'dark/style.css',
        ),
    );
}

/**
 * Get current selected theme
 *
 * @return string Theme slug
 */
function psfa_get_selected_theme() {
    $selected = get_option('psfa_selected_theme', 'amazon-inspired');
    $themes = psfa_get_all_themes();
    
    // Validate theme exists
    if (!isset($themes[$selected])) {
        return 'amazon-inspired';
    }
    
    
    
    return $selected;
}

/**
 * Check if theme is available for current user
 *
 * @param string $theme_slug Theme slug
 * @return bool
 */
function psfa_is_theme_available($theme_slug) {
    $themes = psfa_get_all_themes();
    
    if (!isset($themes[$theme_slug])) {
        return false;
    }
    
    // Free themes are always available
    if (!$themes[$theme_slug]['premium']) {
        return true;
    }
    
    
    return true; // All themes available in free build
}

/**
 * Get theme CSS file path
 *
 * @param string $theme_slug Theme slug
 * @return string|false Theme CSS file path or false if not found
 */
function psfa_get_theme_css_path($theme_slug) {
    $themes = psfa_get_all_themes();
    
    if (!isset($themes[$theme_slug])) {
        return false;
    }
    
    $theme_file = $themes[$theme_slug]['file'];
    $theme_path = PSFA_PLUGIN_DIR . 'assets/themes/' . $theme_file;
    
    if (file_exists($theme_path)) {
        return PSFA_PLUGIN_URL . 'assets/themes/' . $theme_file;
    }
    
    return false;
}

/**
 * Validate and sanitize theme selection
 *
 * @param string $theme_slug Theme slug to validate
 * @return string Valid theme slug
 */
function psfa_validate_theme($theme_slug) {
    $themes = psfa_get_all_themes();
    
    // Check if theme exists
    if (!isset($themes[$theme_slug])) {
        return 'amazon-inspired';
    }
    
    // If premium theme and user is not premium, return free theme
    if ($themes[$theme_slug]['premium'] && !psfa_is_premium()) {
        return 'amazon-inspired';
    }
    
    return $theme_slug;
}

