<?php
/**
 * CSS Generator for customization features
 *
 * Generates dynamic CSS for themes, colors, fonts, buttons, and card layouts.
 * Extracted from main plugin class to improve maintainability and follow SOLID principles.
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Generate all customization CSS
 *
 * @return string Generated CSS
 */
function psfa_get_customization_css() {
    require_once PSFA_PLUGIN_DIR . 'core/shared/customization-helpers.php';
    require_once PSFA_PLUGIN_DIR . 'core/shared/theme-manager.php';
    
    
    $selected_theme = psfa_get_selected_theme();
    
    /* Free version defaults */
    $theme_colors = isset($theme_colors) ? $theme_colors : array();
    $button_style = isset($button_style) ? $button_style : array();
    $font_settings = isset($font_settings) ? $font_settings : array();
    $card_layout = isset($card_layout) ? $card_layout : 'default';
    $is_premium = isset($is_premium) ? $is_premium : false;
    
    $css = '';
    
    // Get default colors and merge with custom colors (custom takes precedence)
    $default_colors = psfa_get_theme_default_colors($selected_theme);
    $final_colors = array_merge($default_colors, $theme_colors);
    
    // Generate CSS for each customization type
    $css .= psfa_generate_button_css($button_style, $final_colors, $selected_theme);
    $css .= psfa_generate_theme_colors_css($selected_theme, $final_colors);
    $css .= psfa_generate_font_css($font_settings);
    $css .= psfa_generate_card_layout_css($card_layout, $is_premium);
    
    return $css;
}

/**
 * Generate CSS for button customization
 *
 * @param array $button_style Button style settings
 * @param array $final_colors Final color settings (defaults + custom)
 * @param string $selected_theme Selected theme slug
 * @return string Generated CSS
 */
function psfa_generate_button_css($button_style, $final_colors, $selected_theme) {
    $css = '';
    
    // Get button colors from theme colors
    $button_bg_from_theme = $final_colors['button_bg'] ?? '';
    $button_text_from_theme = $final_colors['button_text'] ?? '';
    
    // Button customization (text, border radius, font size, font weight)
    $button_css_needed = false;
    $button_css = '';
    if (!empty($button_style['border_radius']) || !empty($button_style['font_size']) || 
        !empty($button_style['font_weight']) || (!empty($button_bg_from_theme) && $button_bg_from_theme !== 'transparent') || !empty($button_text_from_theme)) {
        $button_css_needed = true;
        $button_css = '.psfa-buy-button {';
        if (!empty($button_bg_from_theme) && $button_bg_from_theme !== 'transparent') {
            $button_css .= 'background-color: ' . esc_attr($button_bg_from_theme) . ' !important;';
        } elseif ($button_bg_from_theme === 'transparent') {
            $button_css .= 'background: transparent !important;';
        }
        if (!empty($button_text_from_theme)) {
            $button_css .= 'color: ' . esc_attr($button_text_from_theme) . ' !important;';
        }
        if (!empty($button_style['border_radius'])) {
            $button_css .= 'border-radius: ' . esc_attr($button_style['border_radius']) . ' !important;';
        }
        if (!empty($button_style['font_size'])) {
            $button_css .= 'font-size: ' . esc_attr($button_style['font_size']) . ' !important;';
        }
        if (!empty($button_style['font_weight'])) {
            $button_css .= 'font-weight: ' . esc_attr($button_style['font_weight']) . ' !important;';
        }
        $button_css .= '}';
        
        // Add hover state for minimal theme (transparent button with hover fill)
        if ($selected_theme === 'minimal' && $button_bg_from_theme === 'transparent' && !empty($final_colors['primary_color'])) {
            $button_css .= '.psfa-buy-button:hover {';
            $button_css .= 'background-color: ' . esc_attr($final_colors['primary_color']) . ' !important;';
            $button_css .= 'color: white !important;';
            $button_css .= '}';
        }
    }
    
    if ($button_css_needed) {
        $css .= $button_css;
    }
    
    return $css;
}

/**
 * Generate CSS for theme colors
 *
 * @param string $selected_theme Selected theme slug
 * @param array $final_colors Final color settings (defaults + custom)
 * @return string Generated CSS
 */
function psfa_generate_theme_colors_css($selected_theme, $final_colors) {
    $css = '';
    
    // Get primary color - $final_colors already has defaults merged with custom colors
    // So it should always have a primary_color (either custom or default)
    $primary_color = $final_colors['primary_color'] ?? '';
    
    // Fallback to defaults if somehow primary_color is missing
    if (empty($primary_color)) {
        require_once PSFA_PLUGIN_DIR . 'core/shared/customization-helpers.php';
        $default_colors = psfa_get_theme_default_colors($selected_theme);
        $primary_color = $default_colors['primary_color'] ?? '';
    }
    
    // CSS variables
    $css .= '.psfa-theme-' . esc_attr($selected_theme) . ' {'; 
    if (!empty($primary_color)) {
        $css .= '--psfa-primary-color: ' . esc_attr($primary_color) . ';';
    }
    if (!empty($final_colors['text_color'])) {
        $css .= '--psfa-text-color: ' . esc_attr($final_colors['text_color']) . ';';
    }
    if (!empty($final_colors['price_color'])) {
        $css .= '--psfa-price-color: ' . esc_attr($final_colors['price_color']) . ';';
    }
    $css .= '}';
    
    // Spinner color is generated in psfa_generate_primary_color_accent_css() to avoid duplication
    
    if (empty($final_colors)) {
        return $css;
    }
    
    // Text color
    if (!empty($final_colors['text_color'])) {
        $css .= '.psfa-theme-' . esc_attr($selected_theme) . ' .psfa-product h7, ';
        $css .= '.psfa-theme-' . esc_attr($selected_theme) . ' .psfa-product .psfa-brand {';
        $css .= 'color: ' . esc_attr($final_colors['text_color']) . ' !important;';
        $css .= '}';
    }
    
    // Price color
    if (!empty($final_colors['price_color'])) {
        $css .= '.psfa-theme-' . esc_attr($selected_theme) . ' .psfa-product .psfa-price {';
        $css .= 'color: ' . esc_attr($final_colors['price_color']) . ' !important;';
        $css .= '}';
    }
    
    // Filter icon color - use text_color (now using currentColor in inline SVG, so just set color property)
    if (!empty($final_colors['text_color'])) {
        $css .= '.psfa-theme-' . esc_attr($selected_theme) . ' .psfa-filters-icon {';
        $css .= 'color: ' . esc_attr($final_colors['text_color']) . ' !important;';
        $css .= '}';
    }
    
    // Search button
    if (!empty($final_colors['primary_color'])) {
        $css .= '.psfa-theme-' . esc_attr($selected_theme) . ' .psfa-search-btn {';
        $css .= 'background-color: ' . esc_attr($final_colors['primary_color']) . ' !important;';
        $css .= '}';
        $css .= '.psfa-theme-' . esc_attr($selected_theme) . ' .psfa-search-btn:hover {';
        $css .= 'background-color: ' . esc_attr($final_colors['primary_color']) . ' !important;';
        $css .= 'opacity: 0.9;';
        $css .= '}';
    }
    
    // Reset button - handle minimal theme special case (transparent background)
    if (!empty($final_colors['secondary_color'])) {
        $css .= '.psfa-theme-' . esc_attr($selected_theme) . ' .psfa-reset-btn {';
        // For minimal theme, secondary color is text color, not background
        if ($selected_theme === 'minimal') {
            $css .= 'color: ' . esc_attr($final_colors['secondary_color']) . ' !important;';
        } else {
            $css .= 'background-color: ' . esc_attr($final_colors['secondary_color']) . ' !important;';
        }
        $css .= '}';
    }
    
    // Apply filters button
    if (!empty($final_colors['apply_btn_color'])) {
        $css .= '.psfa-theme-' . esc_attr($selected_theme) . ' .apply-btn {';
        $css .= 'background-color: ' . esc_attr($final_colors['apply_btn_color']) . ' !important;';
        $css .= '}';
    } elseif (!empty($final_colors['primary_color'])) {
        $css .= '.psfa-theme-' . esc_attr($selected_theme) . ' .apply-btn {';
        $css .= 'background-color: ' . esc_attr($final_colors['primary_color']) . ' !important;';
        $css .= '}';
    }
    if (!empty($final_colors['primary_color'])) {
        $css .= '.psfa-theme-' . esc_attr($selected_theme) . ' .apply-btn:hover {';
        $css .= 'background-color: ' . esc_attr($final_colors['primary_color']) . ' !important;';
        $css .= 'opacity: 0.9;';
        $css .= '}';
    }
    
    // Reset filters button - handle minimal theme special case (transparent background)
    if (!empty($final_colors['secondary_color'])) {
        $css .= '.psfa-theme-' . esc_attr($selected_theme) . ' .filter-actions .reset-btn {';
        // For minimal theme, secondary color is text color, not background
        if ($selected_theme === 'minimal') {
            $css .= 'color: ' . esc_attr($final_colors['secondary_color']) . ' !important;';
        } else {
            $css .= 'background-color: ' . esc_attr($final_colors['secondary_color']) . ' !important;';
        }
        $css .= '}';
    }
    
    // Primary color accent elements (spinner CSS is generated here, not duplicated above)
    if (!empty($final_colors['primary_color'])) {
        $css .= psfa_generate_primary_color_accent_css($selected_theme, $final_colors['primary_color']);
    }
    
    // Amazon icon color - use button_text color
    if (!empty($final_colors['button_text'])) {
        $css .= psfa_generate_icon_filter_css($selected_theme, $final_colors['button_text'], '.psfa-buy-button-icon');
    }
    
    return $css;
}

/**
 * Generate CSS for primary color accent elements
 *
 * @param string $selected_theme Selected theme slug
 * @param string $primary_color Primary color value
 * @return string Generated CSS
 */
function psfa_generate_primary_color_accent_css($selected_theme, $primary_color) {
    $css = '';
    
    // Category header border
    $css .= '.psfa-theme-' . esc_attr($selected_theme) . '.psfa-category-header, ';
    $css .= '.psfa-theme-' . esc_attr($selected_theme) . ' .psfa-category-header {';
    $css .= 'border-left-color: ' . esc_attr($primary_color) . ' !important;';
    $css .= '}';
    
    // Upgrade notice border
    $css .= '.psfa-theme-' . esc_attr($selected_theme) . '.psfa-upgrade-notice, ';
    $css .= '.psfa-theme-' . esc_attr($selected_theme) . ' .psfa-upgrade-notice {';
    $css .= 'border-left-color: ' . esc_attr($primary_color) . ' !important;';
    $css .= '}';
    
    // Spinner border-top (also handled in psfa_generate_theme_colors_css for all cases)
    // Keeping here as fallback for when called directly - use same high specificity
    // Handle both: loader with theme class directly, and loader inside theme wrapper
    $css .= '.psfa-theme-' . esc_attr($selected_theme) . ' .infinite-scroll-loader .spinner, ';
    $css .= '.psfa-theme-' . esc_attr($selected_theme) . ' #infinite-scroll-loader .spinner, ';
    $css .= '.psfa-theme-' . esc_attr($selected_theme) . ' div.infinite-scroll-loader .spinner, ';
    $css .= '.psfa-theme-' . esc_attr($selected_theme) . ' div#infinite-scroll-loader .spinner, ';
    // Also target loader with theme class directly
    $css .= '.infinite-scroll-loader.psfa-theme-' . esc_attr($selected_theme) . ' .spinner, ';
    $css .= '#infinite-scroll-loader.psfa-theme-' . esc_attr($selected_theme) . ' .spinner {';
    $css .= 'border-top-color: ' . esc_attr($primary_color) . ' !important;';
    $css .= 'border-top: 4px solid ' . esc_attr($primary_color) . ' !important;';
    $css .= '}';
    
    // Autocomplete highlighted text
    $css .= '.psfa-theme-' . esc_attr($selected_theme) . ' .psfa-autocomplete-item strong {';
    $css .= 'color: ' . esc_attr($primary_color) . ' !important;';
    $css .= '}';
    
    // Category links
    $css .= '.psfa-theme-' . esc_attr($selected_theme) . ' .psfa-search-form-categories a.psfa-active, ';
    $css .= '.psfa-theme-' . esc_attr($selected_theme) . ' .psfa-search-form-categories a.psfa-hover {'; 
    $css .= 'color: ' . esc_attr($primary_color) . ' !important;';
    $css .= 'border-color: ' . esc_attr($primary_color) . ' !important;';
    $css .= '}';
    
    // Dark theme product card hover border and box-shadow
    if ($selected_theme === 'dark') {
        // Convert hex to rgba for box-shadow
        $hex = ltrim($primary_color, '#');
        if (strlen($hex) === 3) {
            $hex = $hex[0] . $hex[0] . $hex[1] . $hex[1] . $hex[2] . $hex[2];
        }
        $r = hexdec(substr($hex, 0, 2));
        $g = hexdec(substr($hex, 2, 2));
        $b = hexdec(substr($hex, 4, 2));
        $rgba = "rgba({$r}, {$g}, {$b}, 0.3)";
        
        $css .= '.psfa-theme-dark .psfa-product:hover {'; 
        $css .= 'border-color: ' . esc_attr($primary_color) . ' !important;';
        $css .= 'box-shadow: 0 4px 12px ' . esc_attr($rgba) . ' !important;';
        $css .= '}';
    }
    
    return $css;
}

/**
 * Generate CSS filter for icon color
 *
 * @param string $selected_theme Selected theme slug
 * @param string $icon_color Icon color hex value
 * @param string $selector CSS selector for the icon
 * @return string Generated CSS
 */
function psfa_generate_icon_filter_css($selected_theme, $icon_color, $selector) {
    $css = '';
    $hex = ltrim($icon_color, '#');
    
    if (strlen($hex) === 3) {
        $hex = $hex[0] . $hex[0] . $hex[1] . $hex[1] . $hex[2] . $hex[2];
    }
    
    if (strlen($hex) === 6) {
        $r = hexdec(substr($hex, 0, 2));
        $g = hexdec(substr($hex, 2, 2));
        $b = hexdec(substr($hex, 4, 2));
        $brightness = ($r * 299 + $g * 587 + $b * 114) / 1000;
        
        // Check if color is grey (R, G, B are approximately equal)
        $is_grey = abs($r - $g) < 10 && abs($g - $b) < 10 && abs($r - $b) < 10;
        
        $css .= '.psfa-theme-' . esc_attr($selected_theme) . ' ' . $selector . ' {';
        
        if ($is_grey) {
            // For grey colors, use a simpler filter without sepia/saturation to avoid color shifts
            // brightness(0) makes it black, then invert to get the grey level
            $invert_percent = round(($brightness / 255) * 100);
            $css .= 'filter: brightness(0) saturate(100%) invert(' . $invert_percent . '%);';
        } else {
            // For colored icons, use the full filter chain
            $css .= 'filter: brightness(0) saturate(100%) invert(' . round(($brightness / 255) * 100) . '%) sepia(100%) saturate(' . round(($r + $g + $b) / 3) . '%) hue-rotate(0deg);';
        }
        
        $css .= '}';
    }
    
    return $css;
}

/**
 * Generate CSS for font customization
 *
 * @param array $font_settings Font settings
 * @return string Generated CSS
 */
function psfa_generate_font_css($font_settings) {
    $css = '';
    $fonts_to_load = array();
    
    if (!empty($font_settings['title_font'])) {
        $fonts_to_load[] = $font_settings['title_font'];
        $css .= '.psfa-title-font { font-family: "' . esc_attr($font_settings['title_font']) . '", sans-serif; }';
    }
    if (!empty($font_settings['description_font'])) {
        $fonts_to_load[] = $font_settings['description_font'];
        $css .= '.psfa-description-font { font-family: "' . esc_attr($font_settings['description_font']) . '", sans-serif; }';
    }
    if (!empty($font_settings['price_font'])) {
        $fonts_to_load[] = $font_settings['price_font'];
        $css .= '.psfa-price-font { font-family: "' . esc_attr($font_settings['price_font']) . '", sans-serif; }';
    }
    
    // Load Google Fonts
    if (!empty($fonts_to_load)) {
        $fonts_to_load = array_unique($fonts_to_load);
        $font_families = implode('|', array_map('urlencode', $fonts_to_load));
        wp_enqueue_style('psfa-google-fonts', 'https://fonts.googleapis.com/css?family=' . $font_families, array(), PSFA_VERSION);
    }
    
    return $css;
}

/**
 * Generate CSS for card layout customization
 *
 * @param string $card_layout Card layout type
 * @param bool $is_premium Whether user is premium
 * @return string Generated CSS
 */
function psfa_generate_card_layout_css($card_layout, $is_premium) {
    $css = '';
    
    if ($card_layout === 'default' || !$is_premium) {
        return $css;
    }
    
    switch ($card_layout) {
        case 'compact':
            $css .= psfa_generate_compact_layout_css();
            break;
        case 'detailed':
            $css .= psfa_generate_detailed_layout_css();
            break;
        case 'image_focused':
            $css .= psfa_generate_image_focused_layout_css();
            break;
    }
    
    return $css;
}

/**
 * Generate CSS for compact card layout
 *
 * @return string Generated CSS
 */
function psfa_generate_compact_layout_css() {
    return '
        .psfa-products.psfa-display-grid .psfa-product.psfa-card-layout-compact {
            padding: 12px !important;
            gap: 8px !important;
            max-width: 220px !important;
        }
        .psfa-products.psfa-display-grid .psfa-product.psfa-card-layout-compact h7 {
            font-size: 0.9rem !important;
            line-height: 1.3 !important;
            margin: 4px 0 !important;
            min-height: 2.6em !important;
        }
        .psfa-products.psfa-display-grid .psfa-product.psfa-card-layout-compact .psfa-brand {
            font-size: 0.8rem !important;
            margin: 2px 0 !important;
        }
        .psfa-products.psfa-display-grid .psfa-product.psfa-card-layout-compact .psfa-price {
            font-size: 0.95rem !important;
            margin: 4px 0 !important;
        }
        .psfa-products.psfa-display-grid .psfa-product.psfa-card-layout-compact .psfa-buy-button {
            padding: 6px 14px !important;
            font-size: 0.9rem !important;
            white-space: nowrap !important;
            overflow: visible !important;
            text-overflow: clip !important;
            width: auto !important;
            min-width: auto !important;
        }
        .psfa-products.psfa-display-grid .psfa-product.psfa-card-layout-compact .psfa-buy-button-icon {
            width: 14px !important;
            height: 14px !important;
        }
        .psfa-products.psfa-display-grid .psfa-product.psfa-card-layout-compact .psfa-image-column {
            flex: 0 0 110px !important;
        }
        .psfa-products.psfa-display-grid .psfa-product.psfa-card-layout-compact .psfa-image-column img {
            max-height: 110px !important;
            max-width: 110px !important;
        }
        .psfa-products.psfa-display-list .psfa-product.psfa-card-layout-compact {
            padding: 18px 20px !important;
        }
        .psfa-products.psfa-display-list .psfa-product.psfa-card-layout-compact .psfa-image-column {
            width: 12% !important;
        }
        .psfa-products.psfa-display-list .psfa-product.psfa-card-layout-compact .psfa-title-column {
            width: 63% !important;
            padding-left: 15px !important;
        }
        .psfa-products.psfa-display-list .psfa-product.psfa-card-layout-compact .psfa-purchase-column {
            width: 18% !important;
        }
        .psfa-products.psfa-display-list .psfa-product.psfa-card-layout-compact h7 {
            font-size: 0.95rem !important;
        }
        .psfa-products.psfa-display-list .psfa-product.psfa-card-layout-compact .psfa-price {
            font-size: 1.05rem !important;
        }
    ';
}

/**
 * Generate CSS for detailed card layout
 *
 * @return string Generated CSS
 */
function psfa_generate_detailed_layout_css() {
    return '
        .psfa-products.psfa-display-grid .psfa-product.psfa-card-layout-detailed {
            padding: 32px !important;
            gap: 20px !important;
            max-width: 300px !important;
        }
        .psfa-products.psfa-display-grid .psfa-product.psfa-card-layout-detailed h7 {
            font-size: 1.25rem !important;
            line-height: 1.6 !important;
            margin: 12px 0 !important;
            min-height: 3.2em !important;
        }
        .psfa-products.psfa-display-grid .psfa-product.psfa-card-layout-detailed .psfa-brand {
            font-size: 1rem !important;
            margin: 8px 0 !important;
        }
        .psfa-products.psfa-display-grid .psfa-product.psfa-card-layout-detailed .psfa-price {
            font-size: 1.4rem !important;
            font-weight: 700 !important;
            margin: 12px 0 !important;
        }
        .psfa-products.psfa-display-grid .psfa-product.psfa-card-layout-detailed .psfa-buy-button {
            padding: 14px 28px !important;
            font-size: 1.1rem !important;
            white-space: nowrap !important;
            overflow: visible !important;
            text-overflow: clip !important;
            width: auto !important;
            min-width: auto !important;
        }
        .psfa-products.psfa-display-grid .psfa-product.psfa-card-layout-detailed .psfa-buy-button-icon {
            width: 20px !important;
            height: 20px !important;
        }
        .psfa-products.psfa-display-grid .psfa-product.psfa-card-layout-detailed .psfa-image-column {
            flex: 0 0 220px !important;
        }
        .psfa-products.psfa-display-grid .psfa-product.psfa-card-layout-detailed .psfa-image-column img {
            max-height: 220px !important;
            max-width: 220px !important;
        }
        .psfa-products.psfa-display-list .psfa-product.psfa-card-layout-detailed {
            padding: 26px 24px !important;
        }
        .psfa-products.psfa-display-list .psfa-product.psfa-card-layout-detailed .psfa-image-column {
            width: 20% !important;
        }
        .psfa-products.psfa-display-list .psfa-product.psfa-card-layout-detailed .psfa-title-column {
            width: 55% !important;
        }
        .psfa-products.psfa-display-list .psfa-product.psfa-card-layout-detailed .psfa-purchase-column {
            width: 20% !important;
        }
        .psfa-products.psfa-display-list .psfa-product.psfa-card-layout-detailed h7 {
            font-size: 1.05rem !important;
        }
        .psfa-products.psfa-display-list .psfa-product.psfa-card-layout-detailed .psfa-price {
            font-size: 1.2rem !important;
            font-weight: 600 !important;
        }
    ';
}

/**
 * Generate CSS for image-focused card layout
 *
 * @return string Generated CSS
 */
function psfa_generate_image_focused_layout_css() {
    return '
        .psfa-product.psfa-card-layout-image_focused {
            flex-direction: row !important;
            max-width: 100% !important;
            align-items: flex-start !important;
        }
        .psfa-product.psfa-card-layout-image_focused .psfa-image-column {
            flex: 0 0 50% !important;
            min-width: 250px !important;
            max-width: 350px !important;
        }
        .psfa-product.psfa-card-layout-image_focused .psfa-image-column img {
            width: 100% !important;
            height: auto !important;
            max-height: 400px !important;
            object-fit: contain !important;
            border-radius: 8px !important;
        }
        .psfa-products.psfa-display-list .psfa-product.psfa-card-layout-image_focused .psfa-image-column {
            width: 30% !important;
            flex: 0 0 30% !important;
            max-width: 380px !important;
        }
        .psfa-products.psfa-display-list .psfa-product.psfa-card-layout-image_focused .psfa-image-column img:not(.psfa-badge-prime-img) {
            width: 100% !important;
            max-width: none !important;
            max-height: 360px !important;
        }
        .psfa-product.psfa-card-layout-image_focused .psfa-title-column {
            flex: 1 !important;
            padding-left: 24px !important;
            padding-right: 12px !important;
        }
        .psfa-product.psfa-card-layout-image_focused .psfa-title-column h7 {
            font-size: 1.2rem !important;
            margin-bottom: 16px !important;
        }
        .psfa-product.psfa-card-layout-image_focused .psfa-purchase-column {
            text-align: center !important;
            display: flex !important;
            flex-direction: column !important;
            align-items: center !important;
            justify-content: center !important;
            /* Ensure the buy button has breathing room from card borders */
            padding: 0 16px !important;
        }
        .psfa-product.psfa-card-layout-image_focused .psfa-brand {
            width: 100% !important;
            margin-top: 16px !important;
        }
        .psfa-product.psfa-card-layout-image_focused .psfa-price {
            width: 100% !important;
        }
        .psfa-product.psfa-card-layout-image_focused .psfa-buy-button {
            width: 100% !important;
            min-width: 100% !important;
            max-width: 100% !important;
        }
    ';
}

