<?php
/**
 * View helper functions for search template
 *
 * Contains helper functions for premium checks, category availability,
 * and other view-related utilities.
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Check if premium (Freemius)
 *
 * @return bool
 */
function psfa_is_premium() {
    $is_premium = false;
    if (function_exists('psfa_fs')) {
        $is_premium = psfa_fs()->is_plan('personal');
    }
    /**
     * Filter to override premium status (for demo mode, testing, etc.)
     *
     * @param bool $is_premium Current premium status
     * @return bool Modified premium status
     */
    return apply_filters('psfa_is_premium', $is_premium);
}

/**
 * Get max pages based on license
 *
 * @return int
 */
function psfa_get_max_pages() {
    
    return 5; // Free version limited to 5 pages
}

/**
 * Get available categories based on license
 *
 * @return array
 */
function psfa_get_available_categories() {
    $all_categories = array(
        'Electronics',
        'Books',
        'Fashion',
        'ToysAndGames',
        'HomeGarden',
        'TodaysDeals',
        'Automotive',
        'Baby',
        'Beauty',
        'Computers',
        'Grocery',
        'HealthPersonalCare',
        'HomeImprovement',
        'Industrial',
        'KindleStore',
        'Luggage',
        'MoviesTV',
        'Music',
        'MusicalInstruments',
        'OfficeProducts',
        'PetSupplies',
        'Software',
        'SportsOutdoors',
        'ToolsHomeImprovement',
        'VideoGames',
        'Watches'
    );
    
    
    // Free version limited to 5 categories
    return array(
        'Electronics',
        'Books',
        'Fashion',
        'ToysAndGames',
        'HomeGarden'
    );
}

/**
 * Check if category is available
 *
 * @param string $category
 * @return bool
 */
function psfa_is_category_available($category) {
    $available = psfa_get_available_categories();
    return in_array($category, $available, true);
}

/**
 * Map category slugs to proper category names
 *
 * @return array
 */
function psfa_get_category_map() {
    return array(
        'fashion' => 'Fashion',
        'electronics' => 'Electronics',
        'books' => 'Books',
        'toysandgames' => 'ToysAndGames',
        'toys' => 'ToysAndGames',
        'homegarden' => 'HomeGarden',
        'home' => 'HomeGarden',
        'todaysdeals' => 'TodaysDeals',
        'deals' => 'TodaysDeals'
    );
}

/**
 * Normalize category from slug or name
 *
 * @param string $raw_category
 * @return string
 */
function psfa_normalize_category($raw_category) {
    $category_map = psfa_get_category_map();
    $normalized = strtolower(trim($raw_category));
    
    if (isset($category_map[$normalized])) {
        return $category_map[$normalized];
    }
    
    return ucfirst($normalized);
}

/**
 * Check if branding should be shown
 *
 * Free version always shows branding. Premium users can enable white-label to hide it.
 *
 * @return bool
 */
function psfa_show_branding() {
    
    return true; // Free version always shows branding
}

/**
 * Render category upgrade notice as a yellow call-to-action
 *
 * @param string $upgrade_url Upgrade URL
 * @return string HTML for upgrade notice
 */
function psfa_render_category_upgrade_notice($upgrade_url) {
    $upgrade_url = esc_url($upgrade_url);
    return '<div class="psfa-category-upgrade-notice" style="margin: 20px 0; padding: 20px; background: #fff3cd; border-left: 4px solid #ffc107; border-radius: 4px; box-shadow: 0 2px 4px rgba(0,0,0,0.1);">
        <div style="display: flex; align-items: center; justify-content: space-between; flex-wrap: wrap; gap: 15px;">
            <div style="flex: 1; min-width: 200px;">
                <strong style="display: block; font-size: 16px; color: #856404; margin-bottom: 8px;">🔒 Premium Category</strong>
                <p style="margin: 0; color: #856404; font-size: 14px;">This category is available in Pro/Premium plans. Upgrade to unlock access to all 26 Amazon categories!</p>
            </div>
            <div>
                <a href="' . $upgrade_url . '" target="_blank" class="button button-primary" style="background: #2271b1; border-color: #2271b1; color: #fff; padding: 10px 20px; font-size: 14px; font-weight: 600; text-decoration: none; border-radius: 4px; display: inline-block; transition: background 0.2s;">
                    Upgrade to Premium
                </a>
            </div>
        </div>
    </div>';
}

