<?php
/**
 * Search parameter parsing and normalization
 *
 * Centralizes reading and sanitizing search parameters from $_GET, $_POST,
 * or override arrays.
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Centralized search parameter parsing (DRY helper).
 *
 * NOTE: This is intentionally conservative for now and focuses on the
 * parameters used by the AJAX handler (GET-based). The $override array
 * allows tests or internal callers to supply explicit values without
 * depending on $_GET directly.
 *
 * @param array $override Optional explicit values that take precedence.
 * @return array {
 *   @type string query
 *   @type string category
 *   @type string brand
 *   @type float  min_price
 *   @type float  max_price
 *   @type string sort
 *   @type string condition
 *   @type string merchant
 *   @type int    page
 * }
 */
function psfa_get_search_params(array $override = array()) {
    $params = array(
        'query'              => '',
        'category'           => '',
        'brand'              => '',
        'min_price'          => 0.0,
        'max_price'          => 0.0,
        'sort'               => '',
        'condition'          => '',
        'merchant'           => '',
        'page'               => 1,
        'has_deal'           => '',      // Premium: Has Deal filter
        'min_rating'         => 0,       // Premium: Minimum customer rating (1-5)
        'min_saving_percent' => 0,       // Premium: Minimum discount percentage
        'availability'       => '',      // Stock status filter (Available, IncludeOutOfStock, OutOfStockOnly)
    );

    // Check if POST data is from a legitimate form submission
    $post_nonce_valid = false;
    if (isset($_POST['psfa_nonce'])) {
        $post_nonce_valid = wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['psfa_nonce'])), 'psfa_search_nonce');
    }

    // Helper to read from override first, then from $_GET, then from $_POST (if nonce valid).
    $read_text = static function($key, $request_key = null) use ($override, $post_nonce_valid) {
        if (array_key_exists($key, $override)) {
            return (string) $override[$key];
        }
        $request_key = $request_key !== null ? $request_key : $key;
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- GET parameters are public URL parameters
        if (isset($_GET[$request_key])) {
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- GET parameters are public URL parameters, wp_unslash and sanitize_text_field handle sanitization
            return sanitize_text_field(wp_unslash($_GET[$request_key]));
        }
        // Nonce already verified above, POST data is safe
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified above via $post_nonce_valid
        if ($post_nonce_valid && isset($_POST[$request_key])) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Nonce verified above via $post_nonce_valid, wp_unslash and sanitize_text_field handle sanitization
            return sanitize_text_field(wp_unslash($_POST[$request_key]));
        }
        return '';
    };

    $read_float = static function($key, $request_key = null) use ($override, $post_nonce_valid) {
        if (array_key_exists($key, $override)) {
            return (float) $override[$key];
        }
        $request_key = $request_key !== null ? $request_key : $key;
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- GET parameters are public URL parameters
        if (isset($_GET[$request_key]) && $_GET[$request_key] !== '') {
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash -- GET parameters are public URL parameters, wp_unslash and float cast handle sanitization
            return (float) wp_unslash($_GET[$request_key]);
        }
        // Nonce already verified above, POST data is safe
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified above via $post_nonce_valid
        if ($post_nonce_valid && isset($_POST[$request_key]) && $_POST[$request_key] !== '') {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash -- Nonce verified above via $post_nonce_valid, wp_unslash and float cast handle sanitization
            return (float) wp_unslash($_POST[$request_key]);
        }
        return 0.0;
    };

    $read_int = static function($key, $request_key = null, $default = 1) use ($override, $post_nonce_valid) {
        if (array_key_exists($key, $override)) {
            return (int) $override[$key];
        }
        $request_key = $request_key !== null ? $request_key : $key;
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- GET parameters are public URL parameters
        if (isset($_GET[$request_key]) && $_GET[$request_key] !== '') {
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash -- GET parameters are public URL parameters, wp_unslash and integer cast handle sanitization
            return (int) wp_unslash($_GET[$request_key]);
        }
        // Nonce already verified above, POST data is safe
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified above via $post_nonce_valid
        if ($post_nonce_valid && isset($_POST[$request_key]) && $_POST[$request_key] !== '') {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash -- Nonce verified above via $post_nonce_valid, wp_unslash and integer cast handle sanitization
            return (int) wp_unslash($_POST[$request_key]);
        }
        return $default;
    };

    $params['query']     = $read_text('query');
    $params['category']  = $read_text('category');
    $params['brand']     = $read_text('brand');
    $params['min_price'] = $read_float('min_price', 'price_min');
    $params['max_price'] = $read_float('max_price', 'price_max');

    // Sort may be URL-encoded in the query string (e.g. Price%3ALowToHigh).
    $raw_sort = $read_text('sort');
    if ($raw_sort !== '') {
        $params['sort'] = urldecode($raw_sort);
    }

    $params['condition'] = $read_text('condition');
    $params['merchant']  = $read_text('merchant');
    $params['page']      = $read_int('page');
    
    // Premium filters (default to 0, not 1)
    $params['has_deal']           = $read_text('has_deal');
    $params['min_rating']         = $read_int('min_rating', null, 0);
    $params['min_saving_percent'] = $read_int('min_saving_percent', null, 0);
    
    // Stock status filter
    $params['availability']       = $read_text('availability');

    // Aliases for compatibility with existing template / URLs.
    // Support ?q=... in addition to ?query=...
    if ($params['query'] === '') {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- GET parameters are public URL parameters
        if (isset($_GET['q'])) {
            $params['query'] = sanitize_text_field(wp_unslash($_GET['q']));
        // Nonce already verified above, POST data is safe
        } elseif ($post_nonce_valid && isset($_POST['q'])) {
            $params['query'] = sanitize_text_field(wp_unslash($_POST['q']));
        }
    }

    // Support ?min_price / ?max_price in addition to price_min / price_max.
    if ($params['min_price'] == 0.0) {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- GET parameters are public URL parameters
        if (isset($_GET['min_price']) && $_GET['min_price'] !== '') {
            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash -- wp_unslash and float cast handle sanitization
            $params['min_price'] = (float) wp_unslash($_GET['min_price']);
        // Nonce already verified above, POST data is safe
        } elseif ($post_nonce_valid && isset($_POST['min_price']) && $_POST['min_price'] !== '') {
            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash -- wp_unslash and float cast handle sanitization
            $params['min_price'] = (float) wp_unslash($_POST['min_price']);
        }
    }

    if ($params['max_price'] == 0.0) {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- GET parameters are public URL parameters
        if (isset($_GET['max_price']) && $_GET['max_price'] !== '') {
            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash -- wp_unslash and float cast handle sanitization
            $params['max_price'] = (float) wp_unslash($_GET['max_price']);
        // Nonce already verified above, POST data is safe
        } elseif ($post_nonce_valid && isset($_POST['max_price']) && $_POST['max_price'] !== '') {
            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash -- wp_unslash and float cast handle sanitization
            $params['max_price'] = (float) wp_unslash($_POST['max_price']);
        }
    }

    return $params;
}

