<?php
/**
 * Search Template for PaapiPlugin Product Search for Amazon
 * This file contains the HTML/UI for the search interface
 */

if (!defined('ABSPATH')) {
    exit;
}

// Include handler and view helpers
require_once PSFA_PLUGIN_DIR . 'core/frontend/handler.php';
require_once PSFA_PLUGIN_DIR . 'core/search/view-helpers.php';
require_once PSFA_PLUGIN_DIR . 'core/shared/theme-manager.php';
require_once PSFA_PLUGIN_DIR . 'core/shared/customization-helpers.php';
require_once PSFA_PLUGIN_DIR . 'core/api/class-offer-data-extractor.php';

// Get shortcode attributes
global $psfa_shortcode_atts;
$psfa_shortcode_atts = isset($psfa_shortcode_atts) ? $psfa_shortcode_atts : array(
    'category'     => '',
    'query'        => '',
    'brand'        => '',
    'sort'         => '',
    'hide_search'  => false,
    'hide_filters' => false,
    'pagination'   => 'infinite',
);

// Process search request and get structured data
// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound -- Local variable, not global
$search_data = psfa_handle_search_request();

// Extract variables for template
$psfa_error_message = $search_data['error_message'];
$psfa_results = $search_data['results'];
$psfa_filtered_results = $search_data['filtered_results'];
$psfa_filter_options = $search_data['filter_options'];
$psfa_search_params = $search_data['search_params'];
$psfa_total_unfiltered_count = $search_data['total_unfiltered_count'];
$psfa_pages_fetched = $search_data['pages_fetched'];
$psfa_initial_category = $search_data['initial_category'];

// Debug mode: add ?psfa_debug=1 to URL to see raw API payload in console
// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only debug flag
$psfa_debug_mode = isset($_GET['psfa_debug']) && $_GET['psfa_debug'] === '1';
if ($psfa_debug_mode) {
    // Auth debug info
    $psfa_auth_type = get_option('psfa_auth_type', 'awssig4');
    $psfa_debug_auth = [
        'auth_type' => $psfa_auth_type,
        'auth_type_label' => $psfa_auth_type === 'oauth' ? 'OAuth 2.0 (Creators API)' : 'AWS Signature V4 (PA-API)',
    ];
    if ($psfa_auth_type === 'oauth') {
        $psfa_debug_auth['oauth_client_id'] = get_option('psfa_oauth_client_id') ? '✓ Set' : '✗ Missing';
        $psfa_debug_auth['oauth_client_secret'] = get_option('psfa_oauth_client_secret') ? '✓ Set' : '✗ Missing';
        $psfa_debug_auth['oauth_credential_version'] = get_option('psfa_oauth_credential_version', '(not set)');
        $psfa_debug_auth['oauth_auth_region'] = get_option('psfa_oauth_auth_region', 'us-east-1');
        $psfa_debug_auth['marketplace'] = get_option('psfa_marketplace', 'www.amazon.com');
        $psfa_debug_auth['oauth_token_cached'] = get_option('psfa_oauth_access_token') ? '✓ Yes' : '✗ No';
        $psfa_debug_auth['oauth_token_expires'] = get_option('psfa_oauth_token_expires', 0);
    } else {
        $psfa_debug_auth['access_key_id'] = get_option('psfa_access_key_id') ? '✓ Set' : '✗ Missing';
        $psfa_debug_auth['secret_access_key'] = get_option('psfa_secret_access_key') ? '✓ Set' : '✗ Missing';
    }
    $psfa_debug_auth['associate_tag'] = get_option('psfa_associate_tag', '(not set)');
    echo '<script>console.log("PSFA Debug - Auth:", ' . wp_json_encode($psfa_debug_auth) . ');</script>';
    
    if ($psfa_results) {
        echo '<script>console.log("PSFA Debug - Raw API Response:", ' . wp_json_encode($psfa_results) . ');</script>';
        echo '<script>console.log("PSFA Debug - Filtered Results:", ' . wp_json_encode($psfa_filtered_results) . ');</script>';
        echo '<script>console.log("PSFA Debug - Features:", ' . wp_json_encode(psfa_get_features()) . ');</script>';
    }
}

// Get URL parameters for display (preserved values for form inputs)
$psfa_request_params = psfa_get_search_params();
$psfa_url_query = $psfa_request_params['query'];
$psfa_url_sort = $psfa_request_params['sort'];

// Use shortcode values as defaults if URL params are empty
if (empty($psfa_url_query) && !empty($psfa_shortcode_atts['query'])) {
    $psfa_url_query = $psfa_shortcode_atts['query'];
}
if (empty($psfa_request_params['brand']) && !empty($psfa_shortcode_atts['brand'])) {
    $psfa_request_params['brand'] = $psfa_shortcode_atts['brand'];
}
if (empty($psfa_url_sort) && !empty($psfa_shortcode_atts['sort'])) {
    $psfa_url_sort = $psfa_shortcode_atts['sort'];
}

// Get display options from shortcode or global settings (URL params can override for demo/preview)
// Priority: URL param > shortcode attr > global option
// Shortcode can explicitly override admin settings with "true" or "false"

// phpcs:disable WordPress.Security.NonceVerification.Recommended -- Public URL parameters for demo preview
if (isset($_GET['hide_search'])) {
    $psfa_hide_search = in_array($_GET['hide_search'], array('true', '1', 'yes'), true);
} elseif (isset($psfa_shortcode_atts['hide_search']) && $psfa_shortcode_atts['hide_search'] !== '') {
    $psfa_hide_search = in_array($psfa_shortcode_atts['hide_search'], array('true', '1', 'yes', true, 1), true);
} else {
    $psfa_hide_search = in_array(get_option('psfa_hide_search', '0'), array('1', 1, true, 'true'), true);
}
if (isset($_GET['hide_filters'])) {
    $psfa_hide_filters = in_array($_GET['hide_filters'], array('true', '1', 'yes'), true);
} elseif (isset($psfa_shortcode_atts['hide_filters']) && $psfa_shortcode_atts['hide_filters'] !== '') {
    $psfa_hide_filters = in_array($psfa_shortcode_atts['hide_filters'], array('true', '1', 'yes', true, 1), true);
} else {
    $psfa_hide_filters = in_array(get_option('psfa_hide_filters', '0'), array('1', 1, true, 'true'), true);
}
$psfa_pagination_type = isset($_GET['pagination']) ? sanitize_text_field( wp_unslash( $_GET['pagination'] ) ) : (isset($psfa_shortcode_atts['pagination']) ? $psfa_shortcode_atts['pagination'] : 'infinite');
// Validate pagination type
if (!in_array($psfa_pagination_type, array('infinite', 'button', 'none'), true)) {
    $psfa_pagination_type = 'infinite';
}
// phpcs:enable WordPress.Security.NonceVerification.Recommended

// Use filtered results if available
$psfa_display_results = $psfa_filtered_results ?? $psfa_results;

// Ensure filter_options is initialized
if ($psfa_filter_options === null) {
    $psfa_filter_options = ['brands' => [], 'price_ranges' => [], 'min_price' => 0, 'max_price' => 0, 'conditions' => []];
}

// Prepare preserved values for form inputs
$psfa_preserved_query = $psfa_url_query;
$psfa_preserved_category = $psfa_initial_category;
$psfa_preserved_sort = $psfa_url_sort;
$psfa_preserved_brand = $psfa_request_params['brand'];
$psfa_preserved_condition = $psfa_request_params['condition'];
$psfa_preserved_merchant = $psfa_request_params['merchant'];
$psfa_preserved_availability = $psfa_request_params['availability'];
$psfa_preserved_min_price = $psfa_request_params['min_price'];
$psfa_preserved_max_price = $psfa_request_params['max_price'];

// Determine if any filters are currently active
$psfa_has_active_filters = (
    ($psfa_preserved_min_price ?? 0) > 0 ||
    ($psfa_preserved_max_price ?? 0) > 0 ||
    !empty($psfa_preserved_brand) ||
    !empty($psfa_preserved_sort) ||
    !empty($psfa_preserved_condition) ||
    !empty($psfa_preserved_merchant) ||
    (!empty($psfa_preserved_availability) && $psfa_preserved_availability !== 'Available') ||
    !empty($psfa_request_params['has_deal'] ?? '') ||
        ($psfa_request_params['min_saving_percent'] ?? 0) > 0
);

// Show filters if we have search context (query or category) - even if no results, so user can adjust filters
$psfa_has_search_context = !empty($psfa_preserved_query) || !empty($psfa_preserved_category);
$psfa_should_show_filters = $psfa_has_search_context && ($psfa_filter_options || (isset($psfa_results['SearchResult']['Items']) && count($psfa_results['SearchResult']['Items']) >= 0));

// Ensure search params are always set for pagination data attributes
if ($psfa_search_params === null) {
    $psfa_search_params = [
        'query' => $psfa_url_query ?? '',
        'category' => $psfa_initial_category ?? '',
        'price_min' => 0,
        'price_max' => 0,
        'brand' => '',
        'sort' => $psfa_url_sort ?? '',
        'condition' => '',
        'merchant' => '',
        'availability' => !empty($psfa_request_params['availability']) ? $psfa_request_params['availability'] : 'Available',
    ];
} else {
    // Always ensure query is set from URL if available, even if $psfa_search_params exists
    if (empty($psfa_search_params['query']) && !empty($psfa_url_query)) {
        $psfa_search_params['query'] = $psfa_url_query;
    }
}
?>

<!-- Search Form -->
<?php
// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound -- Local variable, not global
$selected_theme = psfa_get_selected_theme();
// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound -- Local variable, not global
$theme_class = 'psfa-theme-' . esc_attr($selected_theme);

// Compute Reset Filters URL: link back to the search page without filters,
// preserving the current query (?q=...) if present.
// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound -- Local variable, not global
$psfa_reset_url = get_permalink();
if (!empty($psfa_preserved_query) && function_exists('add_query_arg')) {
    $psfa_reset_url = add_query_arg(
        array(
            'q' => $psfa_preserved_query,
        ),
        $psfa_reset_url
    );
}

// Get filter visibility early - needed for both search toggle button and filters section
$psfa_filter_visibility = get_option('psfa_filter_visibility', array(
    'brand' => true, 'price' => true, 'sort' => true, 'condition' => true,
    'merchant' => true, 'availability' => true, 'has_deal' => true, 'deals' => true,
));
$psfa_any_filter_visible = !empty($psfa_filter_visibility['brand']) 
    || !empty($psfa_filter_visibility['price']) 
    || !empty($psfa_filter_visibility['sort']) 
    || !empty($psfa_filter_visibility['condition']) 
    || !empty($psfa_filter_visibility['merchant']) 
    || !empty($psfa_filter_visibility['availability']) 
    || !empty($psfa_filter_visibility['has_deal'])
    || !empty($psfa_filter_visibility['deals']);
?>
<?php if (!$psfa_hide_search): ?>
<div class="<?php echo esc_attr( $theme_class ); ?>">
<form class="psfa-search-form" method="POST" action="">
    <?php wp_nonce_field('psfa_search_nonce', 'psfa_nonce'); ?>
    <div style="position: relative; width: 400px; max-width: 100%;">
        <input type="text" name="amazon_search_query" id="amazon-search-input" placeholder="Search Amazon products..." 
            style="width: -webkit-fill-available; padding:8px 10px; outline:none; border: 1px solid #ddd; border-radius: 4px;"
            value="<?php echo esc_attr($psfa_preserved_query); ?>" autocomplete="off">
        <div id="autocomplete-suggestions"></div>
    </div>
        
    <button type="submit" class="psfa-search-btn">
        Search
    </button>
    <button type="button" id="clear-search-btn" class="psfa-reset-btn">
        Reset
    </button>
    <?php if ($psfa_should_show_filters && !$psfa_hide_filters && $psfa_any_filter_visible): ?>
    <button type="button" id="toggle-filters-btn" class="psfa-filters-toggle" aria-expanded="false">
        <svg class="psfa-filters-icon" width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
            <path d="M4 5L10 5M10 5C10 6.10457 10.8954 7 12 7C13.1046 7 14 6.10457 14 5M10 5C10 3.89543 10.8954 3 12 3C13.1046 3 14 3.89543 14 5M14 5L20 5M4 12L16 12M16 12C16 13.1046 16.8954 14 18 14C19.1046 14 20 13.1046 20 12C20 10.8954 19.1046 10 18 10C16.8954 10 16 10.8954 16 12ZM8 19L20 19M8 19C8 17.8954 7.10457 17 6 17C4.89543 17 4 17.8954 4 19C4 20.1046 4.89543 21 6 21C7.10457 21 8 20.1046 8 19Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"/>
        </svg>
    </button>
    <?php endif; ?>
</form>
</div>
<?php endif; ?>

<!-- Filters -->
<?php if ($psfa_should_show_filters && !$psfa_hide_filters && $psfa_any_filter_visible): ?>
<?php 
// Always show filters expanded when search is hidden
$psfa_filters_always_visible = $psfa_hide_search;
$psfa_filters_expanded = $psfa_filters_always_visible || $psfa_has_active_filters;
?>
<div class="psfa-filters <?php echo esc_attr( $theme_class ); ?><?php echo $psfa_filters_always_visible ? '' : ' is-collapsed'; ?>" data-initial-expanded="<?php echo $psfa_filters_expanded ? '1' : '0'; ?>">
    <form method="POST" action="">
        <?php wp_nonce_field('psfa_search_nonce', 'psfa_nonce'); ?>
        <input type="hidden" name="amazon_search_query" value="<?php echo esc_attr($psfa_preserved_query); ?>">
        <input type="hidden" name="amazon_search_category" value="<?php echo esc_attr($psfa_preserved_category); ?>">
        
        <?php if (!empty($psfa_filter_visibility['brand']) && !empty($psfa_filter_options['brands'])): ?>
        <div class="psfa-filter-group">
            <select name="amazon_search_brand">
                <option value="">All Brands</option>
                <?php foreach ($psfa_filter_options['brands'] as $psfa_brand): ?>
                    <option value="<?php echo esc_attr($psfa_brand); ?>" 
                        <?php selected(strtolower(trim($psfa_preserved_brand ?? '')), strtolower(trim($psfa_brand))); ?>>
                        <?php echo esc_html($psfa_brand); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>
        <?php endif; ?>
        
        <?php if (!empty($psfa_filter_visibility['price'])): ?>
        <div class="psfa-filter-group price-filter-group">
            <div class="price-inputs">
                <input type="number" name="amazon_search_price_min" placeholder="Min $" 
                    min="0" step="0.01"
                    value="<?php echo esc_attr($psfa_preserved_min_price > 0 ? $psfa_preserved_min_price : ''); ?>">
                <input type="number" name="amazon_search_price_max" placeholder="Max $" 
                    min="0" step="0.01"
                    value="<?php echo esc_attr($psfa_preserved_max_price > 0 ? $psfa_preserved_max_price : ''); ?>">
            </div>
        </div>
        <?php endif; ?>
        
        <?php if (!empty($psfa_filter_visibility['sort'])): ?>
        <div class="psfa-filter-group">
            <select name="amazon_search_sort" id="sort-select" class="psfa-input">
                <option value="">Relevance (Default)</option>
                <option value="Price:LowToHigh" <?php selected($psfa_preserved_sort, 'Price:LowToHigh'); ?>>Price: Low to High</option>
                <option value="Price:HighToLow" <?php selected($psfa_preserved_sort, 'Price:HighToLow'); ?>>Price: High to Low</option>
                <option value="NewestArrivals" <?php selected($psfa_preserved_sort, 'NewestArrivals'); ?>>Newest Arrivals</option>
                <option value="AvgCustomerReviews" <?php selected($psfa_preserved_sort, 'AvgCustomerReviews'); ?>>Highest Rated</option>
                <option value="Featured" <?php selected($psfa_preserved_sort, 'Featured'); ?>>Featured</option>
            </select>
        </div>
        <?php endif; ?>
        
        <?php
        // Use dynamic condition options from SearchRefinements when available.
        $psfa_available_conditions = $psfa_filter_options['conditions'] ?? array();
        $psfa_has_condition_refinements = ! empty($psfa_available_conditions);
        // Full set of supported condition values. We'll filter this list using
        // SearchRefinements in live mode so that options like "Collectible"
        // only appear when Amazon actually offers them for the current query.
        $psfa_condition_values = array(
            'New'         => 'New',
            'Used'        => 'Used',
            'Refurbished' => 'Refurbished',
            'Collectible' => 'Collectible',
        );
        ?>
        <?php if (!empty($psfa_filter_visibility['condition'])): ?>
        <div class="psfa-filter-group">
            <select name="amazon_search_condition" id="condition-select" class="psfa-input">
                <option value="">All Conditions</option>
                <?php foreach ($psfa_condition_values as $psfa_condition_value => $psfa_condition_label): ?>
                    <?php
                    // When we have refinements from PA-API (live mode), only show
                    // condition values that Amazon reports as available.
                    if ($psfa_has_condition_refinements && ! in_array($psfa_condition_value, $psfa_available_conditions, true)) {
                        continue;
                    }
                    ?>
                    <option value="<?php echo esc_attr($psfa_condition_value); ?>" <?php selected($psfa_preserved_condition ?? '', $psfa_condition_value); ?>>
                        <?php echo esc_html($psfa_condition_label); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>
        <?php endif; ?>
        
        <?php if (!empty($psfa_filter_visibility['merchant'])): ?>
        <div class="psfa-filter-group">
            <select name="amazon_search_merchant" id="merchant-select" class="psfa-input">
                <option value="">All Merchants</option>
                <option value="Amazon" <?php selected($psfa_preserved_merchant ?? '', 'Amazon'); ?>>Sold by Amazon</option>
            </select>
        </div>
        <?php endif; ?>
        
        <?php if (!empty($psfa_filter_visibility['availability'])): ?>
        <div class="psfa-filter-group">
            <select name="amazon_search_availability" id="availability-select" class="psfa-input">
                <option value="Available" <?php selected(($psfa_preserved_availability ?? '') === '' || ($psfa_preserved_availability ?? '') === 'Available', true); ?>>In Stock Only</option>
                <option value="IncludeOutOfStock" <?php selected($psfa_preserved_availability ?? '', 'IncludeOutOfStock'); ?>>All Items</option>
                <option value="OutOfStockOnly" <?php selected($psfa_preserved_availability ?? '', 'OutOfStockOnly'); ?>>Out of Stock Only</option>
            </select>
        </div>
        <?php endif; ?>
        
        <?php  ?>
        
        <div class="filter-actions">
            <button type="submit" class="apply-btn">Apply Filters</button>
            <button
                type="button"
                class="reset-btn"
                onclick="window.location.href='<?php echo esc_js( $psfa_reset_url ); ?>';">
                Reset Filters
            </button>
        </div>
        
    </form>
</div>
<?php endif; // $psfa_should_show_filters && !$psfa_hide_filters && $psfa_any_filter_visible ?>

<!-- Error Message -->
<?php if (!empty($psfa_error_message)): ?>
    <?php 
    // Check if it's an upgrade notice (contains upgrade notice class)
    if (strpos($psfa_error_message, 'psfa-category-upgrade-notice') !== false): 
        echo wp_kses_post( $psfa_error_message );
    else: ?>
        <div class="psfa-error">
            <strong>Error:</strong> <?php echo esc_html($psfa_error_message); ?>
        </div>
    <?php endif; ?>
<?php endif; ?>

<!-- Results -->
<?php 
// Calculate has_more for data attribute
$psfa_total_count = $psfa_results['SearchResult']['TotalResultCount'] ?? 0;
$psfa_unfiltered_count = count($psfa_results['SearchResult']['Items'] ?? []);
$psfa_has_more = false;
$psfa_max_pages = psfa_get_max_pages();
$psfa_current_page = isset($psfa_pages_fetched) ? $psfa_pages_fetched : 1;

// Check license limit first
if ($psfa_current_page >= $psfa_max_pages) {
    $psfa_has_more = false;
} elseif ($psfa_unfiltered_count === 0) {
    // If API returned 0 items, we're definitely on the last page
    $psfa_has_more = false;
} elseif ($psfa_unfiltered_count >= 10) {
    // Got a full page from API, likely more pages available
    $psfa_has_more = true;
} elseif ($psfa_total_count > 0) {
    // Use TotalResultCount to determine if more pages exist
    $psfa_total_pages = ceil($psfa_total_count / 10);
    $psfa_has_more = $psfa_current_page < $psfa_total_pages;
}
?>
<?php if (isset($psfa_display_results) && isset($psfa_display_results['SearchResult']['Items']) && count($psfa_display_results['SearchResult']['Items']) > 0): ?>
    <?php 
    // Hide category banner: URL param > shortcode > admin setting
    // phpcs:disable WordPress.Security.NonceVerification.Recommended -- Public URL parameter for demo preview
    if (isset($_GET['hide_category_banner'])) {
        $psfa_hide_banner = in_array($_GET['hide_category_banner'], array('true', '1', 'yes'), true);
        // phpcs:enable WordPress.Security.NonceVerification.Recommended
    } elseif (isset($psfa_shortcode_atts['hide_category_banner']) && $psfa_shortcode_atts['hide_category_banner'] !== '') {
        $psfa_hide_banner = in_array($psfa_shortcode_atts['hide_category_banner'], array('true', '1', 'yes', true, 1), true);
    } else {
        $psfa_hide_banner = in_array(get_option('psfa_hide_category_banner', '0'), array('1', 1, true, 'true'), true);
    }
    if (!empty($psfa_initial_category) && empty($psfa_search_params['query']) && !$psfa_hide_banner): 
    ?>
        <div class="psfa-category-header <?php echo esc_attr( $theme_class ); ?>" style="margin: 20px 0; padding: 15px; background: #f0f8ff; border-left: 4px solid; border-radius: 4px;">
            <h2 style="margin: 0 0 10px 0; color: #333;">Top Products in <?php echo esc_html(ucfirst($psfa_initial_category)); ?></h2>
            <p style="margin: 0; color: #666;">Showing popular products from this category</p>
        </div>
    <?php endif; ?>
    
    <?php
    // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound -- Local variable, not global
    $selected_theme = psfa_get_selected_theme();
    // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound -- Local variable, not global
    $theme_class = 'psfa-theme-' . esc_attr($selected_theme);
    // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound -- Local variable, not global
    $display_mode = get_option('psfa_display_mode', 'grid');
    
    // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound -- Local variable, not global
    $display_class = 'psfa-display-' . esc_attr($display_mode);
    ?>
    <?php
    // Load customization helpers and get settings BEFORE using them in data attributes
    require_once PSFA_PLUGIN_DIR . 'core/shared/customization-helpers.php';
    // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound -- Local variable, not global
    $button_text = psfa_get_button_text();
    // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound -- Local variable, not global
    $button_style = psfa_get_button_style();
    // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound -- Local variable, not global
    $card_layout = psfa_get_card_layout();
    // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound -- Local variable, not global
    $font_settings = psfa_get_font_settings();
    // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound -- Local variable, not global
    $selected_theme = psfa_get_selected_theme();
    // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound -- Local variable, not global
    $theme_colors = psfa_get_theme_colors($selected_theme);
    ?>
    <div class="psfa-products <?php echo esc_attr( $theme_class . ' ' . $display_class ); ?>" 
         id="products-container"
         data-query="<?php echo esc_attr($psfa_search_params['query'] ?? ''); ?>"
         data-category="<?php echo esc_attr($psfa_search_params['category'] ?? ''); ?>"
         data-price-min="<?php echo esc_attr($psfa_search_params['price_min'] ?? 0); ?>"
         data-price-max="<?php echo esc_attr($psfa_search_params['price_max'] ?? 0); ?>"
         data-brand="<?php echo esc_attr($psfa_search_params['brand'] ?? ''); ?>"
         data-sort="<?php echo esc_attr($psfa_search_params['sort'] ?? ''); ?>"
         data-condition="<?php echo esc_attr($psfa_search_params['condition'] ?? ''); ?>"
         data-merchant="<?php echo esc_attr($psfa_search_params['merchant'] ?? ''); ?>"
         data-availability="<?php echo esc_attr($psfa_search_params['availability'] ?? 'Available'); ?>"
         data-has-deal="<?php echo esc_attr($psfa_request_params['has_deal'] ?? ''); ?>"
         data-min-saving-percent="<?php echo esc_attr($psfa_request_params['min_saving_percent'] ?? 0); ?>"
         data-page="<?php echo esc_attr(isset($psfa_pages_fetched) ? ($psfa_pages_fetched + 1) : 1); ?>"
         data-has-more="<?php echo esc_attr($psfa_has_more ? '1' : '0'); ?>"
         data-max-pages="<?php echo esc_attr($psfa_max_pages); ?>"
         data-is-premium="<?php echo psfa_is_premium() ? '1' : '0'; ?>"
         data-auth-type="<?php echo esc_attr(function_exists('psfa_is_oauth_auth_type') && psfa_is_oauth_auth_type() ? 'oauth' : 'awssig4'); ?>"
         data-card-layout="<?php echo esc_attr($card_layout); ?>"
         data-button-text="<?php echo esc_attr($button_text); ?>"
         data-pagination="<?php echo esc_attr($psfa_pagination_type); ?>">
    <?php 
    // Get offer data extractor instance for single source of truth
    $psfa_extractor = psfa_get_offer_extractor();
    ?>
    <?php foreach ($psfa_display_results['SearchResult']['Items'] as $psfa_item): ?>
        <?php 
        // Basic item info
        $psfa_title = $psfa_item['ItemInfo']['Title']['DisplayValue'] ?? '(No Title)';
        $psfa_url = $psfa_item['DetailPageURL'] ?? '#';
        $psfa_img = $psfa_item['Images']['Primary']['Large']['URL'] ?? '';
        $psfa_by = $psfa_item['ItemInfo']['ByLineInfo']['Brand']['DisplayValue'] ?? '';
        $psfa_sales_rank = $psfa_item['BrowseNodeInfo']['WebsiteSalesRank']['SalesRank'] ?? 0;
        
        // Use extractor for offer data (V2 only)
        $psfa_price = $psfa_extractor->get_price_display($psfa_item);
        $psfa_saving_basis = $psfa_extractor->get_original_price_display($psfa_item);
        $psfa_saving_percent = $psfa_extractor->get_savings_percent($psfa_item) ?? 0;
        
        // V2 Premium Features
        $psfa_deal_badge = $psfa_extractor->get_deal_badge($psfa_item);
        $psfa_is_prime_exclusive = $psfa_extractor->is_prime_exclusive_deal($psfa_item);
        $psfa_is_stock_scarce = $psfa_extractor->is_stock_scarce($psfa_item);
        $psfa_deal_countdown = $psfa_extractor->get_deal_countdown($psfa_item);
        
        // Debug per-item badge values
        if ($psfa_debug_mode) {
            $psfa_debug_item = [
                'asin' => $psfa_item['ASIN'] ?? '',
                'price_display' => $psfa_price,
                'offersv2_structure' => $psfa_item['OffersV2']['Listings'][0]['Price'] ?? null,
                'offers_structure' => $psfa_item['Offers']['Listings'][0]['Price'] ?? null,
                'deal_details' => $psfa_item['OffersV2']['Listings'][0]['DealDetails'] ?? null,
                'is_prime_exclusive' => $psfa_is_prime_exclusive,
                'deal_badge' => $psfa_deal_badge,
                'deal_countdown' => $psfa_deal_countdown,
                'is_stock_scarce' => $psfa_is_stock_scarce,
                'saving_percent' => $psfa_saving_percent,
                'is_premium' => psfa_is_premium(),
                'prime_feature_enabled' => psfa_is_feature_enabled('prime_exclusive'),
            ];
            echo '<script>console.log("PSFA Debug Item:", ' . wp_json_encode($psfa_debug_item) . ');</script>';
        }
        ?>
        <div class="psfa-product psfa-card-layout-<?php echo esc_attr($card_layout); ?>">
            <?php if ($psfa_img): ?>
            <div class="psfa-image-column">
                <img src="<?php echo esc_url($psfa_img); ?>" alt="<?php echo esc_attr($psfa_title); ?>">
                <?php  ?>
            </div>
            <?php endif; ?>
            <div class="psfa-title-column">
                <h7 class="psfa-title-font" title="<?php echo esc_attr($psfa_title); ?>"><?php echo esc_html($psfa_title); ?></h7>
            </div>
            <div class="psfa-purchase-column">
            <?php if ($psfa_by): ?>
                <div class="psfa-brand psfa-description-font"><?php echo esc_html($psfa_by); ?></div>
            <?php endif; ?>
            <?php if ($psfa_price): ?>
                <div class="psfa-price psfa-price-font">
                    <?php if (!empty($psfa_saving_basis)): ?>
                    <span class="psfa-original-price" style="text-decoration: line-through; color: #888; font-size: 0.85em;"><?php echo esc_html($psfa_saving_basis); ?></span>
                    <?php endif; ?>
                    <?php echo esc_html($psfa_price); ?>
                </div>
            <?php endif; ?>
            <a href="<?php echo esc_url($psfa_url); ?>" target="_blank" rel="noopener" class="psfa-buy-button" 
               style="<?php 
               echo !empty($button_style['background_color']) ? 'background-color: ' . esc_attr($button_style['background_color']) . ';' : '';
               echo !empty($button_style['text_color']) ? ' color: ' . esc_attr($button_style['text_color']) . ';' : '';
               echo !empty($button_style['border_radius']) ? ' border-radius: ' . esc_attr($button_style['border_radius']) . ';' : '';
               echo !empty($button_style['font_size']) ? ' font-size: ' . esc_attr($button_style['font_size']) . ';' : '';
               echo !empty($button_style['font_weight']) ? ' font-weight: ' . esc_attr($button_style['font_weight']) . ';' : '';
               ?>">
                <img src="<?php echo esc_url( plugins_url('assets/img/amazon-logo.svg', PSFA_PLUGIN_DIR . 'paapi-product-search-for-amazon.php') ); ?>" alt="Buy on Amazon" class="psfa-buy-button-icon"><?php echo esc_html($button_text); ?></a>
            </div>
            <?php
            /**
             * Hook for add-ons to add actions to product cards (e.g., "+ Box" button)
             *
             * @param array  $psfa_item Full product item data from API
             * @param string $asin      Product ASIN
             */
            do_action( 'psfa_product_card_actions', $psfa_item, $psfa_item['ASIN'] ?? '' );
            ?>
        </div>
    <?php endforeach; ?>
    </div>
    
    <?php if ($psfa_pagination_type !== 'none'): ?>
    <!-- Sentinel used to trigger infinite scroll when it enters the viewport -->
    <div id="infinite-scroll-sentinel" <?php echo $psfa_pagination_type !== 'infinite' ? 'style="display:none;"' : ''; ?>></div>
    
    <!-- Load More Button (for manual pagination) -->
    <?php if ($psfa_pagination_type === 'button' && $psfa_has_more): ?>
    <div class="psfa-load-more-wrapper <?php echo esc_attr( $theme_class ); ?>" id="psfa-load-more-wrapper" style="text-align: center; margin: 20px 0;">
        <button type="button" id="psfa-load-more-btn" class="psfa-load-more-btn">
            Load More Products
        </button>
    </div>
    <?php endif; ?>
    
    <!-- Infinite Scroll Loader -->
    <div class="infinite-scroll-loader <?php echo esc_attr( $theme_class ); ?>" id="infinite-scroll-loader" <?php echo $psfa_pagination_type !== 'infinite' ? 'style="display:none;"' : ''; ?>>
        <div class="spinner"></div>
        <p style="margin-top: 10px; color: #666;">Loading more products...</p>
    </div>
    <?php endif; ?>
    
    <!-- Upgrade Notice (shown when free users reach page limit, but not in mock mode) -->
    <?php if (!psfa_is_premium() && !(defined('PSFA_MOCK_MODE') && PSFA_MOCK_MODE)): ?>
    <div class="psfa-upgrade-notice <?php echo esc_attr( $theme_class ); ?>" id="psfa-upgrade-notice" style="display: none; margin: 20px 0; padding: 15px; background: #fff3cd; border-left: 4px solid; border-radius: 4px;">
        <p style="margin: 0; color: #856404;">
            <strong>You've reached the free limit of 5 pages.</strong> 
            <a href="https://paapiplugin.com" target="_blank" style="color: #f90; font-weight: bold;">Upgrade to Premium</a> for unlimited pagination and all Amazon categories!
        </p>
    </div>
    <?php endif; ?>
    
    <!-- End of Results Message -->
    <div class="infinite-scroll-end<?php echo empty($psfa_has_more) ? ' active' : ''; ?>" id="infinite-scroll-end">
        <p>You've reached the end of the results.</p>
    </div>
    
    <?php if (psfa_show_branding()): ?>
    <!-- Powered by PSFA branding -->
    <div class="psfa-branding" style="text-align: center; margin: 20px 0 10px; padding: 8px;">
        <a href="https://paapiplugin.com" target="_blank" rel="noopener" title="Paapi Product Search For Amazon" style="color: #666; font-size: 12px; text-decoration: none; display: inline-flex; align-items: center; gap: 4px;">
            <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M9.937 15.5A2 2 0 0 0 8.5 14.063l-6.135-1.582a.5.5 0 0 1 0-.962L8.5 9.936A2 2 0 0 0 9.937 8.5l1.582-6.135a.5.5 0 0 1 .963 0L14.063 8.5A2 2 0 0 0 15.5 9.937l6.135 1.581a.5.5 0 0 1 0 .964L15.5 14.063a2 2 0 0 0-1.437 1.437l-1.582 6.135a.5.5 0 0 1-.963 0z"/></svg>
            Powered by PSFA
        </a>
    </div>
    <?php endif; ?>
<?php elseif (isset($psfa_display_results)): ?>
    <div class="psfa-no-results">
        <p><strong>No products found matching your filters.</strong></p>
        <?php 
        // Show total unfiltered count if available (includes merged page 2), otherwise use page 1 count
        $psfa_unfiltered_count = isset($psfa_total_unfiltered_count) ? $psfa_total_unfiltered_count : (isset($psfa_results['SearchResult']['Items']) ? count($psfa_results['SearchResult']['Items']) : 0);
        if ($psfa_unfiltered_count > 0): ?>
            <p style="margin-top: 10px;">Try adjusting your filters above to see all <?php echo esc_html($psfa_unfiltered_count); ?> available results.</p>
        <?php endif; ?>
    </div>
<?php endif; ?>
